<?php

namespace App\Exports;

use App\Models\TeacherSubject;
use App\Models\Admin;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;

class ProfessoresExport implements FromCollection, WithHeadings, ShouldAutoSize, WithStyles, WithTitle
{
    protected $filters;

    public function __construct($filters = [])
    {
        $this->filters = $filters;
    }

    public function collection()
    {
        $query = Admin::query()->where('role', 2); // Apenas professores

        // Aplicar filtros de busca
        if (!empty($this->filters['search'])) {
            $query->where(function ($subQuery) {
                $subQuery->where('name', 'like', '%' . $this->filters['search'] . '%')
                    ->orWhere('email', 'like', '%' . $this->filters['search'] . '%')
                    ->orWhere('phone_number', 'like', '%' . $this->filters['search'] . '%');
            });
        }

        // Filtro de status
        if (isset($this->filters['status']) && $this->filters['status'] !== '') {
            $query->where('is_active', $this->filters['status']);
        }

        $teachers = $query->with(['teacherSubjects.subject', 'teacherSubjects.classroom'])->get();

        // Agrupar por professor
        return $teachers->map(function ($teacher) {
            $teacherSubjects = $teacher->teacherSubjects;

            // Aplicar filtros nas atribuições
            if (!empty($this->filters['class'])) {
                $teacherSubjects = $teacherSubjects->filter(function ($ts) {
                    return $ts->classroom && $ts->classroom->class == $this->filters['class'];
                });
            }

            if (!empty($this->filters['turma'])) {
                $teacherSubjects = $teacherSubjects->filter(function ($ts) {
                    return $ts->classroom_id == $this->filters['turma'];
                });
            }

            if (!empty($this->filters['subject'])) {
                $teacherSubjects = $teacherSubjects->filter(function ($ts) {
                    return $ts->subject_id == $this->filters['subject'];
                });
            }

            // Se não tem atribuições após filtros, não incluir
            if ($teacherSubjects->isEmpty()) {
                return null;
            }

            // Agrupar disciplinas, classes e turmas
            $disciplinas = $teacherSubjects->map(function ($ts) {
                return $ts->subject ? $ts->subject->name : '-';
            })->unique()->implode(', ');

            $classes = $teacherSubjects->map(function ($ts) {
                return $ts->classroom ? $ts->classroom->class . 'ª' : '-';
            })->unique()->implode(', ');

            $turmas = $teacherSubjects->map(function ($ts) {
                return $ts->classroom ? $ts->classroom->name : '-';
            })->unique()->implode(', ');

            return [
                'nome' => $teacher->name,
                'email' => $teacher->email,
                'telefone' => $teacher->phone_number ?: '-',
                'disciplinas' => $disciplinas ?: '-',
                'classes' => $classes ?: '-',
                'turmas' => $turmas ?: '-',
                'estado' => $teacher->is_active ? 'Ativo' : 'Desativado',
            ];
        })->filter()->values();
    }

    public function headings(): array
    {
        return [
            'Nome',
            'Email',
            'Telefone',
            'Disciplinas',
            'Classes',
            'Turmas',
            'Estado',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Estilo do cabeçalho
        $sheet->getStyle('A1:G1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
                'size' => 12,
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4e73df'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ]);

        // Estilo das linhas de dados
        $lastRow = $sheet->getHighestRow();
        if ($lastRow > 1) {
            $sheet->getStyle('A2:G' . $lastRow)->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => 'DDDDDD'],
                    ],
                ],
                'alignment' => [
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ]);

            // Zebrar as linhas
            for ($row = 2; $row <= $lastRow; $row++) {
                if ($row % 2 == 0) {
                    $sheet->getStyle('A' . $row . ':G' . $row)->applyFromArray([
                        'fill' => [
                            'fillType' => Fill::FILL_SOLID,
                            'startColor' => ['rgb' => 'F8F9FA'],
                        ],
                    ]);
                }
            }
        }

        // Altura da primeira linha
        $sheet->getRowDimension(1)->setRowHeight(25);

        return [];
    }

    public function title(): string
    {
        return 'Lista de Professores';
    }
}
