<?php

namespace App\Http\Livewire\Admin\Attendance;

use App\Models\Attendance;
use App\Models\Classroom;
use App\Models\TeacherSubject;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithPagination;

class ListAttendance extends Component
{
    use WithPagination, FindGuard, LivewireAlert;

    protected string $paginationTheme = 'bootstrap';

    public $year;
    public $classroom_id;
    public $date_from;
    public $date_to;
    public $search;
    public $statusFilter;
    public bool $readyToLoad = false;

    protected $queryString = ['search', 'classroom_id', 'statusFilter'];

    protected $listeners = ['confirmed', 'cancelled'];

    public $deleteId;

    public function mount()
    {
        $this->year = date('Y');
        $this->date_from = date('Y-m-01');
        $this->date_to = date('Y-m-d');
    }

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function deleteAttendance($id)
    {
        $this->deleteId = $id;
        $this->confirm('Tem certeza de que deseja excluir este registo?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, excluir!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        if ($this->deleteId) {
            Attendance::where('id', $this->deleteId)->delete();
            $this->alert('success', 'Registo excluído com sucesso.');
            $this->deleteId = null;
        }
    }

    public function cancelled()
    {
        $this->alert('error', 'Operação cancelada.');
        $this->deleteId = null;
    }

    public function render()
    {
        $user = $this->findGuardType()->user();
        $isTeacher = ($user->role == 2);

        $classrooms = collect();
        if ($isTeacher) {
            $classroomIds = TeacherSubject::where('teacher_id', $user->id)
                ->whereNotNull('classroom_id')
                ->distinct()
                ->pluck('classroom_id');
            $classrooms = Classroom::whereIn('id', $classroomIds)->orderByRaw('CAST(class AS UNSIGNED)')->orderBy('name')->get();
        } else {
            $classrooms = Classroom::orderByRaw('CAST(class AS UNSIGNED)')->orderBy('name')->get();
        }

        $records = [];
        if ($this->readyToLoad) {
            $query = Attendance::with(['student', 'classroom', 'recorder'])
                ->where('year', $this->year);

            if ($isTeacher) {
                $classroomIds = TeacherSubject::where('teacher_id', $user->id)
                    ->whereNotNull('classroom_id')
                    ->pluck('classroom_id');
                $query->whereIn('classroom_id', $classroomIds);
            }

            if ($this->classroom_id) {
                $query->where('classroom_id', $this->classroom_id);
            }
            if ($this->date_from) {
                $query->where('date', '>=', $this->date_from);
            }
            if ($this->date_to) {
                $query->where('date', '<=', $this->date_to);
            }
            if ($this->statusFilter) {
                $query->where('status', $this->statusFilter);
            }
            if ($this->search) {
                $query->whereHas('student', function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%');
                });
            }

            $records = $query->latest('date')->paginate(50);
        }

        return view('livewire.admin.attendance.list-attendance', [
            'classrooms' => $classrooms,
            'records' => $records,
            'statuses' => Attendance::statuses(),
            'isTeacher' => $isTeacher,
        ]);
    }
}
