<?php

namespace App\Http\Livewire\Admin\Exams;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Subject;
use App\Models\Trimester;
use App\Models\TeacherSubject;
use App\Models\User;
use App\Services\ExcelMarksImportService;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads;
use Note\Note;
use Illuminate\Support\Facades\Storage;

class ExcelImportMarks extends Component
{
    use FindGuard, LivewireAlert, WithFileUploads;

    public $year;
    public $trimester_id;
    public $class_level;
    public $classroom_id;
    public $subject_id;
    public $excel_file;
    public $clean_existing = true;
    public $preview_data = [];
    public $validation_result = null;
    public $import_result = null;
    public $step = 1; // 1: Selection, 2: File Upload, 3: Preview, 4: Import Results

    public $user;
    public $isTeacher = false;
    public $availableClassLevels = [];

    // Properties for handling student corrections
    public $student_corrections = [];
    public $show_corrections_modal = false;
    public $selected_student_for_correction = null;
    public $selected_correction_student = null; // Added missing property
    public $available_students = [];

    protected $listeners = [
        'confirmed',
        'cancelled',
        'applyCorrections'
    ];

    protected $rules = [
        'year' => 'required|integer|min:2020|max:2030',
        'trimester_id' => 'required|in:1,2,3',
        'class_level' => 'required|integer|min:1|max:12',
        'classroom_id' => 'required|exists:classrooms,id',
        'subject_id' => 'required|exists:subjects,id',
        'excel_file' => 'required|file|mimes:xlsx,xls|max:10240',
        'clean_existing' => 'boolean'
    ];

    public function mount()
    {
        $this->year = date('Y');
        $this->user = $this->findGuardType()->user();
        $this->isTeacher = ($this->user->role == 2);
        $this->loadAvailableClassLevels();
    }

    public function loadAvailableClassLevels()
    {
        if ($this->isTeacher) {
            $this->availableClassLevels = TeacherSubject::where('teacher_id', $this->user->id)
                ->whereNotNull('class')
                ->distinct()
                ->pluck('class')
                ->toArray();
        }
    }

    public function updatedYear()
    {
        $this->resetSelections(['class_level', 'classroom_id', 'subject_id']);
    }

    public function updatedClassLevel()
    {
        $this->resetSelections(['classroom_id', 'subject_id']);
    }

    public function updatedClassroomId()
    {
        $this->resetSelections(['subject_id']);
        $this->loadAvailableStudents();
    }

    public function updatedExcelFile()
    {
        if ($this->excel_file) {
            $this->validateFile();
        }
    }

    private function resetSelections(array $fields)
    {
        foreach ($fields as $field) {
            $this->$field = null;
        }
        $this->resetImportState();
    }

    private function resetImportState()
    {
        $this->validation_result = null;
        $this->import_result = null;
        $this->preview_data = [];
        $this->student_corrections = [];
        if ($this->step > 2) {
            $this->step = 2;
        }
    }

    private function loadAvailableStudents()
    {
        if ($this->classroom_id) {
            $this->available_students = User::where('classroom_id', $this->classroom_id)
                ->orderBy('name')
                ->get(['id', 'name', 'student_id'])
                ->toArray();
        }
    }

    public function nextStep()
    {
        if ($this->step == 1) {
            $this->validate([
                'year' => 'required|integer',
                'trimester_id' => 'required',
                'class_level' => 'required',
                'classroom_id' => 'required',
                'subject_id' => 'required'
            ]);
            $this->step = 2;
        }
    }

    public function previousStep()
    {
        if ($this->step > 1) {
            $this->step--;
        }
    }

    public function validateFile()
    {
        try {
            $this->validate(['excel_file' => 'required|file|mimes:xlsx,xls|max:10240']);

            $filePath = $this->excel_file->getRealPath();
            $service = new ExcelMarksImportService();

            $this->validation_result = $service->validateExcelFile($filePath);

            if ($this->validation_result['valid']) {
                $this->step = 3;
                $this->loadPreviewData();
            } else {
                $this->alert('error', $this->validation_result['message']);
            }

        } catch (ValidationException $e) {
            $this->alert('error', 'Please select a valid Excel file');
        } catch (\Exception $e) {
            $this->alert('error', 'Error validating file: ' . $e->getMessage());
        }
    }

    public function loadPreviewData()
    {
        try {
            $filePath = $this->excel_file->getRealPath();
            $service = new ExcelMarksImportService();

            // Get all rows for preview
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $highestRow = $worksheet->getHighestRow();

            $this->preview_data = [];
            for ($row = 1; $row <= $highestRow; $row++) {
                $rowData = [];
                for ($col = 'A'; $col <= 'J'; $col++) {
                    $rowData[] = $worksheet->getCell($col . $row)->getValue();
                }
                $this->preview_data[] = $rowData;
            }

        } catch (\Exception $e) {
            $this->alert('error', 'Error loading preview: ' . $e->getMessage());
        }
    }

    public function importMarks()
    {
        $this->validate();

        $this->confirm('Tem certeza de que deseja importar as notas? Esta ação não pode ser desfeita.', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, importar!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        try {
            // Store file temporarily
            $fileName = 'import_' . time() . '.' . $this->excel_file->getClientOriginalExtension();
            $filePath = $this->excel_file->storeAs('temp', $fileName);
            $fullPath = Storage::path($filePath);

            $service = new ExcelMarksImportService();

            $importData = [
                'year' => $this->year,
                'trimester_id' => $this->trimester_id,
                'classroom_id' => $this->classroom_id,
                'subject_id' => $this->subject_id,
                'clean_existing' => $this->clean_existing,
                'student_corrections' => $this->student_corrections // Pass corrections to service
            ];

            $this->import_result = $service->importMarks($fullPath, $importData);

            // Clean up temporary file
            Storage::delete($filePath);

            if ($this->import_result['success']) {
                $stats = $this->import_result['data'];

                Note::createSystemNotification(
                    Admin::class,
                    'Importação de Notas',
                    "Importação concluída: {$stats['successful']} sucessos, {$stats['failed']} falhas"
                );

                $this->emit('noteAdded');
                $this->step = 4;

                if ($stats['failed'] > 0) {
                    $this->alert('warning', "Importação parcialmente concluída: {$stats['successful']} sucessos, {$stats['failed']} falhas");
                } else {
                    $this->alert('success', "Importação concluída com sucesso: {$stats['successful']} registros");
                }
            } else {
                $this->alert('error', 'Erro na importação: ' . $this->import_result['message']);
            }

        } catch (\Exception $e) {
            $this->alert('error', 'Erro na importação: ' . $e->getMessage());
        }
    }

    public function cancelled()
    {
        $this->alert('info', 'Importação cancelada');
    }

    public function showCorrectionsModal($studentName, $rowNumber)
    {
        $this->selected_student_for_correction = [
            'name' => $studentName,
            'row' => $rowNumber
        ];
        $this->selected_correction_student = null; // Reset selection
        $this->show_corrections_modal = true;
    }

    public function closeCorrectionsModal()
    {
        $this->show_corrections_modal = false;
        $this->selected_student_for_correction = null;
        $this->selected_correction_student = null;
    }

    public function applyCorrectionForStudent()
    {
        if (!$this->selected_correction_student || !$this->selected_student_for_correction) {
            $this->alert('error', 'Por favor selecione um estudante');
            return;
        }

        $originalName = $this->selected_student_for_correction['name'];
        $this->student_corrections[$originalName] = $this->selected_correction_student;

        $this->closeCorrectionsModal();
        $this->alert('success', 'Correção aplicada com sucesso');

        // Optionally refresh the import to apply corrections immediately
        // You might want to add a method to re-run the import with corrections
    }

    public function removeCorrectionForStudent($excelName)
    {
        if (isset($this->student_corrections[$excelName])) {
            unset($this->student_corrections[$excelName]);
            $this->alert('success', 'Correção removida com sucesso');
        }
    }

    public function importWithCorrections()
    {
        if (empty($this->student_corrections)) {
            $this->alert('warning', 'Nenhuma correção foi aplicada');
            return;
        }

        $this->confirm('Tem certeza de que deseja re-importar com as correções aplicadas?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, re-importar!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function resetImport()
    {
        $this->step = 1;
        $this->excel_file = null;
        $this->validation_result = null;
        $this->import_result = null;
        $this->preview_data = [];
        $this->student_corrections = [];
        $this->selected_correction_student = null;
        $this->selected_student_for_correction = null;
    }

    public function moveStudentToCorrectClass($studentId, $targetClassroomId)
    {
        try {
            $student = User::find($studentId);
            if ($student) {
                $student->classroom_id = $targetClassroomId;
                $student->save();

                $this->alert('success', 'Estudante movido para a turma correta com sucesso');

                // Refresh import result to update the display
                $this->loadAvailableStudents();
            }
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao mover estudante: ' . $e->getMessage());
        }
    }

    public function getErrorTypeLabel($errorType)
    {
        switch ($errorType) {
            case 'different_classroom':
                return 'Estudante em turma diferente';
            case 'name_not_found':
                return 'Nome não encontrado';
            default:
                return 'Outro erro';
        }
    }

    public function render()
    {
        $classrooms = collect();
        $subjects = collect();

        // Load classrooms based on selected class level
        if ($this->class_level) {
            if ($this->isTeacher) {
                $classrooms = Classroom::where('class', $this->class_level)
                    ->whereIn('id', function($query) {
                        $query->select('classroom_id')
                            ->from('teacher_subjects')
                            ->where('teacher_id', $this->user->id)
                            ->whereNotNull('classroom_id');
                    })
                    ->orderBy('name')
                    ->get();
            } else {
                $classrooms = Classroom::where('class', $this->class_level)
                    ->orderBy('name')
                    ->get();
            }
        }

        // Load subjects based on selected classroom
        if ($this->classroom_id) {
            if ($this->isTeacher) {
                $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                    ->select('subjects.id', 'subjects.name', 'subjects.slug')
                    ->where('teacher_subjects.teacher_id', $this->user->id)
                    ->where(function($query) {
                        $query->where('teacher_subjects.classroom_id', $this->classroom_id)
                            ->orWhere(function($q) {
                                $classroom = Classroom::find($this->classroom_id);
                                if ($classroom) {
                                    $q->where('teacher_subjects.class', $classroom->class)
                                        ->whereNull('teacher_subjects.classroom_id');
                                }
                            });
                    })
                    ->orderBy('subjects.name')
                    ->distinct()
                    ->get();
            } else {
                $classroom = Classroom::find($this->classroom_id);
                if ($classroom) {
                    $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                        ->select('subjects.id', 'subjects.name', 'subjects.slug')
                        ->where(function($query) use ($classroom) {
                            $query->where('teacher_subjects.classroom_id', $classroom->id)
                                ->orWhere(function($q) use ($classroom) {
                                    $q->where('teacher_subjects.class', $classroom->class)
                                        ->whereNull('teacher_subjects.classroom_id');
                                });
                        })
                        ->orderBy('subjects.name')
                        ->distinct()
                        ->get();
                }
            }
        }

        return view('livewire.admin.exams.excel-import-marks', [
            'classrooms' => $classrooms,
            'subjects' => $subjects,
            'trimesters' => Trimester::orderBy('name')->get(),
        ]);
    }
}
