<?php

namespace App\Http\Livewire\Admin\Fee;

use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use App\Models\FeeStructure;

class AddFeeMaster extends Component
{
    use FindGuard, LivewireAlert;

    // FeeStructure properties
    public $monthly_fee_grade_1_6, $monthly_fee_grade_7_12, $fee_name, $late_penalty_percentage, $payment_start_day, $payment_due_day, $penalty_type;
    public $active = 0;
    public $is_manual = false;
    public $academic_year;
    public $late_penalty_fixed;
    public $editingId = null;
    public $updateMode = false;
    public $feeStructures;
    public $months = [];
    public $grades = [];

    // IVA fields
    public $iva_applicable = false;
    public $price_includes_iva = false;
    public $iva_notes = '';


    protected $rules = [
        'fee_name' => 'required|string|max:255',
        'monthly_fee_grade_1_6' => 'nullable|numeric',
        'monthly_fee_grade_7_12' => 'required|numeric',
        'late_penalty_percentage' => 'nullable|numeric',
        'late_penalty_fixed' => 'nullable|numeric',
        'payment_start_day' => 'required|integer',
        'payment_due_day' => 'required|integer',
        'active' => 'required|boolean',
        'is_manual' => 'boolean',
        'months' => 'required|array', // Validation rule for months
        'grades' => 'required|array', // Validation rule for grades
        'penalty_type' => 'required',
        'academic_year' => 'required|string|max:20', // Validation for academic year
        'iva_applicable' => 'boolean',
        'price_includes_iva' => 'boolean',
        'iva_notes' => 'nullable|string',
    ];

    /**
     * Mount component - load data if editing
     */
    public function mount($id = null)
    {
        if ($id) {
            $feeStructure = FeeStructure::find($id);

            if ($feeStructure) {
                $this->editingId = $id;
                $this->updateMode = true;
                $this->fee_name = $feeStructure->fee_name;
                $this->academic_year = $feeStructure->academic_year;
                $this->monthly_fee_grade_7_12 = $feeStructure->monthly_fee;
                $this->late_penalty_percentage = $feeStructure->late_penalty_percentage;
                $this->payment_start_day = $feeStructure->payment_start_day;
                $this->payment_due_day = $feeStructure->payment_due_day;
                $this->penalty_type = $feeStructure->penalty_type;
                $this->active = $feeStructure->active;
                $this->is_manual = $feeStructure->is_manual ?? false;

                // Load months and grades
                $this->months = !empty($feeStructure->months) ? explode(',', $feeStructure->months) : [];
                $this->grades = !empty($feeStructure->grades) ? explode(',', $feeStructure->grades) : [];

                // Load IVA fields
                $this->iva_applicable = $feeStructure->iva_applicable ?? false;
                $this->price_includes_iva = $feeStructure->price_includes_iva ?? false;
                $this->iva_notes = $feeStructure->iva_notes ?? '';

                // Check if penalty is fixed or percentage
                if ($this->penalty_type === 'fixed') {
                    $this->late_penalty_fixed = $feeStructure->late_penalty_percentage;
                }
            }
        }
    }

    /**
     * Perform real-time validation on updated properties
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function store()
    {
        // Validate the inputs
        $this->validate();

        $data = [
            'fee_name' => $this->fee_name,
            'grade' => 0,
            'monthly_fee' => $this->monthly_fee_grade_7_12,
            'late_penalty_percentage' => $this->penalty_type === 'percentage' ? $this->late_penalty_percentage : $this->late_penalty_fixed,
            'payment_start_day' => $this->payment_start_day,
            'payment_due_day' => $this->payment_due_day,
            'active' => $this->active,
            'is_manual' => $this->is_manual,
            'months' => implode(",",$this->months), // Store selected months as CSV
            'grades' => implode(",",$this->grades),
            'penalty_type' => $this->penalty_type,
            'academic_year' => $this->academic_year, // Store academic year
            'iva_applicable' => $this->iva_applicable,
            'price_includes_iva' => $this->price_includes_iva,
            'iva_notes' => $this->iva_notes,
        ];

        if ($this->editingId) {
            // Update existing record
            $feeStructure = FeeStructure::find($this->editingId);
            $feeStructure->update($data);
            $message = 'Plano de taxas atualizado com sucesso!';
        } else {
            // Create new record
            FeeStructure::create($data);
            $message = 'Plano de taxas criado com sucesso!';
        }

        // Reset fields
        $this->reset();

        // Refresh the list
        $this->feeStructures = FeeStructure::active()->get();

        // Show success message
        $this->alert('success', $message);

        // Redirect to list
        return redirect()->route('admin.list.feemaster');
    }


    public function render()
    {
        return view('livewire.admin.fee.add-fee-master');
    }

}
