<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\Fee_assign;
use App\Models\User;
use App\Services\FeeCalculationService;
use Livewire\Component;
use App\Models\FeeStructure;
use Carbon\Carbon;
use App\Models\FeesDiscount as discount;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class FeeCollection extends Component
{
    public $studentId;
    public $feeGroup;
    public $date;
    public $amount;
    public $discountGroup;
    public $discount = 0;
    public $fine = 0;
    public $paymentMode = 'Cash';
    public $transactionId;
    public $note;
    public $students = [];
    public $selectedStudent;
    public $selectedFee;
    public $selectedFeeAssignId;
    public $month;
    public $year;

    // Cálculos atuais (form do modal)
    public $calculatedBaseFee = 0;
    public $calculatedFine = 0;
    public $calculatedTotal = 0;
    public $isLatePayment = false;
    public $dueDate = null;

    // Filtro/Export
    public $selectedYear;

    // Modal de Referência
    public $showReferenceModal = false;
    public $selectedReference = null;

    protected $feeCalculationService;

    /* -------------------------- Boot / Mount -------------------------- */

    public function boot(FeeCalculationService $feeCalculationService)
    {
        $this->feeCalculationService = $feeCalculationService;
    }

    public function mount($studentId)
    {
        $this->studentId = $studentId ?: "cpmc-1218";
        $this->selectedStudent = User::findOrFail($studentId);
        $this->date = now()->format('d/m/Y');
        $this->selectedYear = now()->year;

        $this->calculateCurrentFee();
    }

    public function updated($propertyName)
    {
        // Multa não é mais editável manualmente, removido 'fine' da lista
        if (in_array($propertyName, ['month', 'year', 'discount'])) {
            $this->calculateCurrentFee();
        }

        // Quando o método de pagamento muda
        if ($propertyName === 'paymentMode') {
            // Limpar Transaction ID quando mudar de POS para Dinheiro
            if ($this->paymentMode !== 'POS') {
                $this->transactionId = null;
            }
            // Emitir evento para JavaScript recalcular
            $this->dispatchBrowserEvent('paymentModeChanged');
        }
    }

    /**
     * Atualizar quando o ano selecionado mudar
     */
    public function updatedSelectedYear()
    {
        // Força re-renderização com novo ano
        $this->render();
    }

    /* ------------------------ Cálculo em tempo real ------------------------ */

    public function calculateCurrentFee()
    {
        if (!$this->selectedStudent) return;

        try {
            $calculation = $this->feeCalculationService->calculateFeeForStudent(
                $this->selectedStudent,
                $this->month,
                $this->year ? (int) $this->year : null
            );

            $this->calculatedBaseFee = (float) $calculation['base_amount'];
            $this->calculatedFine    = (float) $calculation['fine_amount'] + (float) $this->fine; // auto + manual
            $this->calculatedTotal   = $this->calculatedBaseFee + $this->calculatedFine - (float) $this->discount;
            $this->isLatePayment     = (bool) $calculation['is_late_payment'];
            $this->dueDate           = $calculation['due_date'];

            // O campo "amount" no DB é LÍQUIDO (base + multa - desconto)
            $this->amount = max(0, $this->calculatedTotal);

            $this->emit('feeCalculated', [
                'baseFee'  => $this->calculatedBaseFee,
                'autoFine' => (float) $calculation['fine_amount'],
                'manualFine' => (float) $this->fine,
                'totalFine'  => $this->calculatedFine,
                'total'      => $this->calculatedTotal,
                'isLate'     => $this->isLatePayment,
                'dueDate'    => $this->dueDate?->format('d/m/Y'),
            ]);
        } catch (\Exception $e) {
            \Log::error('Error calculating fee', [
                'student_id' => $this->studentId,
                'month' => $this->month,
                'year' => $this->year,
                'error' => $e->getMessage(),
            ]);
            $this->fallbackCalculation();
        }
    }

    private function fallbackCalculation()
    {
        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        if (!$studentClassroom) {
            $this->calculatedBaseFee = 0;
            $this->calculatedFine = (float) $this->fine;
            $this->calculatedTotal = (float) $this->fine - (float) $this->discount;
            return;
        }

        $feeStructures = FeeStructure::where('active', 1)->get();
        $filtered = [];

        foreach ($feeStructures as $fs) {
            $gradesArray = explode(',', $fs->grades);
            if (in_array($studentClassroom, $gradesArray)) {
                $filtered[] = $fs;
            }
        }

        $base = 0.0;
        foreach ($filtered as $fs) {
            $base += (float) $fs->monthly_fee;
        }

        $this->calculatedBaseFee = $base;
        $this->calculatedFine = (float) $this->fine;
        $this->calculatedTotal = $base + (float) $this->fine - (float) $this->discount;
        $this->amount = max(0, $this->calculatedTotal);
    }

    /* ------------------------------ Pagamento ------------------------------ */

    public function collectFee()
    {
        $rules = [
            'amount'      => 'required|numeric|min:0',
            'discount'    => 'numeric|min:0',
            'fine'        => 'numeric|min:0',
            'paymentMode' => 'required',
        ];

        // Se o método for POS, Transaction ID é obrigatório
        if ($this->paymentMode === 'POS') {
            $rules['transactionId'] = 'required|string|min:3';
        }

        $this->validate($rules);

        try {
            $paymentData = [
                'month'        => $this->month ?: date('F'),
                'year'         => $this->year ?: date('Y'),
                'amount'       => (float) $this->amount,
                'fine'         => (float) $this->fine,
                'discount'     => (float) $this->discount,
                'payment_mode' => $this->paymentMode,
                'paymentMode'  => $this->paymentMode,
                'pay_type'     => strtolower($this->paymentMode),
                'transaction_id' => $this->paymentMode === 'POS' ? $this->transactionId : null,
                'note'         => $this->note,
                'payment_date' => now(),
            ];

            $feeAssign = $this->feeCalculationService->createFeePayment(
                $this->selectedStudent,
                $paymentData
            );

            session()->flash('message', 'Pagamento registado com sucesso.');

            \Log::info('Admin fee payment collected', [
                'student_id'      => $this->studentId,
                'fee_assign_id'   => $feeAssign->id,
                'payment_mode'    => $this->paymentMode,
                'total_paid_net'  => $this->amount,
                'base_calculated' => $this->calculatedBaseFee,
                'fine_preserved'  => $feeAssign->fine,
                'discount_applied'=> $this->discount,
                'is_late_payment' => $this->isLatePayment,
                'month'           => $this->month,
                'year'            => $this->year,
            ]);

            $this->dispatchBrowserEvent('closeModal');
            $this->reset(['amount', 'discount', 'fine', 'paymentMode', 'transactionId', 'note']);
            $this->calculateCurrentFee();
        } catch (\Exception $e) {
            \Log::error('Error collecting fee', [
                'student_id' => $this->studentId,
                'error'      => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao processar pagamento: ' . $e->getMessage());
        }
    }

    /* ------------------------------ Helpers ------------------------------ */

    /**
     * Divide um pagamento real do DB.
     * IMPORTANTE: No DB, "amount" é o LÍQUIDO (base + multa - desconto)
     * @return array [base, multa, desconto, liquido]
     */
//    private function splitPayment(?Fee_assign $p): array
//    {
//        if (!$p) return [0.0, 0.0, 0.0, 0.0];
//        $net      = (float) ($p->amount ?? 0);
//        $fine     = (float) ($p->fine ?? 0);
//        $discount = (float) ($p->discount ?? 0);
//        $base     = max(0, $net - $fine + $discount);
//        return [$base, $fine, $discount, $net];
//    }
    
    /**
     * Divide um pagamento do DB em [base, multa, desconto, liquido].
     * Decide automaticamente entre:
     *  - H1: amount = líquido (padrão do sistema)
     *  - H2: amount = base   (linhas antigas de 'Reference')
     */
    private function splitPayment(?Fee_assign $p): array
    {
        if (!$p) return [0.0, 0.0, 0.0, 0.0];

        $net      = (float) ($p->amount ?? 0);
        $fine     = max(0.0, (float) ($p->fine ?? 0));
        $discount = max(0.0, (float) ($p->discount ?? 0));

        // H1: amount é LÍQUIDO
        $base_h1 = max(0.0, $net - $fine + $discount);
        $net_h1  = $net;

        // H2: amount é BASE (caso legacy em Reference)
        $base_h2 = max(0.0, (float) ($p->amount ?? 0));
        $net_h2  = max(0.0, $base_h2 + $fine - $discount);

        $isReference = in_array(strtolower($p->pay_type ?? $p->paymentMode ?? ''), ['reference'], true);
        $hasFine     = $fine > 0.0;
        $eps         = 0.01;

        if ($isReference && $hasFine) {
            // Tenta estimar BASE esperada do mês para o aluno.
            try {
                $student = \App\Models\User::find($p->student_id);
                $classroom = $student->class->class
                    ?? $student->classroom->class
                    ?? null;

                $expectedBase = null;
                if ($classroom && ($p->month ?? null) && ($p->year ?? null)) {
                    $all = \App\Models\FeeStructure::where('active', 1)
                        ->whereRaw("FIND_IN_SET(?, grades)", [$classroom])
                        ->get();

                    $monthName = (string) $p->month;
                    $expectedBase = (float) $all->filter(function ($s) use ($monthName) {
                        return empty($s->months) || in_array($monthName, explode(',', $s->months));
                    })->sum('monthly_fee');
                }

                if ($expectedBase !== null && $expectedBase > 0) {
                    // escolhe a hipótese mais próxima da base esperada
                    $d1 = abs($base_h1 - $expectedBase);
                    $d2 = abs($base_h2 - $expectedBase);
                    if ($d2 + $eps < $d1) {
                        return [$base_h2, $fine, $discount, $net_h2];
                    }
                    return [$base_h1, $fine, $discount, $net_h1];
                }

                // Sem expectedBase: favorece H2 para Reference com multa (caso típico)
                return [$base_h2, $fine, $discount, $net_h2];

            } catch (\Throwable $e) {
                // Se der algum erro na estimativa, cai no fallback “favor H2”
                return [$base_h2, $fine, $discount, $net_h2];
            }
        }

        // Caso normal (não-Reference ou sem multa): usar H2 sempre
        // porque amount no DB sempre é BASE (não líquido)
        return [$base_h2, $fine, $discount, $net_h2];
    }


    private function getFeeNamesForMonth($feeStructures, $month)
    {
        $names = [];
        foreach ($feeStructures as $structure) {
            if (empty($structure->months) || in_array($month, explode(',', $structure->months))) {
                $names[] = $structure->fee_name ?? 'Taxa Escolar';
            }
        }
        return empty($names) ? 'Taxa Escolar' : implode(', ', array_unique($names));
    }

    private function getRealFeeNameFromPayment($payment, $feeStructuresForMonth)
    {
        try {
            if (isset($payment->fee_group_id) && $payment->fee_group_id) {
                $feeGroup = \App\Models\FeeGroup::find($payment->fee_group_id);
                if ($feeGroup && $feeGroup->name) return $feeGroup->name;
            }
            if (isset($payment->fee_structure_id) && $payment->fee_structure_id) {
                $feeStructure = \App\Models\FeeStructure::find($payment->fee_structure_id);
                if ($feeStructure && $feeStructure->fee_name) return $feeStructure->fee_name;
            }

            $base = $this->splitPayment($payment)[0];
            foreach ($feeStructuresForMonth as $structure) {
                if (abs($structure->monthly_fee - $base) <= 0.01) {
                    return $structure->fee_name ?? 'Taxa Escolar';
                }
            }

            if (!empty($payment->note)) {
                $note = mb_strtolower($payment->note);
                if (str_contains($note, 'propina')) return 'Propina';
                if (str_contains($note, 'transporte')) return 'Transporte';
                if (str_contains($note, 'aliment')) return 'Alimentação';
                if (str_contains($note, 'material')) return 'Material';
            }
        } catch (\Exception $e) {
            \Log::warning('getRealFeeNameFromPayment fallback', ['error' => $e->getMessage()]);
        }

        $fallback = $feeStructuresForMonth->pluck('fee_name')->filter()->first();
        return $fallback ?: 'Taxa Escolar';
    }

    private function calculateDueDateForMonth($feeStructures, $monthName, $year)
    {
        foreach ($feeStructures as $structure) {
            $dueDateDay = (int) ($structure->payment_due_day ?? 5);
            $monthIndex = array_search($monthName, [
                'January','February','March','April','May','June',
                'July','August','September','October','November','December'
            ]) + 1;

            return str_pad($dueDateDay, 2, '0', STR_PAD_LEFT) . '/'
                 . str_pad($monthIndex, 2, '0', STR_PAD_LEFT) . '/'
                 . $year;
        }
        return 'N/A';
    }

    private function getPaymentReferenceSimple($month, $year)
    {
        static $cachedReference = null;

        if ($cachedReference === null) {
            $ref = \App\Models\PaymentReference::where('student_id', $this->studentId)
                ->where('fee_year', $year)
                ->whereNotNull('reference_number')
                ->first();
            $cachedReference = $ref ? $ref->reference_number : 'REF-' . $this->studentId;
        }

        $monthNumber = array_search($month, [
            'January','February','March','April','May','June',
            'July','August','September','October','November','December'
        ]) + 1;

        if (strlen($cachedReference) >= 11) {
            return substr($cachedReference, 0, 9) . str_pad($monthNumber, 2, '0', STR_PAD_LEFT);
        }

        return $cachedReference . str_pad($monthNumber, 2, '0', STR_PAD_LEFT);
    }

    /* ------------------------------ Export Excel ------------------------------ */

    public function exportExcel()
    {
        try {
            $data = $this->getExportDataComplete();

            // Buscar referências de pagamento agrupadas (incluindo fee_assigns sem referência)
            $data['payment_references'] = $this->getAllPaymentReferencesAndFees($this->selectedYear ?? now()->year);

            return Excel::download(new class($data)
                implements
                    \Maatwebsite\Excel\Concerns\FromArray,
                    \Maatwebsite\Excel\Concerns\WithStyles,
                    \Maatwebsite\Excel\Concerns\WithTitle,
                    \Maatwebsite\Excel\Concerns\WithColumnWidths,
                    \Maatwebsite\Excel\Concerns\WithEvents {

                private $data;

                public function __construct($data) { $this->data = $data; }

                public function array(): array
                {
                    $rows = [];

                    // Título
                    $rows[] = ['COPMOZ - EXTRATO DE PAGAMENTOS ' . $this->data['year']];
                    $rows[] = [''];

                    // Cabeçalho estudante (em uma linha, estilo antigo)
                    $rows[] = [
                        'Estudante: ' . $this->data['student']->name,
                        'ID: ' . $this->data['student']->student_id,
                        ($this->data['student']->class->class ?? 'N/A') . ' Classe - Turma: ' . ($this->data['student']->class->name ?? 'N/A'),
                        'Gerado: ' . now()->format('d/m/Y H:i'),
                    ];
                    $rows[] = [''];

                    // Verificar se há referências
                    if (isset($this->data['payment_references']) && $this->data['payment_references']->count() > 0) {
                        // Calcular resumo
                        $totalPago = $this->data['payment_references']->where('status', 'paid')->sum('amount');
                        $totalEsperado = $this->data['payment_references']->sum('amount');
                        $percent = $totalEsperado > 0 ? round(($totalPago / $totalEsperado) * 100, 1) : 0;

                        $rows[] = [
                            '💰 Total Pago: MT ' . number_format($totalPago, 2, ',', '.'),
                            '📊 Total Geral: MT ' . number_format($totalEsperado, 2, ',', '.'),
                            '📈 Progresso: ' . $percent . '%',
                            ''
                        ];
                        $rows[] = [''];

                        // Agrupar por mês/ano
                        $groupedRefs = $this->data['payment_references']->groupBy(function($ref) {
                            return $ref->fee_month . '-' . $ref->fee_year;
                        });

                        // Ordenar cronologicamente
                        $monthsOrder = ['Janeiro' => 1, 'Fevereiro' => 2, 'Março' => 3, 'Abril' => 4,
                                       'Maio' => 5, 'Junho' => 6, 'Julho' => 7, 'Agosto' => 8,
                                       'Setembro' => 9, 'Outubro' => 10, 'Novembro' => 11, 'Dezembro' => 12];

                        $sortedGroups = $groupedRefs->sortBy(function($group, $key) use ($monthsOrder) {
                            list($month, $year) = explode('-', $key);
                            $monthNum = $monthsOrder[$month] ?? $monthsOrder[strtolower($month)] ?? 99;
                            return $year . str_pad($monthNum, 2, '0', STR_PAD_LEFT);
                        });

                        // Processar cada mês
                        foreach ($sortedGroups as $key => $monthRefs) {
                            list($month, $year) = explode('-', $key);
                            $count = $monthRefs->count();

                            // Cabeçalho do mês (diretamente acima da tabela)
                            $rows[] = ["📅 {$month} / {$year} - {$count} pagamento(s)"];

                            // Cabeçalhos da tabela
                            $rows[] = ['Referência', 'Descrição', 'Vencimento', 'Status', 'Valor Base (MT)', 'Multa (MT)', 'Desconto (MT)', 'Valor Pago (MT)', 'Data Pagamento'];

                            // Dados do mês
                            foreach ($monthRefs as $ref) {
                                $baseAmount = $ref->amount - ($ref->fine_amount ?? 0);
                                $fineAmount = $ref->fine_amount ?? 0;
                                $discountAmount = $ref->discount_amount ?? 0;

                                $status = match($ref->status) {
                                    'paid' => '✅ Pago',
                                    'pending' => ($ref->approval_status ?? '') === 'approved' ? '⏳ Aprovado' : '⏸️ Pendente',
                                    'expired' => '⏰ Expirado',
                                    default => $ref->status
                                };

                                $description = is_array($ref->metadata ?? null)
                                    ? ($ref->metadata['description'] ?? $ref->metadata['custom_fee_type'] ?? 'N/A')
                                    : 'N/A';

                                $expiresAt = $ref->expires_at
                                    ? (is_object($ref->expires_at) ? $ref->expires_at->format('d/m/Y') : $ref->expires_at)
                                    : 'N/A';

                                $createdAt = is_object($ref->created_at)
                                    ? $ref->created_at->format('d/m/Y H:i')
                                    : $ref->created_at;

                                $rows[] = [
                                    $ref->reference_number,
                                    $description,
                                    $expiresAt,
                                    $status,
                                    number_format($baseAmount, 2, ',', '.'),
                                    number_format($fineAmount, 2, ',', '.'),
                                    number_format($discountAmount, 2, ',', '.'),
                                    number_format($ref->amount, 2, ',', '.'),
                                    $createdAt
                                ];
                            }
                        }

                        // Totais finais
                        $rows[] = [''];
                        $rows[] = [
                            '📊 TOTAIS:', '', '', '',
                            'MT ' . number_format($this->data['payment_references']->sum(fn($r) => $r->amount - ($r->fine_amount ?? 0)), 2, ',', '.'),
                            'MT ' . number_format($this->data['payment_references']->sum('fine_amount'), 2, ',', '.'),
                            'MT ' . number_format($this->data['payment_references']->sum('discount_amount'), 2, ',', '.'),
                            'MT ' . number_format($totalEsperado, 2, ',', '.'),
                            ''
                        ];

                        // Estatísticas
                        $rows[] = [''];
                        $rows[] = [
                            '📈 ESTATÍSTICAS:',
                            '✅ Pagos: ' . $this->data['payment_references']->where('status', 'paid')->count() . ' de ' . $this->data['payment_references']->count(),
                            '📊 Taxa de Cumprimento: ' . $percent . '%',
                            ''
                        ];
                    } else {
                        $rows[] = ['Nenhuma referência ou pagamento encontrado.'];
                    }

                    return $rows;
                }

                public function title(): string {
                    return 'Pagamentos ' . $this->data['year'];
                }

                public function columnWidths(): array {
                    return ['A' => 20, 'B' => 30, 'C' => 15, 'D' => 12, 'E' => 12, 'F' => 12, 'G' => 12, 'H' => 12, 'I' => 20];
                }

                public function styles(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet)
                {
                    return [
                        1 => ['font'=>['bold'=>true,'size'=>14,'color'=>['rgb'=>'1565C0']], 'alignment'=>['horizontal'=>'center']],
                        3 => ['font'=>['bold'=>true]],
                        5 => ['font'=>['bold'=>true,'color'=>['rgb'=>'2E7D32']]],
                    ];
                }

                public function registerEvents(): array
                {
                    return [
                        \Maatwebsite\Excel\Events\AfterSheet::class => function(\Maatwebsite\Excel\Events\AfterSheet $event) {
                            $sheet = $event->sheet->getDelegate();
                            $sheet->mergeCells('A1:I1');
                            $sheet->getRowDimension(1)->setRowHeight(24);

                            foreach ($sheet->getRowIterator() as $row) {
                                $cellValue = $sheet->getCell('A' . $row->getRowIndex())->getValue();

                                if (is_string($cellValue) && str_starts_with($cellValue, '📅')) {
                                    $rowIndex = $row->getRowIndex();
                                    $sheet->mergeCells("A{$rowIndex}:I{$rowIndex}");
                                    $sheet->getStyle("A{$rowIndex}:I{$rowIndex}")->getFill()->setFillType('solid')->getStartColor()->setRGB('1565C0');
                                    $sheet->getStyle("A{$rowIndex}:I{$rowIndex}")->getFont()->setBold(true)->setSize(12)->getColor()->setRGB('FFFFFF');
                                    $sheet->getStyle("A{$rowIndex}:I{$rowIndex}")->getAlignment()->setHorizontal('left')->setVertical('center');
                                    $sheet->getRowDimension($rowIndex)->setRowHeight(22);
                                }

                                if ($cellValue === 'Referência') {
                                    $headerRow = $row->getRowIndex();
                                    $sheet->getStyle("A{$headerRow}:I{$headerRow}")->getFill()->setFillType('solid')->getStartColor()->setRGB('1565C0');
                                    $sheet->getStyle("A{$headerRow}:I{$headerRow}")->getFont()->setBold(true)->getColor()->setRGB('FFFFFF');
                                    $sheet->getStyle("A{$headerRow}:I{$headerRow}")->getAlignment()->setHorizontal('center')->setVertical('center');
                                    $sheet->getRowDimension($headerRow)->setRowHeight(20);
                                }

                                $statusCell = $sheet->getCell('D' . $row->getRowIndex())->getValue();
                                if (is_string($statusCell)) {
                                    $fill = 'FFFFFF';
                                    if (str_contains($statusCell, 'Pago')) $fill = 'C8E6C9';
                                    elseif (str_contains($statusCell, 'Expirado')) $fill = 'FFCDD2';
                                    elseif (str_contains($statusCell, 'Pendente')) $fill = 'FFF9C4';
                                    elseif (str_contains($statusCell, 'Aprovado')) $fill = 'B3E5FC';

                                    if ($fill !== 'FFFFFF') {
                                        $sheet->getStyle("D{$row->getRowIndex()}")->getFill()->setFillType('solid')->getStartColor()->setRGB($fill);
                                        $sheet->getStyle("D{$row->getRowIndex()}")->getFont()->setBold(true);
                                        $sheet->getStyle("D{$row->getRowIndex()}")->getAlignment()->setHorizontal('center');
                                    }
                                }
                            }

                            $lastRow = $sheet->getHighestRow();
                            $sheet->getStyle("A7:I{$lastRow}")
                                  ->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN)
                                  ->getColor()->setRGB('CFD8DC');

                            $sheet->getStyle("A7:B{$lastRow}")->getAlignment()->setHorizontal('left');
                            $sheet->getStyle("C7:D{$lastRow}")->getAlignment()->setHorizontal('center');
                            $sheet->getStyle("E7:H{$lastRow}")->getAlignment()->setHorizontal('right');
                        }
                    ];
                }

            }, "pagamentos_{$this->selectedStudent->student_id}_{$this->selectedYear}.xlsx");

        } catch (\Exception $e) {
            \Log::error('Excel export failed', ['error' => $e->getMessage()]);
            session()->flash('error', 'Erro ao exportar Excel: ' . $e->getMessage());
            return back();
        }
    }


    private function getExportDataComplete()
    {
        $currentYear = $this->selectedYear ?? now()->year;
        $months = ['Janeiro','Fevereiro','Março','Abril','Maio','Junho','Julho','Agosto','Setembro','Outubro','Novembro','Dezembro'];

        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        $allFeeStructures = collect();
        if ($studentClassroom) {
            $allFeeStructures = FeeStructure::where('active', 1)
                ->whereRaw("FIND_IN_SET(?, grades)", [$studentClassroom])
                ->get();
        }

        $payments = Fee_assign::where('student_id', $this->studentId)
            ->where('year', $currentYear)
            ->get()
            ->keyBy('month');

        $monthlyData = [];
        $summary = [
            'total_annual_value'    => 0.0,
            'total_paid'            => 0.0, // base recebida
            'total_fines_paid'      => 0.0,
            'total_discounts_given' => 0.0,
            'total_net_received'    => 0.0,
            'months_with_fees'      => 0,
            'months_paid_count'     => 0,
        ];

        foreach ($months as $monthName) {
            $payment = $payments->get($monthName);
            $isPaid  = (bool) $payment;

            $feeStructuresForMonth = $allFeeStructures->filter(function($s) use ($monthName) {
                return empty($s->months) || in_array($monthName, explode(',', $s->months));
            });

            $baseAmountForMonth = (float) $feeStructuresForMonth->sum('monthly_fee');
            if (!$isPaid && $baseAmountForMonth <= 0) continue;

            if ($isPaid) {
                [$baseAmount, $fineAmount, $discountAmount, $netPaidAmount] = $this->splitPayment($payment);
                $paymentDate = $payment->created_at->format('d/m/Y H:i');
                $status = 'Pago';
                $feeNames = $this->getRealFeeNameFromPayment($payment, $feeStructuresForMonth);
                $dueDate = $this->calculateDueDateForMonth($feeStructuresForMonth, $monthName, $currentYear);
            } else {
                $baseAmount = $baseAmountForMonth;
                $dueDate = $this->calculateDueDateForMonth($feeStructuresForMonth, $monthName, $currentYear);

                $isOverdue = false;
                if ($dueDate !== 'N/A') {
                    try { $isOverdue = now()->gt(Carbon::createFromFormat('d/m/Y', $dueDate)); }
                    catch (\Exception $e) { $isOverdue = false; }
                }

                // Multa prevista (somente para exibição)
                $fineAmount = 0.0;
                if ($isOverdue) {
                    foreach ($feeStructuresForMonth as $structure) {
                        if (($structure->penalty_type ?? null) === 'fixed') {
                            $fineAmount += (float) ($structure->late_penalty_percentage ?? 0);
                        } else {
                            $fineAmount += ((float) $structure->monthly_fee) * ((float) ($structure->late_penalty_percentage ?? 0)) / 100;
                        }
                    }
                }

                $feeNames = $feeStructuresForMonth->pluck('fee_name')->filter()->unique()->implode(', ');
                if (!$feeNames) $feeNames = 'Taxa Escolar';

                $discountAmount = 0.0;
                $netPaidAmount  = 0.0;
                $paymentDate    = '--';
                $status         = $isOverdue ? 'Em Atraso' : 'Não Pago';
            }

            $monthlyData[] = [
                'month'            => $monthName,
                'year'             => $currentYear,
                'reference'        => $this->getPaymentReferenceSimple($monthName, $currentYear),
                'fee_names'        => $feeNames,
                'due_date'         => $dueDate,
                'status'           => $status,
                'base_amount'      => $baseAmount,       // BASE
                'fine_amount'      => $fineAmount,       // multa (real ou prevista)
                'discount_applied' => $discountAmount,
                'paid_amount'      => $netPaidAmount,    // LÍQUIDO
                'payment_date'     => $paymentDate,
                'is_paid'          => $isPaid,
                'is_overdue'       => $status === 'Em Atraso',
            ];

            if ($baseAmount > 0) {
                $summary['total_annual_value'] += $baseAmount; // esperado = soma das bases
                $summary['months_with_fees']++;
            }

            if ($isPaid) {
                $summary['total_paid']            += $baseAmount;    // base recebida
                $summary['total_fines_paid']      += $fineAmount;
                $summary['total_discounts_given'] += $discountAmount;
                $summary['total_net_received']    += $netPaidAmount; // líquido
                $summary['months_paid_count']++;
            }
        }

        $summary['payment_rate'] = $summary['months_with_fees'] > 0
            ? round(($summary['months_paid_count'] / $summary['months_with_fees']) * 100, 1)
            : 0;

        return [
            'monthlyData' => $monthlyData,
            'summary'     => $summary,
            'student'     => $this->selectedStudent,
            'year'        => $currentYear,
        ];
    }

    /* ------------------------------ Export PDF ------------------------------ */

    public function exportPdf()
    {
        try {
            $data = $this->getExportData();

            // Buscar referências de pagamento agrupadas (incluindo fee_assigns sem referência)
            $paymentReferences = $this->getAllPaymentReferencesAndFees($this->selectedYear ?? now()->year);

            $pdf = Pdf::loadView('exports.student-finance-pdf', [
                'student'      => $this->selectedStudent,
                'year'         => $this->selectedYear,
                'data'         => $data,
                'payment_references' => $paymentReferences,
                'generated_at' => now()->format('d/m/Y H:i'),
            ]);

            $filename = "extrato_financeiro_{$this->selectedStudent->student_id}_{$this->selectedYear}.pdf";

            return response()->streamDownload(function() use ($pdf) {
                echo $pdf->output();
            }, $filename, [
                'Content-Type' => 'application/pdf',
            ]);
        } catch (\Exception $e) {
            \Log::error('PDF export failed', ['error' => $e->getMessage()]);
            session()->flash('error', 'Erro ao exportar PDF: ' . $e->getMessage());
        }
    }

    private function getExportData()
    {
        $currentYear = $this->selectedYear ?? now()->year;
        $months = ['Janeiro','Fevereiro','Março','Abril','Maio','Junho','Julho','Agosto','Setembro','Outubro','Novembro','Dezembro'];

        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        $filteredFeeStructures = [];
        if ($studentClassroom) {
            $feeStructures = FeeStructure::where('active', 1)->get();
            foreach ($feeStructures as $fs) {
                $grades = explode(',', $fs->grades);
                if (in_array($studentClassroom, $grades)) $filteredFeeStructures[] = $fs;
            }
        }

        $exportRows = [];
        $summary = [
            'total_annual_value'    => 0.0,
            'total_paid'            => 0.0, // base recebida
            'total_due'             => 0.0,
            'total_overdue'         => 0.0,
            'total_fines_paid'      => 0.0,
            'total_discounts_given' => 0.0,
            'total_net_received'    => 0.0,
            'months_with_fees'      => 0,
            'months_paid_count'     => 0,
        ];

        foreach ($months as $monthName) {
            try {
                $calc = $this->feeCalculationService->calculateFeeForStudent(
                    $this->selectedStudent, $monthName, $currentYear
                );
                if ($calc['base_amount'] <= 0) continue;

                $existing = $this->feeCalculationService->getPaymentForPeriod(
                    $this->selectedStudent, $monthName, $currentYear
                );

                $isPaid = (bool) $existing;
                [$basePaid, $finePaid, $discount, $netPaid] = $this->splitPayment($existing);

                $fineToShow = $isPaid ? $finePaid : (float) $calc['fine_amount'];
                $status = $isPaid ? 'Pago' : ($calc['is_late_payment'] ? 'Em Atraso' : 'Não Pago');
                $feeNames = $this->getFeeNamesForMonth($filteredFeeStructures, $monthName);
                $dueDate = $calc['due_date'] ? $calc['due_date']->format('d/m/Y') : 'N/A';
                $paymentDate = $isPaid ? $existing->created_at->format('d/m/Y H:i') : '--';

                // linhas para o PDF/Excel simples
                $exportRows[] = [
                    $monthName . ' ' . $currentYear,
                    $feeNames,
                    $dueDate,
                    $status,
                    number_format($calc['base_amount'], 2, ',', '.'),
                    number_format($fineToShow, 2, ',', '.'),
                    number_format($discount, 2, ',', '.'),
                    number_format($netPaid, 2, ',', '.'),
                    $paymentDate,
                ];

                // totais (esperado = soma das bases)
                $summary['total_annual_value'] += (float) $calc['base_amount'];
                $summary['months_with_fees']++;

                if ($isPaid) {
                    $summary['total_paid']            += (float) $basePaid; // base recebida
                    $summary['total_fines_paid']      += (float) $finePaid;
                    $summary['total_discounts_given'] += (float) $discount;
                    $summary['total_net_received']    += (float) $netPaid;
                    $summary['months_paid_count']++;
                } else {
                    if ($calc['is_late_payment']) {
                        $summary['total_overdue'] += (float) $calc['base_amount'];
                    } else {
                        $summary['total_due']     += (float) $calc['base_amount'];
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Error preparing export data', [
                    'month' => $monthName,
                    'year'  => $currentYear,
                    'error' => $e->getMessage(),
                ]);
                continue;
            }
        }

        $summary['payment_rate'] = $summary['months_with_fees'] > 0
            ? round(($summary['months_paid_count'] / $summary['months_with_fees']) * 100, 1)
            : 0;

        return [
            'rows'    => $exportRows,
            'summary' => $summary,
            'student' => $this->selectedStudent,
            'year'    => $currentYear,
        ];
    }

    /* ------------------------------ UI helpers ------------------------------ */

    public function openModal($feeAssignId, $amount, $month, $year)
    {
        $this->selectedFeeAssignId = $feeAssignId;
        $this->month = (string) $month;
        $this->year  = (int) $year;

        $this->calculateCurrentFee();

        $this->dispatchBrowserEvent('openModal');
        $this->emit('openModal', $feeAssignId, $amount, $month, $year);
    }

    public function setMonth($month) { $this->month = $month; $this->calculateCurrentFee(); }
    public function setYear($year)   { $this->year  = $year;  $this->calculateCurrentFee(); }
    public function addFine($amount) { $this->fine += $amount; $this->calculateCurrentFee(); }
    public function addDiscount($amount) { $this->discount += $amount; $this->calculateCurrentFee(); }
    public function resetCalculation()
    {
        $this->fine = 0;
        $this->discount = 0;
        $this->month = null;
        $this->year = null;
        $this->calculateCurrentFee();
    }

    public function checkExistingPayment($month, $year)
    {
        return $this->feeCalculationService->hasPaymentForPeriod(
            $this->selectedStudent, $month, $year
        );
    }

    public function getCalculationDetailsProperty()
    {
        return [
            'base_fee'             => $this->calculatedBaseFee,
            'calculated_auto_fine' => max(0, $this->calculatedFine - (float) $this->fine),
            'manual_fine'          => (float) $this->fine,
            'total_fine'           => $this->calculatedFine,
            'discount'             => (float) $this->discount,
            'total'                => $this->calculatedTotal,
            'is_late'              => $this->isLatePayment,
            'due_date'             => $this->dueDate?->format('d/m/Y'),
            'month'                => $this->month ?: 'Current',
            'year'                 => $this->year ?: date('Y'),
            'preservation_note'    => 'As multas são preservadas no histórico para auditoria completa',
        ];
    }

    /* ------------------------------ Ver Referência ------------------------------ */

    /**
     * Abrir modal para ver detalhes da referência
     */
    public function viewReference($paymentId)
    {
        try {
            $payment = Fee_assign::with(['student.classroom', 'paymentReference'])->findOrFail($paymentId);

            // Buscar referência associada
            $reference = $payment->paymentReference;

            // Se não houver relação direta, buscar por student_id, month e year
            if (!$reference) {
                $reference = \App\Models\PaymentReference::where('student_id', $payment->student_id)
                    ->where('fee_month', $payment->month)
                    ->where('fee_year', $payment->year)
                    ->first();
            }

            if ($reference) {
                $this->selectedReference = [
                    'id' => $reference->id,
                    'reference_number' => $reference->reference_number,
                    'entity_code' => $reference->entity_code,
                    'amount' => $reference->amount,
                    'status' => $reference->status,
                    'fee_month' => $reference->fee_month,
                    'fee_year' => $reference->fee_year,
                    'expires_at' => $reference->expires_at?->format('d/m/Y H:i'),
                    'paid_at' => $reference->paid_at?->format('d/m/Y H:i'),
                    'created_at' => $reference->created_at?->format('d/m/Y H:i'),
                    'student_name' => $payment->student->name ?? 'N/A',
                    'student_id' => $payment->student->student_id ?? 'N/A',
                    'class' => ($payment->student->classroom->class ?? '') . ' - ' . ($payment->student->classroom->name ?? ''),
                ];

                $this->showReferenceModal = true;
            } else {
                session()->flash('error', 'Referência não encontrada para este pagamento.');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'Erro ao carregar detalhes da referência: ' . $e->getMessage());
        }
    }

    /**
     * Ver referência por mês e ano (para referências geradas mas não pagas)
     */
    public function viewReferenceByMonth($month, $year)
    {
        try {
            // Converter mês de inglês para português (payment_references usa português)
            $monthMap = [
                'January' => 'Janeiro', 'February' => 'Fevereiro', 'March' => 'Março',
                'April' => 'Abril', 'May' => 'Maio', 'June' => 'Junho',
                'July' => 'Julho', 'August' => 'Agosto', 'September' => 'Setembro',
                'October' => 'Outubro', 'November' => 'Novembro', 'December' => 'Dezembro'
            ];
            $monthInPortuguese = $monthMap[$month] ?? $month;

            // Buscar referência gerada para este mês
            $reference = \App\Models\PaymentReference::where('student_id', $this->studentId)
                ->where('fee_month', $monthInPortuguese)
                ->where('fee_year', $year)
                ->first();

            if ($reference) {
                $this->selectedReference = [
                    'id' => $reference->id,
                    'reference_number' => $reference->reference_number,
                    'entity_code' => $reference->entity_code,
                    'amount' => $reference->amount,
                    'status' => $reference->status,
                    'fee_month' => $reference->fee_month,
                    'fee_year' => $reference->fee_year,
                    'expires_at' => $reference->expires_at?->format('d/m/Y H:i'),
                    'paid_at' => $reference->paid_at?->format('d/m/Y H:i'),
                    'created_at' => $reference->created_at?->format('d/m/Y H:i'),
                    'student_name' => $this->selectedStudent->name ?? 'N/A',
                    'student_id' => $this->selectedStudent->student_id ?? 'N/A',
                    'class' => ($this->selectedStudent->classroom->class ?? '') . ' - ' . ($this->selectedStudent->classroom->name ?? ''),
                ];

                $this->showReferenceModal = true;
            } else {
                session()->flash('error', 'Referência não encontrada para este mês.');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'Erro ao carregar detalhes da referência: ' . $e->getMessage());
        }
    }

    /**
     * Ver referência por ID (chamado da tabela de referências)
     * Suporta tanto PaymentReference quanto fee_assigns (pagamentos diretos)
     */
    public function viewReferenceById($referenceId)
    {
        try {
            // Verificar se é um pagamento direto (fee_assign)
            if (str_starts_with($referenceId, 'fee_')) {
                $feeAssignId = str_replace('fee_', '', $referenceId);
                $feeAssign = \App\Models\Fee_assign::with('student.classroom')->findOrFail($feeAssignId);

                // Buscar nomes das taxas
                $studentClassroom = $feeAssign->student->class->class ?? $feeAssign->student->classroom->class ?? null;
                $feeNames = 'Pagamento Direto';
                if ($studentClassroom) {
                    $feeStructures = FeeStructure::where('active', 1)
                        ->whereRaw("FIND_IN_SET(?, grades)", [$studentClassroom])
                        ->get();

                    $feeNames = $feeStructures->filter(function($s) use ($feeAssign) {
                        return empty($s->months) || in_array($feeAssign->month, explode(',', $s->months));
                    })->pluck('name')->implode(', ') ?: 'Pagamento Direto';
                }

                $this->selectedReference = [
                    'id' => $feeAssign->id,
                    'reference_number' => 'N/A',
                    'entity_code' => 'N/A',
                    'amount' => $feeAssign->amount + ($feeAssign->fine ?? 0),
                    'status' => 'paid',
                    'approval_status' => 'approved',
                    'fee_month' => $feeAssign->month,
                    'fee_year' => $feeAssign->year,
                    'expires_at' => 'N/A',
                    'paid_at' => $feeAssign->payment_date ? date('d/m/Y H:i', strtotime($feeAssign->payment_date)) : $feeAssign->created_at?->format('d/m/Y H:i'),
                    'created_at' => $feeAssign->created_at?->format('d/m/Y H:i'),
                    'student_name' => $feeAssign->student->name ?? 'N/A',
                    'student_id' => $feeAssign->student->student_id ?? 'N/A',
                    'class' => ($feeAssign->student->classroom->class ?? '') . ' - ' . ($feeAssign->student->classroom->name ?? ''),
                    'description' => $feeNames,
                    'is_direct_payment' => true,
                ];
            } else {
                // PaymentReference normal
                $reference = \App\Models\PaymentReference::with('student.classroom')->findOrFail($referenceId);

                $this->selectedReference = [
                    'id' => $reference->id,
                    'reference_number' => $reference->reference_number,
                    'entity_code' => $reference->entity_code,
                    'amount' => $reference->amount,
                    'status' => $reference->status,
                    'approval_status' => $reference->approval_status,
                    'fee_month' => $reference->fee_month,
                    'fee_year' => $reference->fee_year,
                    'expires_at' => $reference->expires_at?->format('d/m/Y H:i'),
                    'paid_at' => $reference->paid_at?->format('d/m/Y H:i'),
                    'created_at' => $reference->created_at?->format('d/m/Y H:i'),
                    'student_name' => $reference->student->name ?? 'N/A',
                    'student_id' => $reference->student->student_id ?? 'N/A',
                    'class' => ($reference->student->classroom->class ?? '') . ' - ' . ($reference->student->classroom->name ?? ''),
                    'description' => $reference->metadata['description'] ?? $reference->metadata['custom_fee_type'] ?? 'N/A',
                    'is_direct_payment' => false,
                ];
            }

            $this->showReferenceModal = true;
        } catch (\Exception $e) {
            session()->flash('error', 'Erro ao carregar detalhes da referência: ' . $e->getMessage());
        }
    }

    /**
     * Fechar modal de referência
     */
    public function closeReferenceModal()
    {
        $this->showReferenceModal = false;
        $this->selectedReference = null;
    }

    /**
     * Buscar todas as referências de pagamento E fee_assigns sem referência
     * para mostrar tudo agrupado
     */
    private function getAllPaymentReferencesAndFees($year)
    {
        // 1. Buscar todas as referências de pagamento
        $paymentReferences = \App\Models\PaymentReference::where('student_id', $this->studentId)
            ->where('fee_year', $year)
            ->get();

        // 2. Buscar fee_assigns sem referência de pagamento
        $feeAssigns = \App\Models\Fee_assign::where('student_id', $this->studentId)
            ->where('year', $year)
            ->whereDoesntHave('paymentReference')
            ->get();

        // Buscar fee structures para obter nomes das taxas
        $studentClassroom = $this->selectedStudent->class->class ?? $this->selectedStudent->classroom->class ?? null;
        $feeStructures = collect();
        if ($studentClassroom) {
            $feeStructures = FeeStructure::where('active', 1)
                ->whereRaw("FIND_IN_SET(?, grades)", [$studentClassroom])
                ->get();
        }

        // 3. Converter fee_assigns para o formato de PaymentReference
        $convertedFees = $feeAssigns->map(function($fee) use ($feeStructures) {
            // Buscar nomes das taxas para este mês
            $feeNames = $feeStructures->filter(function($s) use ($fee) {
                return empty($s->months) || in_array($fee->month, explode(',', $s->months));
            })->pluck('name')->implode(', ') ?: 'Pagamento Direto';

            // Criar um objeto fake que parece PaymentReference
            $fakeReference = new \stdClass();
            $fakeReference->id = 'fee_' . $fee->id;
            $fakeReference->reference_number = 'N/A'; // Sem referência BCI
            $fakeReference->entity_code = 'N/A';
            $fakeReference->amount = $fee->amount;
            $fakeReference->fine_amount = $fee->fine ?? 0;
            $fakeReference->discount_amount = $fee->discount ?? 0;
            $fakeReference->status = 'paid'; // fee_assign já foi pago
            $fakeReference->approval_status = 'approved';
            $fakeReference->fee_month = $fee->month;
            $fakeReference->fee_year = $fee->year;
            $fakeReference->expires_at = null;
            $fakeReference->paid_at = $fee->created_at;
            $fakeReference->created_at = $fee->created_at;
            $fakeReference->updated_at = $fee->updated_at;

            // Metadata com descrição baseada no fee_assign
            $fakeReference->metadata = [
                'description' => $feeNames,
                'is_fee_assign' => true,
                'fee_assign_id' => $fee->id,
            ];

            return $fakeReference;
        });

        // 4. Combinar e retornar como collection
        return $paymentReferences->concat($convertedFees);
    }

    /**
     * Baixar PDF da referência de pagamento
     */
    public function downloadReferencePdf()
    {
        if (!$this->selectedReference) {
            session()->flash('error', 'Nenhuma referência selecionada.');
            return;
        }

        try {
            $data = [
                'reference' => $this->selectedReference,
                'title' => 'Referência de Pagamento'
            ];

            $pdf = Pdf::loadView('admin.fee.reference-pdf', $data);

            // Criar nome do arquivo: 00001250387_NomeEstudante.pdf
            $studentName = $this->selectedReference['student_name'];
            // Remover espaços e caracteres especiais do nome
            $studentNameClean = preg_replace('/[^a-zA-Z0-9]/', '', $studentName);
            $refNumber = $this->selectedReference['reference_number'];
            $filename = $refNumber . '_' . $studentNameClean . '.pdf';

            return response()->streamDownload(function() use ($pdf) {
                echo $pdf->output();
            }, $filename);
        } catch (\Exception $e) {
            session()->flash('error', 'Erro ao gerar PDF: ' . $e->getMessage());
        }
    }

    /* ------------------------------ Render ------------------------------ */

    public function render()
    {
        $studentClassroom = $this->selectedStudent->class->class
            ?? $this->selectedStudent->classroom->class
            ?? null;

        $filteredFeeStructures = [];
        if ($studentClassroom) {
            $feeStructures = FeeStructure::where('active', 1)->get();
            foreach ($feeStructures as $fs) {
                $gradesArray = explode(',', $fs->grades);
                if (in_array($studentClassroom, $gradesArray)) $filteredFeeStructures[] = $fs;
            }
        }

        $currentYear = $this->selectedYear ?? date('Y');
        $months = ['Janeiro','Fevereiro','Março','Abril','Maio','Junho','Julho','Agosto','Setembro','Outubro','Novembro','Dezembro'];

        $monthlyData = [];

        // Totais compatibilidade
        $totalExpected = 0.0; // AGORA = soma das BASES
        $totalPaid     = 0.0; // líquido recebido
        $totalPending  = 0.0; // base pendente
        $totalOverdue  = 0.0; // base atrasada

        // Novos totais (widgets)
        $totalAnnualValue   = 0.0;
        $totalBasePaid      = 0.0;
        $totalDue           = 0.0;
        $totalOverdueBase   = 0.0;
        $totalFinesPaid     = 0.0;
        $totalDiscountsGiven= 0.0;
        $totalNetReceived   = 0.0;
        $monthsWithFees     = 0;
        $monthsPaidCount    = 0;

        foreach ($months as $monthName) {
            try {
                $calculation = $this->feeCalculationService->calculateFeeForStudent(
                    $this->selectedStudent, $monthName, $currentYear
                );
                if ($calculation['base_amount'] <= 0) continue;

                // NOVO: Buscar TODOS os pagamentos (múltiplas taxas separadas)
                $existingPayments = $this->feeCalculationService->getPaymentsForPeriod(
                    $this->selectedStudent, $monthName, $currentYear
                );

                $isPaid = $existingPayments->isNotEmpty();

                if ($isPaid) {
                    // Verificar se TODOS os pagamentos têm fee_structure_id (novos) ou NENHUM tem (antigos)
                    $allHaveFeeStructureId = $existingPayments->every(fn($p) => !empty($p->fee_structure_id));

                    if ($allHaveFeeStructureId && $existingPayments->count() > 1) {
                        // NOVO SISTEMA: Múltiplos pagamentos com fee_structure_id - mostrar separadamente
                        foreach ($existingPayments as $payment) {
                            [$baseAmountPaid, $fineAmountPaid, $discountAmount, $netPaidAmount] = $this->splitPayment($payment);

                            $feeName = 'Taxa Escolar';
                            if ($payment->feeStructure) {
                                $feeName = $payment->feeStructure->fee_name ?? $payment->feeStructure->name ?? 'Taxa Escolar';
                            }

                            $monthlyData[] = [
                                'month'            => $monthName,
                                'year'             => $currentYear,
                                'fee_names'        => $feeName,
                                'due_date'         => $calculation['due_date'] ? $calculation['due_date']->format('d/m/Y') : 'N/A',
                                'status'           => 'Pago',
                                'base_amount'      => $baseAmountPaid,
                                'fine_amount'      => $fineAmountPaid,
                                'total_expected'   => $baseAmountPaid,
                                'discount_applied' => $discountAmount,
                                'paid_amount'      => $netPaidAmount,
                                'balance_amount'   => 0.0,
                                'payment_date'     => $payment->created_at->format('d/m/Y H:i'),
                                'is_paid'          => true,
                                'is_overdue'       => false,
                                'existing_payment' => $payment,
                                'data_status'      => 'paid',
                            ];

                            $totalBasePaid       += $baseAmountPaid;
                            $totalFinesPaid      += $fineAmountPaid;
                            $totalDiscountsGiven += $discountAmount;
                            $totalNetReceived    += $netPaidAmount;
                            $totalPaid           += $netPaidAmount;
                        }
                    } else {
                        // SISTEMA ANTIGO ou pagamento único: Mostrar agrupado
                        $totalBaseAmountPaid = 0;
                        $totalFineAmountPaid = 0;
                        $totalDiscountAmount = 0;
                        $totalNetPaidAmount = 0;
                        $paymentDate = 'N/A';
                        $firstPayment = null;

                        foreach ($existingPayments as $payment) {
                            [$baseAmountPaid, $fineAmountPaid, $discountAmount, $netPaidAmount] = $this->splitPayment($payment);
                            $totalBaseAmountPaid += $baseAmountPaid;
                            $totalFineAmountPaid += $fineAmountPaid;
                            $totalDiscountAmount += $discountAmount;
                            $totalNetPaidAmount += $netPaidAmount;

                            if (!$firstPayment) {
                                $firstPayment = $payment;
                                $paymentDate = $payment->created_at->format('d/m/Y H:i');
                            }

                            $totalBasePaid       += $baseAmountPaid;
                            $totalFinesPaid      += $fineAmountPaid;
                            $totalDiscountsGiven += $discountAmount;
                            $totalNetReceived    += $netPaidAmount;
                            $totalPaid           += $netPaidAmount;
                        }

                        $monthlyData[] = [
                            'month'            => $monthName,
                            'year'             => $currentYear,
                            'fee_names'        => $this->getFeeNamesForMonth($filteredFeeStructures, $monthName),
                            'due_date'         => $calculation['due_date'] ? $calculation['due_date']->format('d/m/Y') : 'N/A',
                            'status'           => 'Pago',
                            'base_amount'      => $totalBaseAmountPaid,
                            'fine_amount'      => $totalFineAmountPaid,
                            'total_expected'   => $totalBaseAmountPaid,
                            'discount_applied' => $totalDiscountAmount,
                            'paid_amount'      => $totalNetPaidAmount,
                            'balance_amount'   => 0.0,
                            'payment_date'     => $paymentDate,
                            'is_paid'          => true,
                            'is_overdue'       => false,
                            'existing_payment' => $firstPayment,
                            'data_status'      => 'paid',
                        ];
                    }

                    $monthsWithFees++;
                    $monthsPaidCount++;
                    $totalAnnualValue += (float) $calculation['base_amount'];
                    $totalExpected += (float) $calculation['base_amount'];

                } else {
                    // Sem pagamento: mostrar CADA TAXA SEPARADAMENTE como pendência
                    $feeStructuresApplied = collect($calculation['fee_structures_applied'] ?? []);

                    if ($feeStructuresApplied->isEmpty()) {
                        // Fallback: mostrar linha única se não houver fee structures
                        $fineAmountToShow = (float) $calculation['fine_amount'];
                        $status = $calculation['is_late_payment'] ? 'Em Atraso' : 'Não Pago';
                        $balanceAmount = (float) $calculation['base_amount'];

                        $monthlyData[] = [
                            'month'            => $monthName,
                            'year'             => $currentYear,
                            'fee_names'        => 'Taxa Escolar',
                            'due_date'         => $calculation['due_date'] ? $calculation['due_date']->format('d/m/Y') : 'N/A',
                            'status'           => $status,
                            'base_amount'      => (float) $calculation['base_amount'],
                            'fine_amount'      => $fineAmountToShow,
                            'total_expected'   => (float) $calculation['base_amount'],
                            'discount_applied' => 0,
                            'paid_amount'      => 0,
                            'balance_amount'   => $balanceAmount,
                            'payment_date'     => 'N/A',
                            'is_paid'          => false,
                            'is_overdue'       => (bool) $calculation['is_late_payment'],
                            'existing_payment' => null,
                            'data_status'      => $calculation['is_late_payment'] ? 'overdue' : 'pending',
                        ];
                    } else {
                        // Mostrar CADA FeeStructure como linha separada (mesmo sem pagamento)
                        $totalBaseFee = $calculation['base_amount'];
                        $totalFineAmount = $calculation['fine_amount'];

                        foreach ($feeStructuresApplied as $feeStructure) {
                            $feeName = $feeStructure['name'] ?? $feeStructure['fee_name'] ?? 'Taxa Escolar';
                            $feeAmount = (float)($feeStructure['monthly_fee'] ?? 0);

                            if ($feeAmount <= 0) continue;

                            // Calcular proporção desta taxa no total
                            $proportion = $totalBaseFee > 0 ? ($feeAmount / $totalBaseFee) : 0;
                            $proportionalFine = round($totalFineAmount * $proportion, 2);

                            $status = $calculation['is_late_payment'] ? 'Em Atraso' : 'Não Pago';

                            $monthlyData[] = [
                                'month'            => $monthName,
                                'year'             => $currentYear,
                                'fee_names'        => $feeName, // Nome específico da taxa
                                'due_date'         => $calculation['due_date'] ? $calculation['due_date']->format('d/m/Y') : 'N/A',
                                'status'           => $status,
                                'base_amount'      => $feeAmount,
                                'fine_amount'      => $proportionalFine,
                                'total_expected'   => $feeAmount,
                                'discount_applied' => 0,
                                'paid_amount'      => 0,
                                'balance_amount'   => $feeAmount,
                                'payment_date'     => 'N/A',
                                'is_paid'          => false,
                                'is_overdue'       => (bool) $calculation['is_late_payment'],
                                'existing_payment' => null,
                                'data_status'      => $calculation['is_late_payment'] ? 'overdue' : 'pending',
                            ];
                        }
                    }

                    // Totais (já calculado pelo total base, não por cada taxa individual)
                    $monthsWithFees++;
                    $totalAnnualValue += (float) $calculation['base_amount'];
                    $totalExpected += (float) $calculation['base_amount'];

                    if ($calculation['is_late_payment']) {
                        $totalOverdueBase += (float) $calculation['base_amount'];
                        $totalOverdue += (float) $calculation['base_amount'];
                    } else {
                        $totalDue += (float) $calculation['base_amount'];
                        $totalPending += (float) $calculation['base_amount'];
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Error calculating month data', [
                    'month' => $monthName,
                    'year'  => $currentYear,
                    'student_id' => $this->studentId,
                    'error' => $e->getMessage(),
                ]);

                $monthlyData[] = [
                    'month' => $monthName,
                    'year'  => $currentYear,
                    'fee_names' => 'Taxa Escolar',
                    'due_date' => 'N/A',
                    'status' => 'Erro',
                    'base_amount' => 0,
                    'fine_amount' => 0,
                    'total_expected' => 0,
                    'discount_applied' => 0,
                    'paid_amount' => 0,
                    'balance_amount' => 0,
                    'payment_date' => 'N/A',
                    'is_paid' => false,
                    'is_overdue' => false,
                    'existing_payment' => null,
                    'data_status' => 'error',
                ];
            }
        }

        $paymentRate = $monthsWithFees > 0 ? round(($monthsPaidCount / $monthsWithFees) * 100, 1) : 0;

        $fees = Fee_assign::where('student_id', $this->studentId)
            ->where('year', $currentYear)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($fee) {
                $fee->total_expected   = $fee->amount + $fee->fine - $fee->discount;
                $fee->has_fine         = $fee->fine > 0;
                $fee->fine_percentage  = $fee->amount > 0 ? round(($fee->fine / $fee->amount) * 100, 1) : 0;
                $fee->payment_status   = 'complete';
                return $fee;
            });

        $discounts = discount::where('student_id', $this->studentId)->get();

        return view('livewire.admin.fee.fee-collection', [
            'student'          => $this->selectedStudent,
            'fees'             => $fees,
            'fee'              => $this->calculatedTotal,
            'feestructures'    => $filteredFeeStructures,
            'discounts'        => $discounts,
            'calculatedBaseFee'=> $this->calculatedBaseFee,
            'calculatedFine'   => $this->calculatedFine,
            'calculatedTotal'  => $this->calculatedTotal,
            'isLatePayment'    => $this->isLatePayment,
            'dueDate'          => $this->dueDate,

            // Tabela mensal
            'monthlyData' => $monthlyData,

            // Resumo compatível
            'yearSummary' => [
                'total_expected' => $totalExpected, // soma das bases
                'total_paid'     => $totalPaid,     // líquido recebido
                'total_pending'  => $totalPending,  // base pendente
                'total_overdue'  => $totalOverdue,  // base atrasada
                'current_year'   => $currentYear,
            ],

            'selectedYear' => $this->selectedYear,
            'payment_references' => $this->getAllPaymentReferencesAndFees($currentYear),

            // Widgets
            'widgetData' => [
                'total_annual_value'    => $totalAnnualValue,
                'total_base_paid'       => $totalBasePaid,
                'total_net_received'    => $totalNetReceived,
                'total_due'             => $totalDue,
                'total_overdue_base'    => $totalOverdueBase,
                'total_fines_paid'      => $totalFinesPaid,
                'total_discounts_given' => $totalDiscountsGiven,
                'payment_rate'          => $paymentRate,
                'months_with_fees'      => $monthsWithFees,
                'months_paid_count'     => $monthsPaidCount,
            ],

            'analysisData' => [
                'total_students_in_class' => 1,
                'average_payment_delay'   => 0,
                'most_delayed_month'      => null,
                'payment_trend'           => 'stable',
            ],

            'uiConfig' => [
                'show_advanced_stats' => true,
                'enable_bulk_actions' => false,
                'default_payment_mode'=> 'Cash',
                'currency_symbol'     => 'MT',
                'date_format'         => 'd/m/Y',
                'enable_debug_mode'   => app()->environment('local'),
            ],

            'reportMetadata' => [
                'generated_at'          => now(),
                'generated_by'          => auth()->user()->name ?? 'Sistema',
                'academic_year'         => $currentYear,
                'student_class'         => $studentClassroom,
                'total_months_analyzed' => count($months),
                'months_with_data'      => count(array_filter($monthlyData, fn($m) => $m['base_amount'] > 0)),
                'data_quality_score'    => $paymentRate,
            ],
        ]);
    }

    /**
     * Baixar recibo de pagamento direto (Cash, POS, etc.)
     */
    public function downloadPaymentReceipt($feeAssignId)
    {
        try {
            $feeAssign = Fee_assign::findOrFail($feeAssignId);

            // Gerar ou obter número de recibo sequencial
            $receiptNumber = \App\Services\ReceiptNumberService::getOrGenerateReceiptNumber(
                'fee_assign',
                $feeAssign->id,
                'admin',
                auth()->id() ? (int)auth()->id() : null
            );

            // Preparar dados compatíveis com receipt-official.blade.php
            $baseAmount = $feeAssign->amount;
            $fineAmount = $feeAssign->fine ?? 0;
            $discountAmount = $feeAssign->discount ?? 0;
            $totalPaid = $baseAmount + $fineAmount - $discountAmount;

            // Criar objeto simulado para compatibilidade
            $paymentData = (object)[
                'id' => $feeAssign->id,
                'fee_month' => $feeAssign->month,
                'fee_year' => $feeAssign->year,
            ];

            // Determinar método de pagamento
            $paymentMethod = 'Dinheiro';
            if ($feeAssign->payment_mode === 'POS' || strtolower($feeAssign->pay_type ?? '') === 'pos') {
                $paymentMethod = 'POS';
            }

            // Gerar PDF do recibo oficial
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.receipt-official', [
                'payment' => $paymentData,
                'student' => $this->selectedStudent,
                'base_amount' => $baseAmount,
                'fine_amount' => $fineAmount,
                'discount_amount' => $discountAmount,
                'amount_paid' => $totalPaid,
                'paid_at' => \Carbon\Carbon::parse($feeAssign->payment_date ?? $feeAssign->created_at),
                'payment_method' => $paymentMethod,
                'transaction_id' => $feeAssign->transaction_id ?? null,
                'receipt_number' => $receiptNumber,
            ]);

            return response()->streamDownload(function() use ($pdf) {
                echo $pdf->output();
            }, "recibo_pagamento_{$feeAssign->student_id}_{$feeAssign->month}_{$feeAssign->year}.pdf");

        } catch (\Exception $e) {
            \Log::error('Erro ao gerar recibo de pagamento', [
                'fee_assign_id' => $feeAssignId,
                'error' => $e->getMessage()
            ]);

            session()->flash('error', 'Erro ao gerar recibo. Tente novamente.');
        }
    }

    /**
     * Baixar comprovativo de pagamento (ADMIN)
     */
    public function downloadReceipt($paymentId)
    {
        try {
            $payment = \App\Models\PaymentReference::findOrFail($paymentId);

            // Verificar se o pagamento está pago
            if ($payment->status !== 'paid') {
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Comprovativo disponível apenas para pagamentos confirmados'
                ]);
                return;
            }

            // Usar o serviço para gerar/obter o comprovativo
            $receiptService = new \App\Services\PaymentReceiptService();
            $result = $receiptService->downloadReceipt($payment);

            if ($result['success']) {
                $this->dispatchBrowserEvent('download-receipt', [
                    'url' => $result['url'],
                    'message' => 'Comprovativo gerado com sucesso!'
                ]);
            } else {
                $this->dispatchBrowserEvent('error', [
                    'message' => $result['message'] ?? 'Erro ao gerar comprovativo'
                ]);
            }

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Erro ao baixar comprovativo (Admin)', [
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);

            $this->dispatchBrowserEvent('error', [
                'message' => 'Erro ao processar comprovativo. Tente novamente.'
            ]);
        }
    }

    /**
     * Baixar recibo oficial de pagamento (ADMIN)
     */
    public function downloadOfficialReceipt($paymentId)
    {
        try {
            $payment = \App\Models\PaymentReference::findOrFail($paymentId);

            // Verificar se o pagamento está pago
            if ($payment->status !== 'paid') {
                $this->dispatchBrowserEvent('error', [
                    'message' => 'Recibo disponível apenas para pagamentos confirmados'
                ]);
                return;
            }

            // Usar o serviço para gerar/obter o recibo oficial
            $receiptService = new \App\Services\PaymentReceiptService();
            $result = $receiptService->downloadOfficialReceipt($payment);

            if ($result['success']) {
                $this->dispatchBrowserEvent('download-receipt', [
                    'url' => $result['url'],
                    'message' => 'Recibo oficial gerado com sucesso!'
                ]);
            } else {
                $this->dispatchBrowserEvent('error', [
                    'message' => $result['message'] ?? 'Erro ao gerar recibo oficial'
                ]);
            }

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Erro ao baixar recibo oficial (Admin)', [
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);

            $this->dispatchBrowserEvent('error', [
                'message' => 'Erro ao processar recibo oficial. Tente novamente.'
            ]);
        }
    }
}

