<?php

namespace App\Http\Livewire\Admin\Fee;

use Livewire\Component;
use App\Models\FeeMonthDeadline;
use Carbon\Carbon;

class FeeMonthDeadlineManager extends Component
{
    public $year;
    public $deadlines = [];
    public $editingMonth = null;
    public $deadline;
    public $allowLatePayment;
    public $notes;

    protected $months = [
        'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
        'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'
    ];

    public function mount()
    {
        $this->year = now()->year;
        $this->loadDeadlines();
    }

    public function loadDeadlines()
    {
        $this->deadlines = [];

        foreach ($this->months as $month) {
            $deadline = FeeMonthDeadline::where('month', $month)
                ->where('year', $this->year)
                ->first();

            $this->deadlines[$month] = [
                'exists' => $deadline !== null,
                'deadline' => $deadline?->deadline?->format('Y-m-d'),
                'allow_late_payment' => $deadline?->allow_late_payment ?? true,
                'notes' => $deadline?->notes,
                'is_expired' => $deadline && $deadline->isExpired(),
            ];
        }
    }

    public function updatedYear()
    {
        $this->loadDeadlines();
    }

    public function editMonth($month)
    {
        $this->editingMonth = $month;
        $this->deadline = $this->deadlines[$month]['deadline'] ?? null;
        $this->allowLatePayment = $this->deadlines[$month]['allow_late_payment'] ?? true;
        $this->notes = $this->deadlines[$month]['notes'] ?? '';
    }

    public function saveDeadline()
    {
        $this->validate([
            'deadline' => 'nullable|date',
            'allowLatePayment' => 'boolean',
            'notes' => 'nullable|string|max:500',
        ]);

        FeeMonthDeadline::updateOrCreate(
            [
                'month' => $this->editingMonth,
                'year' => $this->year,
            ],
            [
                'deadline' => $this->deadline,
                'allow_late_payment' => $this->allowLatePayment,
                'notes' => $this->notes,
            ]
        );

        $this->loadDeadlines();
        $this->editingMonth = null;
        $this->reset(['deadline', 'allowLatePayment', 'notes']);

        session()->flash('message', 'Prazo configurado com sucesso!');
    }

    public function removeDeadline($month)
    {
        FeeMonthDeadline::where('month', $month)
            ->where('year', $this->year)
            ->delete();

        $this->loadDeadlines();
        session()->flash('message', 'Prazo removido com sucesso!');
    }

    public function cancelEdit()
    {
        $this->editingMonth = null;
        $this->reset(['deadline', 'allowLatePayment', 'notes']);
    }

    public function setQuickDeadline($month, $type)
    {
        $monthIndex = array_search($month, $this->months);
        $date = null;

        switch ($type) {
            case 'end_of_month':
                $date = Carbon::create($this->year, $monthIndex + 1, 1)->endOfMonth();
                break;
            case 'day_5':
                $date = Carbon::create($this->year, $monthIndex + 1, 5);
                break;
            case 'day_10':
                $date = Carbon::create($this->year, $monthIndex + 1, 10);
                break;
            case 'no_deadline':
                $date = null;
                break;
        }

        FeeMonthDeadline::updateOrCreate(
            [
                'month' => $month,
                'year' => $this->year,
            ],
            [
                'deadline' => $date,
                'allow_late_payment' => true,
            ]
        );

        $this->loadDeadlines();
        session()->flash('message', 'Prazo rápido aplicado!');
    }

    public function render()
    {
        return view('livewire.admin.fee.fee-month-deadline-manager', [
            'months' => $this->months,
        ])->layout('layouts.app');
    }
}
