<?php

namespace App\Http\Livewire\Admin\Noticeboard;

use App\Models\Noticeboard;

use App\Models\User;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithPagination;
use Note\Note;
use Auth;

class Notice extends Component
{

    use WithPagination, FindGuard, LivewireAlert;

    public $search, $slug;
    public $selectedType = '';
    public $selectedClass = '';
    public $selectedTurma = '';
    public $turmas = [];

    // Bulk delete properties
    public $showBulkDeleteModal = false;
    public $bulkDeleteType = '';
    public $bulkDeleteFromDate = '';
    public $bulkDeleteToDate = '';
    public $bulkDeleteSelectedClass = '';
    public $bulkDeleteSelectedTurma = '';

    protected $queryString = ['search', 'selectedType', 'selectedClass', 'selectedTurma'];

    protected string $paginationTheme = 'bootstrap';

    public function exportPDF()
    {
        return redirect()->route('admin.comunicados.export.pdf', [
            'type' => $this->selectedType,
            'class' => $this->selectedClass,
            'turma' => $this->selectedTurma,
            'search' => $this->search
        ]);
    }

    public function exportExcel()
    {
        return redirect()->route('admin.comunicados.export.excel', [
            'type' => $this->selectedType,
            'class' => $this->selectedClass,
            'turma' => $this->selectedTurma,
            'search' => $this->search
        ]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedSelectedType()
    {
        $this->resetPage();
    }

    public function updatedSelectedClass($value)
    {
        $this->selectedTurma = '';
        $this->resetPage();

        if ($value) {
            $this->turmas = \App\Models\Classroom::where('class', $value)->get();
        } else {
            $this->turmas = [];
        }
    }

    public function updatedSelectedTurma()
    {
        $this->resetPage();
    }

    public bool $readyToLoad = false;

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function submit($id)
    {
        // Apenas armazena o slug, não o modelo inteiro (evita erro de hydration)
        $notice = Noticeboard::query()->firstWhere('slug', $id);

        if (!$notice) {
            $this->alert('error', 'Comunicado não encontrado.');
            return;
        }

        $this->slug = $notice->slug;

        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => "confirmed",
            'onDismissed' => 'cancelled',
        ]);
    }

    public function confirmed()
    {
        if (!$this->slug) {
            $this->alert('error', 'Erro: Comunicado não identificado.');
            return;
        }

        Noticeboard::where('slug', $this->slug)->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Comunicado', 'Deletado com sucesso');
        $this->emit('noteAdded');
        $this->alert('success', 'Comunicado deletado com sucesso!');

        // Limpar slug após deleção
        $this->slug = null;
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function openBulkDeleteModal($type)
    {
        $this->bulkDeleteType = $type;
        $this->showBulkDeleteModal = true;
    }

    public function closeBulkDeleteModal()
    {
        $this->reset(['showBulkDeleteModal', 'bulkDeleteType', 'bulkDeleteFromDate', 'bulkDeleteToDate', 'bulkDeleteSelectedClass', 'bulkDeleteSelectedTurma']);
    }

    public function confirmBulkDelete()
    {
        switch ($this->bulkDeleteType) {
            case 'all':
                $this->deleteAll();
                break;
            case 'byTurma':
                $this->deleteByTurma();
                break;
            case 'byClasse':
                $this->deleteByClasse();
                break;
            case 'byDateRange':
                $this->deleteByDateRange();
                break;
        }

        $this->closeBulkDeleteModal();
    }

    public function deleteAll()
    {
        $count = Noticeboard::count();
        Noticeboard::query()->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Comunicado', "Todos os comunicados deletados ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Todos os comunicados foram deletados com sucesso ({$count} itens)!");
    }

    public function deleteByTurma()
    {
        if (!$this->bulkDeleteSelectedTurma) {
            $this->alert('error', 'Por favor selecione uma turma!');
            return;
        }

        $count = Noticeboard::where('class_id', $this->bulkDeleteSelectedTurma)->count();
        Noticeboard::where('class_id', $this->bulkDeleteSelectedTurma)->delete();

        $turma = \App\Models\Classroom::find($this->bulkDeleteSelectedTurma);
        $turmaName = $turma ? $turma->class . ' Classe - ' . $turma->name : 'Turma';

        Note::createSystemNotification(\App\Models\Admin::class, 'Comunicado', "Comunicados da {$turmaName} deletados ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Comunicados da {$turmaName} deletados com sucesso ({$count} itens)!");
    }

    public function deleteByClasse()
    {
        if (!$this->bulkDeleteSelectedClass) {
            $this->alert('error', 'Por favor selecione uma classe!');
            return;
        }

        $classroomIds = \App\Models\Classroom::where('class', $this->bulkDeleteSelectedClass)->pluck('id');
        $count = Noticeboard::whereIn('class_id', $classroomIds)->count();
        Noticeboard::whereIn('class_id', $classroomIds)->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Comunicado', "Comunicados da {$this->bulkDeleteSelectedClass}ª Classe deletados ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Comunicados da {$this->bulkDeleteSelectedClass}ª Classe deletados com sucesso ({$count} itens)!");
    }

    public function deleteByDateRange()
    {
        if (!$this->bulkDeleteFromDate || !$this->bulkDeleteToDate) {
            $this->alert('error', 'Por favor selecione as datas de início e fim!');
            return;
        }

        $count = Noticeboard::whereBetween('created_at', [
            $this->bulkDeleteFromDate . ' 00:00:00',
            $this->bulkDeleteToDate . ' 23:59:59'
        ])->count();

        Noticeboard::whereBetween('created_at', [
            $this->bulkDeleteFromDate . ' 00:00:00',
            $this->bulkDeleteToDate . ' 23:59:59'
        ])->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Comunicado', "Comunicados entre {$this->bulkDeleteFromDate} e {$this->bulkDeleteToDate} deletados ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Comunicados entre {$this->bulkDeleteFromDate} e {$this->bulkDeleteToDate} deletados com sucesso ({$count} itens)!");
    }

    public function render()
    {
        $user = Auth::user();

        // Buscar todas as classes para o filtro
        $classes = \App\Models\Classroom::select('class')->distinct()->orderByRaw('CAST(class AS UNSIGNED)')->get();

        if(isset($user->role) && ($user->role == 0 || $user->role == 3)){
            // Admin ou Academic Department - vê todos os comunicados
            return view('livewire.admin.noticeboard.notice', [
                    'models' => $this->readyToLoad
                        ? Noticeboard::query()
                            ->when($this->search, function ($query) {
                                $query->where('title', 'like', '%' . $this->search . '%');
                            })
                            // Filtro por Tipo
                            ->when($this->selectedType, function ($query) {
                                $query->where('type', $this->selectedType);
                            })
                            // Filtro por Classe
                            ->when($this->selectedClass, function ($query) {
                                $classroomIds = \App\Models\Classroom::where('class', $this->selectedClass)->pluck('id');
                                $query->whereIn('class_id', $classroomIds);
                            })
                            // Filtro por Turma específica
                            ->when($this->selectedTurma, function ($query) {
                                $query->where('class_id', $this->selectedTurma);
                            })
                            ->orderByDesc('created_at')
                            ->paginate(10)
                        : [],
                    'classes' => $classes
                ]
            );
        }elseif(!isset($user->role)){
            // Estudante - vê apenas comunicados para todos, sua classe, ou específicos para ele
            $userr = User::where("id", $user->id)->first();

            return view('livewire.admin.noticeboard.notice', [
                    'models' => $this->readyToLoad
                        ? Noticeboard::query()
                            ->where(function ($query) use ($userr) {
                                $query->where('type', 'all')
                                      ->orWhere('class_id', $userr->classroom_id)
                                      ->orWhere('student_id', $userr->id);
                            })
                            ->when($this->search, function ($query) {
                                $query->where('title', 'like', '%' . $this->search . '%');
                            })
                            // Filtro por Tipo
                            ->when($this->selectedType, function ($query) {
                                $query->where('type', $this->selectedType);
                            })
                            // Filtro por Classe
                            ->when($this->selectedClass, function ($query) {
                                $classroomIds = \App\Models\Classroom::where('class', $this->selectedClass)->pluck('id');
                                $query->whereIn('class_id', $classroomIds);
                            })
                            // Filtro por Turma específica
                            ->when($this->selectedTurma, function ($query) {
                                $query->where('class_id', $this->selectedTurma);
                            })
                            ->orderByDesc('created_at')
                            ->paginate(10)
                        : [],
                    'student' => true,
                    'classes' => $classes
                ]
            );
        }

    }
}



