<?php

namespace App\Http\Livewire\Admin\POS;

use App\Models\POS\Category;
use Illuminate\Support\Str;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;
use Livewire\WithPagination;

class ManageCategories extends Component
{
    use LivewireAlert, WithPagination;

    public $search = '';

    // Form fields
    public $categoryId;
    public $name, $description, $icon, $color = '#4f46e5';
    public $is_active = true;

    public $showModal = false;
    public $editMode = false;

    protected $listeners = [
        'confirmedDelete',
        'cancelledDelete'
    ];

    public function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'icon' => 'nullable|string|max:50',
            'color' => 'required|string|max:7',
            'is_active' => 'boolean',
        ];
    }

    public function openModal()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function edit($id)
    {
        $category = Category::findOrFail($id);

        $this->categoryId = $category->id;
        $this->name = $category->name;
        $this->description = $category->description;
        $this->icon = $category->icon;
        $this->color = $category->color;
        $this->is_active = $category->is_active;

        $this->editMode = true;
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        try {
            $data = [
                'name' => $this->name,
                'slug' => Str::slug($this->name),
                'description' => $this->description,
                'icon' => $this->icon,
                'color' => $this->color,
                'is_active' => $this->is_active,
            ];

            if ($this->editMode) {
                $category = Category::findOrFail($this->categoryId);
                $category->update($data);
                $message = 'Categoria atualizada com sucesso';
            } else {
                Category::create($data);
                $message = 'Categoria criada com sucesso';
            }

            $this->alert('success', $message);
            $this->closeModal();
            $this->resetPage();
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao salvar categoria: ' . $e->getMessage());
        }
    }

    public function confirmDelete($id)
    {
        $this->categoryId = $id;
        $this->confirm('Tem certeza que deseja excluir esta categoria?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, excluir',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedDelete',
            'onDismissed' => 'cancelledDelete'
        ]);
    }

    public function confirmedDelete()
    {
        try {
            $category = Category::findOrFail($this->categoryId);

            // Check if has products
            if ($category->products()->count() > 0) {
                $this->alert('error', 'Não é possível excluir. Esta categoria possui produtos associados.');
                return;
            }

            $category->delete();
            $this->alert('success', 'Categoria excluída com sucesso');
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao excluir categoria: ' . $e->getMessage());
        }
    }

    public function cancelledDelete()
    {
        $this->alert('info', 'Exclusão cancelada');
    }

    public function resetForm()
    {
        $this->reset(['categoryId', 'name', 'description', 'icon']);
        $this->color = '#4f46e5';
        $this->is_active = true;
        $this->editMode = false;
    }

    public function render()
    {
        $query = Category::query()
            ->when($this->search, function($q) {
                $q->where('name', 'like', '%' . $this->search . '%');
            })
            ->withCount('products')
            ->orderBy('created_at', 'desc');

        return view('livewire.admin.pos.manage-categories', [
            'categories' => $query->paginate(15),
        ]);
    }
}
