<?php

namespace App\Http\Livewire\Admin\POS;

use App\Models\POS\Category;
use App\Models\POS\Product;
use App\Models\POS\Sale;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Livewire\Component;

class POSDashboard extends Component
{
    public $period = '7days';
    public $customDateFrom;
    public $customDateTo;

    protected $queryString = ['period', 'customDateFrom', 'customDateTo'];

    public function mount()
    {
        $this->customDateFrom = now()->subDays(7)->format('Y-m-d');
        $this->customDateTo = now()->format('Y-m-d');
    }

    private function getDateRange()
    {
        switch ($this->period) {
            case '7days':
                return [now()->subDays(6)->startOfDay(), now()];
            case '30days':
                return [now()->subDays(29)->startOfDay(), now()];
            case '1year':
                return [now()->subYear()->startOfDay(), now()];
            case 'custom':
                if ($this->customDateFrom && $this->customDateTo) {
                    return [
                        Carbon::parse($this->customDateFrom)->startOfDay(),
                        Carbon::parse($this->customDateTo)->endOfDay()
                    ];
                }
                return [now()->subDays(6)->startOfDay(), now()];
            default:
                return [now()->subDays(6)->startOfDay(), now()];
        }
    }

    public function getSalesStats()
    {
        $dateRange = $this->getDateRange();

        $sales = Sale::completed()
            ->whereBetween('sale_date', $dateRange)
            ->get();

        $todaySales = Sale::completed()
            ->whereDate('sale_date', today())
            ->sum('total');

        $dailySales = Sale::completed()
            ->whereBetween('sale_date', $dateRange)
            ->select(
                DB::raw('DATE(sale_date) as date'),
                DB::raw('SUM(total) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        return [
            'total_revenue' => $sales->sum('total'),
            'total_sales' => $sales->count(),
            'today_revenue' => $todaySales,
            'average_sale' => $sales->count() > 0 ? $sales->sum('total') / $sales->count() : 0,
            'daily_sales' => $dailySales,
        ];
    }

    public function getProductStats()
    {
        return [
            'total_products' => Product::active()->count(),
            'low_stock_products' => Product::active()->lowStock()->count(),
            'out_of_stock_products' => Product::active()->outOfStock()->count(),
            'total_categories' => Category::active()->count(),
        ];
    }

    public function getTopProducts()
    {
        $dateRange = $this->getDateRange();

        return DB::table('pos_sale_items')
            ->join('pos_sales', 'pos_sale_items.sale_id', '=', 'pos_sales.id')
            ->join('pos_products', 'pos_sale_items.product_id', '=', 'pos_products.id')
            ->whereBetween('pos_sales.sale_date', $dateRange)
            ->where('pos_sales.status', 'completed')
            ->select(
                'pos_products.name',
                'pos_products.sku',
                DB::raw('SUM(pos_sale_items.quantity) as total_quantity'),
                DB::raw('SUM(pos_sale_items.total) as total_revenue')
            )
            ->groupBy('pos_products.id', 'pos_products.name', 'pos_products.sku')
            ->orderByDesc('total_quantity')
            ->limit(10)
            ->get();
    }

    public function render()
    {
        return view('livewire.admin.pos.pos-dashboard', [
            'sales_stats' => $this->getSalesStats(),
            'product_stats' => $this->getProductStats(),
            'top_products' => $this->getTopProducts(),
        ]);
    }
}
