<?php

namespace App\Http\Livewire\Admin\POS;

use App\Models\POS\Category;
use App\Models\POS\Product;
use App\Models\POS\Sale;
use App\Models\POS\SaleItem;
use App\Models\POS\Setting;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class POSInterface extends Component
{
    use LivewireAlert;

    public $categories = [];
    public $products = [];
    public $selectedCategory = null;
    public $search = '';

    // Cart
    public $cart = [];
    public $subtotal = 0;
    public $discount = 0;
    public $tax = 0;
    public $taxRate = 0;
    public $taxEnabled = false;
    public $total = 0;

    // Customer/Student
    public $student_id = null;
    public $student = null;

    // Payment
    public $payment_method = 'cash';
    public $notes = '';

    protected $listeners = [
        'confirmedSale',
        'cancelledSale'
    ];

    public function mount()
    {
        $this->loadCategories();
        $this->loadProducts();
        $this->loadSettings();
    }

    public function loadSettings()
    {
        $this->taxEnabled = Setting::get('tax_enabled', false);
        $this->taxRate = Setting::get('tax_rate', 16);
    }

    public function loadCategories()
    {
        $this->categories = Category::active()->withCount('products')->get();
    }

    public function loadProducts()
    {
        $query = Product::with('category')
            ->active()
            ->inStock();

        if ($this->selectedCategory) {
            $query->where('category_id', $this->selectedCategory);
        }

        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('sku', 'like', '%' . $this->search . '%');
            });
        }

        $this->products = $query->get();
    }

    public function selectCategory($categoryId)
    {
        $this->selectedCategory = $categoryId == $this->selectedCategory ? null : $categoryId;
        $this->loadProducts();
    }

    public function updated($propertyName)
    {
        if ($propertyName == 'search') {
            $this->loadProducts();
        }

        if ($propertyName == 'student_id') {
            $this->loadStudent();
        }

        if (in_array($propertyName, ['discount', 'tax'])) {
            $this->calculateTotals();
        }
    }

    public function loadStudent()
    {
        if ($this->student_id) {
            $this->student = User::where('student_id', $this->student_id)->first();

            if (!$this->student) {
                $this->alert('warning', 'Estudante não encontrado');
                $this->student_id = null;
            }
        } else {
            $this->student = null;
        }
    }

    public function addToCart($productId, $variant = null)
    {
        $product = Product::find($productId);

        if (!$product) {
            $this->alert('error', 'Produto não encontrado');
            return;
        }

        if ($product->quantity <= 0) {
            $this->alert('error', 'Produto sem stock');
            return;
        }

        $cartKey = $productId . ($variant ? '-' . $variant : '');

        if (isset($this->cart[$cartKey])) {
            // Check stock
            if ($this->cart[$cartKey]['quantity'] >= $product->quantity) {
                $this->alert('warning', 'Stock insuficiente');
                return;
            }
            $this->cart[$cartKey]['quantity']++;
        } else {
            $this->cart[$cartKey] = [
                'product_id' => $product->id,
                'name' => $product->name,
                'sku' => $product->sku,
                'variant' => $variant,
                'quantity' => 1,
                'unit_price' => $product->effective_price,
                'image' => $product->image,
                'max_quantity' => $product->quantity,
            ];
        }

        $this->calculateTotals();
        $this->alert('success', 'Produto adicionado ao carrinho');
    }

    public function updateQuantity($cartKey, $quantity)
    {
        if (!isset($this->cart[$cartKey])) {
            return;
        }

        $quantity = max(1, $quantity);

        if ($quantity > $this->cart[$cartKey]['max_quantity']) {
            $this->alert('warning', 'Stock insuficiente');
            return;
        }

        $this->cart[$cartKey]['quantity'] = $quantity;
        $this->calculateTotals();
    }

    public function removeFromCart($cartKey)
    {
        if (isset($this->cart[$cartKey])) {
            unset($this->cart[$cartKey]);
            $this->calculateTotals();
            $this->alert('info', 'Produto removido do carrinho');
        }
    }

    public function calculateTotals()
    {
        $this->subtotal = collect($this->cart)->sum(function ($item) {
            return $item['quantity'] * $item['unit_price'];
        });

        // Calculate tax automatically if enabled
        if ($this->taxEnabled) {
            $this->tax = ($this->subtotal - $this->discount) * ($this->taxRate / 100);
        }

        $this->total = $this->subtotal - $this->discount + $this->tax;
    }

    public function processSale()
    {
        if (empty($this->cart)) {
            $this->alert('warning', 'Carrinho vazio');
            return;
        }

        $this->confirm('Confirmar venda?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, confirmar',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedSale',
            'onDismissed' => 'cancelledSale'
        ]);
    }

    public function confirmedSale()
    {
        try {
            DB::beginTransaction();

            // Create sale
            $sale = Sale::create([
                'user_id' => $this->student->id ?? null,
                'cashier_id' => auth('admin')->id(),
                'subtotal' => $this->subtotal,
                'discount' => $this->discount,
                'tax' => $this->tax,
                'total' => $this->total,
                'payment_method' => $this->payment_method,
                'status' => 'completed',
                'notes' => $this->notes,
                'sale_date' => now(),
            ]);

            // Create sale items and update stock
            foreach ($this->cart as $item) {
                SaleItem::create([
                    'sale_id' => $sale->id,
                    'product_id' => $item['product_id'],
                    'product_name' => $item['name'],
                    'product_sku' => $item['sku'],
                    'variant' => $item['variant'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'discount' => 0,
                    'subtotal' => $item['quantity'] * $item['unit_price'],
                    'total' => $item['quantity'] * $item['unit_price'],
                ]);

                // Update product stock
                $product = Product::find($item['product_id']);
                $product->adjustStock(
                    -$item['quantity'],
                    'sale',
                    auth('admin')->id(),
                    "Venda #{$sale->sale_number}"
                );
            }

            DB::commit();

            $this->alert('success', 'Venda realizada com sucesso! Número: ' . $sale->sale_number);

            // Reset
            $this->resetSale();

            // Redirect to receipt or continue
            // return redirect()->route('admin.pos.receipt', $sale->id);

        } catch (\Exception $e) {
            DB::rollBack();
            $this->alert('error', 'Erro ao processar venda: ' . $e->getMessage());
        }
    }

    public function cancelledSale()
    {
        $this->alert('info', 'Venda cancelada');
    }

    public function resetSale()
    {
        $this->cart = [];
        $this->subtotal = 0;
        $this->discount = 0;
        $this->tax = 0;
        $this->total = 0;
        $this->student_id = null;
        $this->student = null;
        $this->payment_method = 'cash';
        $this->notes = '';
        $this->loadProducts();
    }

    public function render()
    {
        return view('livewire.admin.pos.pos-interface', [
            'tax_enabled' => $this->taxEnabled,
            'tax_rate' => $this->taxRate,
        ]);
    }
}
