<?php

namespace App\Http\Livewire\Admin\Reports;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Exam;
use App\Models\User;
use App\Models\Payment;
use App\Models\Project;
use App\Models\Subject;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Note\Note;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\ExportExams;
use App\Models\TeacherSubject;

class GenerateReports extends Component
{
    use FindGuard, LivewireAlert;

    public $trimester_id;
    public $class_id;
    public $user_id, $class, $year;
    public $subject_id;
    public $report = [];
    public $reports;
    public bool $readyToLoad = false;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    public function mount()
    {
        // Set default year to current year
        $this->year = date("Y");
    }

    public function loadReport()
    {
        $this->readyToLoad = true;
    }

    protected array $rules = [
        'user_id' => ['string', 'nullable'],
        'trimester_id' => ['numeric', 'required'],
        'class_id' => ['string', 'required'],
        'class' => ['string', 'required'],
        'subject_id' => ['string', 'required'],
        'year' => ['required']
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    /**
     * Reset fields when year is changed
     */
    public function updatedYear()
    {
        // Reset class, classroom and subject when year changes
        $this->class = null;
        $this->class_id = null;
        $this->subject_id = null;
    }

    /**
     * Reset fields when class is changed
     */
    public function updatedClass()
    {
        // Reset classroom and subject when class level changes
        $this->class_id = null;
        $this->subject_id = null;
    }

    /**
     * generate report
     * @throws Exception
     */
    public function generateReport()
    {
        $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed1()
    {
        Note::createSystemNotification(Admin::class, 'Relatório trimestral de cada disciplina', 'Relatório gerado com sucesso ');
        $this->emit('noteAdded');
        $this->alert('success', 'Relatório gerado com sucesso ');
        $condition = ['class_id' => $this->class_id, 'trimester_id'=>$this->trimester_id, 'subject_id'=> $this->subject_id, 'year'=>$this->year];
        return Excel::download(new ExportExams($condition), Subject::query()->where('id',  $this->subject_id)->first()->name.'.xlsx');
        //get the id

    }


    public function confirmed()
    {
        // Criar uma notificação
        Note::createSystemNotification(Admin::class, 'Relatório trimestral de cada disciplina', 'Relatório gerado com sucesso');
        $this->emit('noteAdded');
        $this->alert('success', 'Relatório gerado com sucesso');

        // Obter informações da Classe, Turma, Disciplina, Trimestre e Ano
        $classroom = Classroom::query()->where('id', $this->class_id)->first();
        $class_name = $classroom ? $classroom->name : 'Turma'; // Nome da turma (ex: "Turma B")
        $classe = $this->class ?? 'Classe'; // Nome da classe (ex: "6ª Classe")
        $disciplina = Subject::query()->where('id', $this->subject_id)->first()->name ?? 'Disciplina'; // Nome da disciplina
        $ano = $this->year ?? date('Y'); // Ano (ex: "2024")

        // Obter o nome do trimestre
        $trimestres = [1 => '1 Trimestre', 2 => '2 Trimestre', 3 => '3 Trimestre'];
        $trimestre = $trimestres[$this->trimester_id] ?? 'Trimestre'; // Nome do trimestre

        // Criar condição para exportar os dados
        $condition = [
            'class_id' => $this->class_id,
            'trimester_id' => $this->trimester_id,
            'subject_id' => $this->subject_id,
            'year' => $this->year,
        ];

        // Nome formatado do arquivo
        $fileName = "{$classe} - {$class_name} - {$disciplina} - {$trimestre} ({$ano}).xlsx";

        // Retornar o download do Excel com o nome formatado
        return Excel::download(new ExportExams($condition), $fileName);
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        $this->user = $this->findGuardType()->user();
        $ccclass = ($this->class) ? $this->class : '1';

        // Default to empty array
        $availableClassLevels = [];
        $isTeacher = ($this->user->role == 2);

        if ($isTeacher && $this->year) {
            // Get classes where the teacher has exams for the selected year
//            $examClassLevels = DB::table('exams')
//                ->join('teacher_subjects', function ($join) {
//                    $join->on('exams.subject_id', '=', 'teacher_subjects.subject_id');
//                })
//                ->join('classrooms', 'exams.classroom_id', '=', 'classrooms.id')
//                ->where('teacher_subjects.teacher_id', '=', $this->user->id)
//                ->where('exams.year', '=', $this->year)
//                ->select('classrooms.class')
//                ->distinct()
//                ->pluck('class')
//                ->toArray();

            // Get classes directly assigned to the teacher in TeacherSubject
            $directlyAssignedLevels = TeacherSubject::where('teacher_id', $this->user->id)
                ->whereNotNull('class')
                ->distinct()
                ->pluck('class')
                ->toArray();

            // Combine both arrays and remove duplicates
            $availableClassLevels = array_values($directlyAssignedLevels);
        }

        // Get available classrooms based on the selected class level and year
        if ($isTeacher) {
            // Check if class exists in available class levels before querying
            if (!in_array($ccclass, $availableClassLevels) && !empty($availableClassLevels)) {
                $classrooms = collect(); // Return empty collection if teacher doesn't teach this class
            } else {
                $classrooms = Classroom::where('class', $ccclass)
                    ->where(function($query) {
                        // Classrooms directly assigned to this teacher
                        $query->whereIn('id', function($subquery) {
                            $subquery->select('classroom_id')
                                    ->from('teacher_subjects')
                                    ->where('teacher_id', $this->user->id)
                                    ->whereNotNull('classroom_id');
                        });

                        // OR Classrooms where teacher has exams for this year
                        $query->orWhereIn('id', function($subquery) {
                            $subquery->select('exams.classroom_id')
                                    ->from('exams')
                                    ->join('teacher_subjects', 'exams.subject_id', '=', 'teacher_subjects.subject_id')
                                    ->where('teacher_subjects.teacher_id', '=', $this->user->id)
                                    ->where('exams.year', '=', $this->year)
                                    ->distinct();
                        });
                    })
                    ->orderBy('name')
                    ->get();
            }
        } else {
            // For admins, show all classrooms for the selected class level
            $classrooms = Classroom::where('class', $ccclass)
                ->orderBy('name')
                ->get();
        }

        // Get subjects based on the selected class level and year
        if ($isTeacher) {
            // Check if class exists in available class levels before querying
            if (!in_array($ccclass, $availableClassLevels) && !empty($availableClassLevels)) {
                $subjects = collect(); // Return empty collection if teacher doesn't teach this class
            } else {
                // Get subjects assigned to this teacher generally
                $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                    ->select('subjects.id', 'subjects.name', 'subjects.slug')
                    ->where('teacher_subjects.teacher_id', $this->user->id)
                    ->where('teacher_subjects.class', $ccclass)
                    ->orderBy('subjects.name')
                    ->distinct()
                    ->get();

                // If additional filtering by exam year is needed:
                if (!empty($this->year)) {
                    $subjectsWithExams = Subject::join('exams', 'subjects.id', '=', 'exams.subject_id')
                        ->join('classrooms', 'exams.classroom_id', '=', 'classrooms.id')
                        ->select('subjects.id', 'subjects.name', 'subjects.slug')
                        ->where('exams.year', $this->year)
                        ->where('classrooms.class', $ccclass)
                        ->whereIn('exams.classroom_id', function($query) {
                            $query->select('classroom_id')
                                ->from('teacher_subjects')
                                ->where('teacher_id', $this->user->id)
                                ->whereNotNull('classroom_id');
                        })
                        ->orderBy('subjects.name')
                        ->distinct()
                        ->get();

                    // Merge the collections and remove duplicates
                    $subjects = $subjects->merge($subjectsWithExams)->unique('id');
                }
            }
        } else {
            // For admins, show all subjects for the selected class level
            $subjects = Subject::join('teacher_subjects', 'subjects.id', '=', 'teacher_subjects.subject_id')
                ->select('subjects.id', 'subjects.name', 'subjects.slug')
                ->where('teacher_subjects.class', $ccclass)
                ->orderBy('subjects.name')
                ->distinct()
                ->get();
        }

        return view('livewire.admin.reports.generate-reports', [
            'subjects' => $subjects,
            'classes' => $classrooms,
            'teacher' => TeacherSubject::where("teacher_id", $this->user->id)->get(),
            'class_id' => $ccclass,
            'availableClassLevels' => $availableClassLevels,
            'isTeacher' => $isTeacher
        ]);
    }
}
