<?php

namespace App\Http\Livewire\Admin\Secretary;

use App\Models\Classroom;
use App\Models\PreEnrollment;
use App\Services\GoogleVisionOCRService;
use Exception;
use Illuminate\Support\Facades\Storage;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads;

class EditPreEnrollment extends Component
{
    use FindGuard, LivewireAlert, WithFileUploads;

    public $preEnrollmentId;

    // Dados Pessoais
    public $candidate_name, $candidate_sex, $candidate_dob;
    public $candidate_phone, $candidate_email;
    public $marital_status, $nationality, $nuit;
    public $chronic_disease = false, $chronic_disease_details;

    // Documentos
    public $document_type, $document_id;
    public $document_issue_place, $document_expiry_date;
    public $entry_type;
    public $previous_institution, $previous_course;

    // Naturalidade & Morada
    public $birth_place, $birth_district, $birth_province;
    public $address_province, $address_district, $address_bairro;
    public $address_street, $address_house_number;
    public $candidate_address, $candidate_whatsapp;

    // Filiação
    public $father_name, $father_phone;
    public $mother_name, $mother_phone;

    // Encarregado
    public $guardian_name, $guardian_relationship;
    public $guardian_phone, $guardian_email, $guardian_address;
    public $guardian_document_id, $guardian_document_issue_place;
    public $guardian_document_issue_date;
    public $guardian_bairro, $guardian_quarteirao, $guardian_workplace;

    // Emergência
    public $emergency_contact_name, $emergency_contact_phone;

    // Curso
    public $desired_class, $desired_classroom_id, $desired_shift;
    public $academic_year, $observation;

    // OCR / BI
    public $bi_document;
    public $processing_bi = false;
    public $bi_processed = false;
    public $extracted_photo_path = null;
    public $avatar;

    protected $listeners = ['confirmed', 'cancelled'];

    public function mount($id)
    {
        $pre = PreEnrollment::findOrFail($id);
        $this->preEnrollmentId = $pre->id;

        // Dados Pessoais
        $this->candidate_name = $pre->candidate_name;
        $this->candidate_sex = $pre->candidate_sex;
        $this->candidate_dob = $pre->candidate_dob ? $pre->candidate_dob->format('Y-m-d') : '';
        $this->candidate_phone = $pre->candidate_phone;
        $this->candidate_email = $pre->candidate_email;
        $this->marital_status = $pre->marital_status;
        $this->nationality = $pre->nationality;
        $this->nuit = $pre->nuit;
        $this->chronic_disease = $pre->chronic_disease ?? false;
        $this->chronic_disease_details = $pre->chronic_disease_details;

        // Documentos
        $this->document_type = $pre->document_type;
        $this->document_id = $pre->document_id;
        $this->document_issue_place = $pre->document_issue_place;
        $this->document_expiry_date = $pre->document_expiry_date ? $pre->document_expiry_date->format('Y-m-d') : '';
        $this->entry_type = $pre->entry_type;
        $this->previous_institution = $pre->previous_institution;
        $this->previous_course = $pre->previous_course;

        // Naturalidade & Morada
        $this->birth_place = $pre->birth_place;
        $this->birth_district = $pre->birth_district;
        $this->birth_province = $pre->birth_province;
        $this->address_province = $pre->address_province;
        $this->address_district = $pre->address_district;
        $this->address_bairro = $pre->address_bairro;
        $this->address_street = $pre->address_street;
        $this->address_house_number = $pre->address_house_number;
        $this->candidate_address = $pre->candidate_address;
        $this->candidate_whatsapp = $pre->candidate_whatsapp;

        // Filiação
        $this->father_name = $pre->father_name;
        $this->father_phone = $pre->father_phone;
        $this->mother_name = $pre->mother_name;
        $this->mother_phone = $pre->mother_phone;

        // Encarregado
        $this->guardian_name = $pre->guardian_name;
        $this->guardian_relationship = $pre->guardian_relationship;
        $this->guardian_phone = $pre->guardian_phone;
        $this->guardian_email = $pre->guardian_email;
        $this->guardian_address = $pre->guardian_address;
        $this->guardian_document_id = $pre->guardian_document_id;
        $this->guardian_document_issue_place = $pre->guardian_document_issue_place;
        $this->guardian_document_issue_date = $pre->guardian_document_issue_date ? $pre->guardian_document_issue_date->format('Y-m-d') : '';
        $this->guardian_bairro = $pre->guardian_bairro;
        $this->guardian_quarteirao = $pre->guardian_quarteirao;
        $this->guardian_workplace = $pre->guardian_workplace;

        // Emergência
        $this->emergency_contact_name = $pre->emergency_contact_name;
        $this->emergency_contact_phone = $pre->emergency_contact_phone;

        // Curso
        $this->desired_class = $pre->desired_class;
        $this->desired_classroom_id = $pre->desired_classroom_id;
        $this->desired_shift = $pre->desired_shift;
        $this->academic_year = $pre->academic_year;
        $this->observation = $pre->observation;
    }

    protected function rules()
    {
        return [
            'candidate_name' => ['required', 'string', 'max:255'],
            'candidate_sex' => ['required', 'string', 'in:Masculino,Feminino'],
            'candidate_dob' => ['required', 'date'],
            'candidate_phone' => ['required', 'string', 'max:20'],
            'candidate_email' => ['nullable', 'email', 'max:255'],
            'marital_status' => ['nullable', 'string', 'in:Solteiro,Casado'],
            'nationality' => ['nullable', 'string', 'max:255'],
            'nuit' => ['nullable', 'string', 'max:50'],
            'chronic_disease' => ['nullable', 'boolean'],
            'chronic_disease_details' => ['nullable', 'string', 'max:500'],
            'document_type' => ['nullable', 'string', 'in:BI,Cedula,Passaporte'],
            'document_id' => ['nullable', 'string', 'max:255'],
            'document_issue_place' => ['nullable', 'string', 'max:255'],
            'document_expiry_date' => ['nullable', 'date'],
            'entry_type' => ['required', 'string', 'in:Normal,Transferencia,Reingresso'],
            'previous_institution' => ['nullable', 'string', 'max:255'],
            'previous_course' => ['nullable', 'string', 'max:255'],
            'birth_place' => ['nullable', 'string', 'max:255'],
            'birth_district' => ['nullable', 'string', 'max:255'],
            'birth_province' => ['nullable', 'string'],
            'address_province' => ['nullable', 'string'],
            'address_district' => ['nullable', 'string', 'max:255'],
            'address_bairro' => ['nullable', 'string', 'max:255'],
            'address_street' => ['nullable', 'string', 'max:255'],
            'address_house_number' => ['nullable', 'string', 'max:50'],
            'candidate_address' => ['nullable', 'string', 'max:500'],
            'candidate_whatsapp' => ['nullable', 'string', 'max:20'],
            'father_name' => ['nullable', 'string', 'max:255'],
            'father_phone' => ['nullable', 'string', 'max:20'],
            'mother_name' => ['nullable', 'string', 'max:255'],
            'mother_phone' => ['nullable', 'string', 'max:20'],
            'guardian_name' => ['nullable', 'string', 'max:255'],
            'guardian_relationship' => ['nullable', 'string', 'max:100'],
            'guardian_phone' => ['nullable', 'string', 'max:20'],
            'guardian_email' => ['nullable', 'email', 'max:255'],
            'guardian_address' => ['nullable', 'string', 'max:500'],
            'guardian_document_id' => ['nullable', 'string', 'max:255'],
            'guardian_document_issue_place' => ['nullable', 'string', 'max:255'],
            'guardian_document_issue_date' => ['nullable', 'date'],
            'guardian_bairro' => ['nullable', 'string', 'max:255'],
            'guardian_quarteirao' => ['nullable', 'string', 'max:255'],
            'guardian_workplace' => ['nullable', 'string', 'max:255'],
            'emergency_contact_name' => ['nullable', 'string', 'max:255'],
            'emergency_contact_phone' => ['nullable', 'string', 'max:20'],
            'desired_class' => ['required', 'string'],
            'desired_classroom_id' => ['nullable', 'string'],
            'desired_shift' => ['nullable', 'string', 'in:Diurno,Tarde'],
            'academic_year' => ['required', 'string'],
            'observation' => ['nullable', 'string', 'max:1000'],
        ];
    }

    public function updated($propertyName)
    {
        if ($propertyName === 'bi_document') {
            $this->processBiDocument();
        }
    }

    public function processBiDocument()
    {
        if (!$this->bi_document) {
            return;
        }

        $this->processing_bi = true;

        try {
            $tempPath = $this->bi_document->store('temp', 'public');
            $fullPath = storage_path('app/public/' . $tempPath);

            $ocrService = new GoogleVisionOCRService();
            $result = $ocrService->processDocument($fullPath);

            if ($result['success']) {
                $extractedData = $ocrService->extractBIData($result['text']);

                if (!empty($extractedData['full_name'])) $this->candidate_name = $extractedData['full_name'];
                if (!empty($extractedData['dob'])) $this->candidate_dob = $extractedData['dob'];
                if (!empty($extractedData['sex'])) {
                    $this->candidate_sex = $extractedData['sex'] === 'M' ? 'Masculino' :
                        ($extractedData['sex'] === 'F' ? 'Feminino' : $extractedData['sex']);
                }
                if (!empty($extractedData['father_name'])) $this->father_name = $extractedData['father_name'];
                if (!empty($extractedData['mother_name'])) $this->mother_name = $extractedData['mother_name'];
                if (!empty($extractedData['birth_place'])) $this->birth_place = $extractedData['birth_place'];
                if (!empty($extractedData['birth_district'])) $this->birth_district = $extractedData['birth_district'];
                if (!empty($extractedData['birth_province'])) $this->birth_province = $extractedData['birth_province'];
                if (!empty($extractedData['document_id'])) $this->document_id = $extractedData['document_id'];
                if (!empty($extractedData['document_type'])) $this->document_type = $extractedData['document_type'];

                // Extract face photo
                $facePath = $ocrService->extractFaceFromDocument($fullPath);
                if ($facePath && file_exists($facePath)) {
                    $photoContent = file_get_contents($facePath);
                    $photoStoragePath = 'extracted_photos/' . uniqid('photo_') . '.jpg';
                    Storage::disk('public')->put($photoStoragePath, $photoContent);
                    $this->extracted_photo_path = $photoStoragePath;
                    @unlink($facePath);
                }

                $this->bi_processed = true;
                $msg = $this->extracted_photo_path
                    ? 'Dados do BI extraídos com sucesso! Foto detectada.'
                    : 'Dados do BI extraídos com sucesso! Foto não detectada.';
                $this->alert('success', $msg);
            } else {
                $this->alert('error', 'Erro ao processar BI: ' . $result['error']);
            }

            Storage::disk('public')->delete($tempPath);
        } catch (Exception $e) {
            $this->alert('error', 'Erro ao processar BI: ' . $e->getMessage());
        }

        $this->processing_bi = false;
    }

    public function submit()
    {
        $this->validate();

        $this->confirm('Guardar alterações?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, guardar!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $pre = PreEnrollment::findOrFail($this->preEnrollmentId);

        $pre->update([
            'candidate_name' => $this->candidate_name,
            'candidate_sex' => $this->candidate_sex,
            'candidate_dob' => $this->candidate_dob,
            'candidate_phone' => $this->candidate_phone,
            'candidate_email' => $this->candidate_email,
            'candidate_origin' => $this->entry_type === 'Normal' ? 'Interno' : 'Externo',
            'marital_status' => $this->marital_status,
            'nationality' => $this->nationality,
            'nuit' => $this->nuit,
            'chronic_disease' => $this->chronic_disease,
            'chronic_disease_details' => $this->chronic_disease ? $this->chronic_disease_details : null,
            'document_type' => $this->document_type,
            'document_id' => $this->document_id,
            'document_issue_place' => $this->document_issue_place,
            'document_expiry_date' => $this->document_expiry_date ?: null,
            'entry_type' => $this->entry_type,
            'previous_institution' => $this->previous_institution,
            'previous_course' => $this->previous_course,
            'birth_place' => $this->birth_place,
            'birth_district' => $this->birth_district,
            'birth_province' => $this->birth_province,
            'address_province' => $this->address_province,
            'address_district' => $this->address_district,
            'address_bairro' => $this->address_bairro,
            'address_street' => $this->address_street,
            'address_house_number' => $this->address_house_number,
            'candidate_address' => $this->candidate_address,
            'candidate_whatsapp' => $this->candidate_whatsapp,
            'father_name' => $this->father_name,
            'father_phone' => $this->father_phone,
            'mother_name' => $this->mother_name,
            'mother_phone' => $this->mother_phone,
            'guardian_name' => $this->guardian_name,
            'guardian_relationship' => $this->guardian_relationship,
            'guardian_phone' => $this->guardian_phone,
            'guardian_email' => $this->guardian_email,
            'guardian_address' => $this->guardian_address,
            'guardian_document_id' => $this->guardian_document_id,
            'guardian_document_issue_place' => $this->guardian_document_issue_place,
            'guardian_document_issue_date' => $this->guardian_document_issue_date ?: null,
            'guardian_bairro' => $this->guardian_bairro,
            'guardian_quarteirao' => $this->guardian_quarteirao,
            'guardian_workplace' => $this->guardian_workplace,
            'emergency_contact_name' => $this->emergency_contact_name,
            'emergency_contact_phone' => $this->emergency_contact_phone,
            'desired_class' => $this->desired_class,
            'desired_classroom_id' => $this->desired_classroom_id,
            'desired_shift' => $this->desired_shift,
            'academic_year' => $this->academic_year,
            'observation' => $this->observation,
        ]);

        // Update avatar on resulting student if photo was extracted/uploaded
        if ($pre->resulting_user_id) {
            $student = \App\Models\User::find($pre->resulting_user_id);
            if ($student) {
                if ($this->avatar) {
                    $student->update(['avatar' => $this->avatar->store('avatars', 'public')]);
                } elseif ($this->extracted_photo_path) {
                    $photoContent = Storage::disk('public')->get($this->extracted_photo_path);
                    $avatarPath = 'avatars/' . basename($this->extracted_photo_path);
                    Storage::disk('public')->put($avatarPath, $photoContent);
                    $student->update(['avatar' => $avatarPath]);
                    Storage::disk('public')->delete($this->extracted_photo_path);
                }
            }
        }

        $this->alert('success', 'Dados actualizados com sucesso!');

        return redirect()->route('admin.secretary.show', $this->preEnrollmentId);
    }

    public function cancelled()
    {
        $this->alert('info', 'Operação cancelada.');
    }

    public function render()
    {
        $this->findGuardType()->user();

        $classrooms = Classroom::query()
            ->when($this->desired_class, fn($q) => $q->where('class', $this->desired_class))
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $availableClasses = Classroom::where('is_active', true)
            ->select('class')
            ->distinct()
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->pluck('class');

        return view('livewire.admin.secretary.edit-pre-enrollment', [
            'classrooms' => $classrooms,
            'availableClasses' => $availableClasses,
        ]);
    }
}
