<?php

namespace App\Http\Livewire\Admin\Sendsms;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\Guardian;
use App\Models\User;
use App\Models\SendSms;
use App\Services\SmsService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads;
use Note\Note;
use App\Http\Controllers\SystemController;
use Exception;

class AddSms extends Component
{
    use FindGuard, LivewireAlert, WithFileUploads;

    public $title, $description, $category_id, $photo, $pdf, $author,  $validatedData, $type, $class_id, $student_id, $student_type, $created_by;
    public $class_ids = [];
    public $guardian_id;
    public $useCustomNumbers = false;
    public $customNumbers;
    public $updateMode = false;
    public $user;
    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    /**
     * Get validation rules based on mode
     *
     * @return array
     */
    protected function rules()
    {
        if ($this->useCustomNumbers) {
            return [
                'customNumbers' => ['required', 'string', 'min:9'],
                'title' => ['required', 'string', 'max:500'],
            ];
        }

        $rules = [
            'title' => ['required', 'string', 'max:255'],
        ];

        // student_type only needed for student-based types
        if (in_array($this->type, ['all', 'class', 'student', null, ''])) {
            $rules['student_type'] = ['required'];
        }

        if ($this->type == 'guardian') {
            $rules['guardian_id'] = ['required'];
        }

        return $rules;
    }

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);

    }

    /**
     * Parse phone numbers from textarea input - INTELLIGENT PARSER
     * Accepts ANY format: with/without commas, spaces, dashes, dots, parentheses
     * Examples:
     *   - 843211234
     *   - 258843211234
     *   - +258 84 321 1234
     *   - (84) 321-1234
     *   - 84.321.1234
     *   - Mixed: "843211234, 258823456789 841234567"
     *
     * @param string $input
     * @return array
     */
    private function parsePhoneNumbers($input)
    {
        // First, extract all potential phone numbers using regex
        // Match sequences of digits with optional + at start and any separators in between
        preg_match_all('/[\+]?[\d][\d\s\-\.\(\)]+[\d]/', $input, $matches);

        $formatted = [];

        // Process each matched number
        foreach ($matches[0] as $number) {
            // Remove ALL non-digit characters except + at the start
            $cleaned = preg_replace('/[^0-9+]/', '', $number);

            // Skip if too short (less than 9 digits)
            if (strlen(str_replace('+', '', $cleaned)) < 9) continue;

            // Remove any + that's not at the beginning
            $cleaned = preg_replace('/(?!^)\+/', '', $cleaned);

            // Format based on pattern
            if (str_starts_with($cleaned, '+258')) {
                // Already has +258, keep first 12 characters (+258 + 9 digits)
                $formatted[] = substr($cleaned, 0, 13);

            } elseif (str_starts_with($cleaned, '258')) {
                // Has 258 but no +, add it and keep 12 digits
                $formatted[] = '+' . substr($cleaned, 0, 12);

            } elseif (str_starts_with($cleaned, '+') && strlen($cleaned) >= 12) {
                // Has + but not 258, might be valid international
                $formatted[] = substr($cleaned, 0, 13);

            } elseif (strlen($cleaned) == 9 && in_array($cleaned[0], ['8', '9'])) {
                // 9 digits starting with 8 or 9 - Mozambique mobile
                $formatted[] = '+258' . $cleaned;

            } elseif (strlen($cleaned) == 10 && $cleaned[0] == '0' && in_array($cleaned[1], ['8', '9'])) {
                // 10 digits starting with 08 or 09 - remove leading 0
                $formatted[] = '+258' . substr($cleaned, 1);

            } elseif (strlen($cleaned) >= 9) {
                // Fallback: take last 9 digits and add +258
                $last9 = substr($cleaned, -9);
                if (in_array($last9[0], ['8', '9'])) {
                    $formatted[] = '+258' . $last9;
                }
            }
        }

        // Remove duplicates and reindex
        return array_values(array_unique($formatted));
    }

    public function submit()
    {
        $this->validate();

        $user = Auth::user();

        // Ensure the user is not null
        if (!$user) {
            $this->alert('error', 'Usuário não autenticado.');
            return;
        }

        // Handle custom numbers mode
        if ($this->useCustomNumbers) {
            // Parse phone numbers
            $phoneNumbers = $this->parsePhoneNumbers($this->customNumbers);

            if (empty($phoneNumbers)) {
                $this->alert('error', 'Nenhum número válido encontrado');
                return;
            }

            // Send SMS first
            $smsService = new SmsService();
            $successCount = 0;
            $failCount = 0;

            foreach ($phoneNumbers as $phoneNumber) {
                try {
                    $result = $smsService->send($phoneNumber, $this->title);

                    if ($result['success']) {
                        $successCount++;
                    } else {
                        $failCount++;
                        Log::warning("SMS falhou para {$phoneNumber}: " . ($result['error'] ?? 'Erro desconhecido'));
                    }
                } catch (Exception $e) {
                    $failCount++;
                    Log::error("Erro ao enviar SMS para {$phoneNumber}: " . $e->getMessage());
                }
            }

            // Create database record with phone numbers and statistics
            $sendSms = SendSms::create([
                'title' => $this->title,
                'type' => 'custom',
                'description' => implode(', ', $phoneNumbers), // Store phone numbers for history
                'created_by' => $user->id,
                'total_sent' => count($phoneNumbers),
                'success_count' => $successCount,
                'fail_count' => $failCount,
            ]);

            // Show result
            if ($failCount > 0 && $successCount > 0) {
                $this->alert('warning', "SMS enviados: {$successCount} sucesso, {$failCount} falhas");
            } elseif ($failCount > 0 && $successCount === 0) {
                $this->alert('error', 'Erro ao enviar todas as mensagens por SMS');
            } else {
                $this->alert('success', "{$successCount} SMS enviados com sucesso");
            }

            Note::createSystemNotification(Admin::class, 'Nova SMS', 'SMS enviada para números personalizados: ' . $successCount . ' enviados');
            $this->emit('noteAdded');
            $this->reset(['title', 'customNumbers', 'useCustomNumbers']);
            return;
        }

        // Fix: Define type BEFORE saving to database
        if(empty($this->type)){
            $this->type = "all";
        }

        // Create DB record
        $smsData = [
            'title' => $this->title,
            'type' => $this->type,
            'class_id' => $this->class_id,
            'student_id' => $this->student_id,
            'created_by' => $user->id,
        ];

        if(!empty($this->photo)){
            $details = SystemController::store_media($this->photo);
            $smsData['media_name'] = $details[0];
            $smsData['media_url'] = $details[1];
        }

        $downloadcenter = SendSms::query()->create($smsData);

        // Determine recipients based on type
        $recipients = collect();
        $isGuardianType = in_array($this->type, ['all_guardians', 'class_guardians', 'guardian']);

        switch ($this->type) {
            case 'all':
                $recipients = User::where("is_paid", $this->student_type)->where("is_active", "1")->get();
                break;

            case 'all_guardians':
                $recipients = Guardian::where('is_active', true)->whereNotNull('phone')->get();
                Log::info("Enviando SMS para todos os encarregados. Total: " . $recipients->count());
                break;

            case 'class':
                $recipients = User::whereIn("classroom_id", $this->class_ids)
                    ->where("is_active", "1")
                    ->where("year_edu", date('Y'))
                    ->get();
                Log::info("Enviando SMS para " . count($this->class_ids) . " turma(s). Total de estudantes: " . $recipients->count());
                break;

            case 'class_guardians':
                $studentIds = User::whereIn('classroom_id', $this->class_ids)
                    ->where('is_active', '1')
                    ->where('year_edu', date('Y'))
                    ->pluck('id');
                $recipients = Guardian::where('is_active', true)
                    ->whereNotNull('phone')
                    ->whereHas('students', function ($q) use ($studentIds) {
                        $q->whereIn('users.id', $studentIds);
                    })->get();
                Log::info("Enviando SMS para encarregados de " . count($this->class_ids) . " turma(s). Total: " . $recipients->count());
                break;

            case 'student':
                $recipients = User::where("id", $this->student_id)->get();
                break;

            case 'guardian':
                $recipients = Guardian::where('id', $this->guardian_id)->get();
                break;
        }

        // Send SMS
        $smsService = new SmsService();
        $successCount = 0;
        $failCount = 0;

        foreach ($recipients as $recipient) {
            try {
                $phone = $isGuardianType ? $recipient->phone : $recipient->phone_number;
                $message = "Ola " . $recipient->name . " " . $this->title;

                $result = $smsService->send($phone, $message);

                if ($result['success']) {
                    $successCount++;
                } else {
                    $failCount++;
                    Log::warning("SMS falhou para {$phone}: " . ($result['error'] ?? 'Erro desconhecido'));
                }
            } catch (Exception $e) {
                $failCount++;
                Log::error("Erro ao enviar SMS para {$recipient->name}: " . $e->getMessage());
            }
        }

        // Update the record with statistics
        $downloadcenter->update([
            'total_sent' => $recipients->count(),
            'success_count' => $successCount,
            'fail_count' => $failCount,
        ]);

        // Mostrar resultado
        if ($failCount > 0 && $successCount > 0) {
            $this->alert('warning', "SMS enviados: {$successCount} sucesso, {$failCount} falhas");
        } elseif ($failCount > 0 && $successCount === 0) {
            $this->alert('error', 'Erro ao enviar todas as mensagens por SMS');
        } else {
            $this->alert('success', "{$successCount} SMS enviados com sucesso");
        }

        Note::createSystemNotification(Admin::class, 'Nova SMS', 'Nova SMS enviada com sucesso ' . $this->title);
        $this->emit('noteAdded');
        $this->reset();
    }

    public function confirmed()
    {
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
        
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }
    public function render()
    {
        return view('livewire.admin.sendsms.add-sms', [
            'Classroom' => Classroom::query()->orderByRaw('CAST(class AS UNSIGNED)')->get(),
            'students' => User::when($this->class_id, function($query) {
                return $query->where("classroom_id", $this->class_id);
            })->get(),
            'guardians' => Guardian::where('is_active', true)->orderBy('name')->get(),
            'class_ids' => $this->class_ids,
        ]);
    }
}
