<?php

namespace App\Http\Livewire\Auth;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class Reset extends Component
{
    use LivewireAlert;

    public $email;
    public $password;
    public $password_confirmation;

    public function mount(string $token)
    {
        if (!request()->hasValidSignature()) {
            session()->flash('error', 'Por favor, gere um novo link de redefinição de senha.');
            return redirect()->route('forgot');
        }

        $this->email = decrypt($token, true);
    }

    protected $rules = [
        'email' => ['required', 'string', 'email', 'exists:users', 'max:255'],
        'password' => ['required', 'string', 'min:8', 'confirmed', 'same:password_confirmation']
    ];

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function resetPassword()
    {
        $this->validate();

        $user = User::query()->firstWhere('email', $this->email);
        if (Hash::check($this->password, $user->password)) {
            $this->alert('warning', 'A nova senha não pode ser igual à sua senha atual.');
            return redirect()->back();
        }

        $user->update([
            'password' => bcrypt($this->password)
        ]);

        auth()->login($user);
        $this->alert('success', 'A senha da conta foi atualizada.');
        return redirect()->intended('home');
    }

    public function render()
    {
        return view('livewire.auth.reset')->layout('layouts.auth');
    }
}
