<?php

namespace App\Http\Livewire\Guardian;

use App\Models\Attendance;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Livewire\Component;

class Presencas extends Component
{
    public $viewMonth;
    public $viewYear;

    protected $listeners = ['studentSwitched' => '$refresh'];

    public function mount()
    {
        $this->viewMonth = now()->month;
        $this->viewYear  = now()->year;
    }

    public function previousMonth()
    {
        $d = Carbon::create($this->viewYear, $this->viewMonth, 1)->subMonth();
        $this->viewMonth = $d->month;
        $this->viewYear  = $d->year;
    }

    public function nextMonth()
    {
        $d = Carbon::create($this->viewYear, $this->viewMonth, 1)->addMonth();
        if ($d->isFuture() && $d->month > now()->month) return;
        $this->viewMonth = $d->month;
        $this->viewYear  = $d->year;
    }

    public function getActiveStudentProperty(): ?User
    {
        $id = Session::get('guardian_student_id');
        $g  = Auth::guard('guardian')->user();
        if (!$id || !$g) return null;
        return $g->students()->with('classroom')->find($id);
    }

    public function getAttendancesProperty()
    {
        $st = $this->active_student;
        if (!$st) return collect([]);

        $from = Carbon::create($this->viewYear, $this->viewMonth, 1)->startOfMonth();
        $to   = $from->copy()->endOfMonth();

        return Attendance::where('student_id', $st->id)
            ->whereBetween('date', [$from, $to])
            ->orderBy('date')->orderBy('subject')
            ->get();
    }

    public function getSummaryProperty(): array
    {
        $all     = $this->attendances;
        $total   = $all->count();
        $present = $all->where('status', 'present')->count();
        return [
            'total'     => $total,
            'present'   => $present,
            'absent'    => $all->where('status', 'absent')->count(),
            'late'      => $all->where('status', 'late')->count(),
            'justified' => $all->where('status', 'justified')->count(),
            'rate'      => $total > 0 ? round(($present / $total) * 100) : 0,
        ];
    }

    public function getCalendarDataProperty(): array
    {
        $st = $this->active_student;
        if (!$st) return [];

        $firstDay    = Carbon::create($this->viewYear, $this->viewMonth, 1);
        $lastDay     = $firstDay->copy()->endOfMonth();
        $startOfWeek = $firstDay->copy()->startOfWeek();

        $records = Attendance::where('student_id', $st->id)
            ->whereMonth('date', $this->viewMonth)
            ->whereYear('date', $this->viewYear)
            ->get()
            ->groupBy(fn($r) => $r->date->format('Y-m-d'));

        $calendar = [];
        $current  = $startOfWeek->copy();

        while ($current <= $lastDay || $current->dayOfWeek != 1) {
            $dateStr        = $current->format('Y-m-d');
            $isCurrentMonth = $current->month == $this->viewMonth;
            $dayRecords     = $records->get($dateStr, collect());

            $status = 'none';
            if ($isCurrentMonth && $dayRecords->isNotEmpty()) {
                if ($dayRecords->where('status', 'absent')->isNotEmpty())    $status = 'absent';
                elseif ($dayRecords->where('status', 'late')->isNotEmpty())  $status = 'late';
                elseif ($dayRecords->where('status', 'justified')->isNotEmpty()) $status = 'justified';
                else $status = 'present';
            }

            $calendar[] = [
                'date'             => $dateStr,
                'day'              => $current->day,
                'is_current_month' => $isCurrentMonth,
                'is_today'         => $current->isToday(),
                'status'           => $status,
            ];

            $current->addDay();
            if (!$isCurrentMonth && $current->month != $this->viewMonth && $current->dayOfWeek == 1) break;
        }

        return $calendar;
    }

    public function render()
    {
        return view('livewire.guardian.presencas', [
            'active_student' => $this->active_student,
            'attendances'    => $this->attendances,
            'summary'        => $this->summary,
            'calendarData'   => $this->calendarData,
            'monthLabel'     => Carbon::create($this->viewYear, $this->viewMonth, 1)
                                    ->locale('pt')->isoFormat('MMMM [de] YYYY'),
        ])->layout('layouts.app');
    }
}
