<?php

namespace App\Http\Livewire\Secretary;

use App\Models\Classroom;
use App\Models\PreEnrollment;
use App\Models\User;
use App\Services\GoogleVisionOCRService;
use App\Services\SmsService;
use Exception;
use Illuminate\Support\Facades\Storage;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;
use Livewire\WithFileUploads;

class PublicPreEnrollment extends Component
{
    use LivewireAlert, WithFileUploads;

    public $currentStep = 1;
    public $totalSteps = 6;

    // Step 1: Dados Pessoais
    public $candidate_name, $candidate_sex, $candidate_dob;
    public $candidate_phone, $candidate_email;
    public $marital_status, $nationality = 'Moçambicana', $nuit;
    public $chronic_disease = false, $chronic_disease_details;

    // Step 2: Documentos & Ingresso
    public $document_type, $document_id;
    public $document_issue_place, $document_expiry_date;
    public $entry_type = 'Normal';
    public $previous_institution, $previous_course;

    // Step 3: Morada
    public $birth_place, $birth_district, $birth_province;
    public $address_province, $address_district, $address_bairro;
    public $address_street, $address_house_number;
    public $candidate_address, $candidate_whatsapp;

    // Step 4: Filiação
    public $father_name, $father_phone;
    public $mother_name, $mother_phone;

    // Step 5: Encarregado
    public $guardian_name, $guardian_relationship;
    public $guardian_phone, $guardian_email, $guardian_address;
    public $guardian_document_id, $guardian_document_issue_place;
    public $guardian_document_issue_date;
    public $guardian_bairro, $guardian_quarteirao, $guardian_workplace;
    public $emergency_contact_name, $emergency_contact_phone;

    // Step 6: Curso/Turma
    public $desired_class, $desired_classroom_id;
    public $desired_shift;
    public $academic_year;
    public $observation;

    // OCR BI + Foto
    public $bi_document;
    public $processing_bi = false;
    public $bi_processed = false;
    public $extracted_photo_path = null;
    public $candidate_photo_upload;

    // Result
    public $submittedCode = null;
    public $submittedId = null;

    protected $listeners = ['confirmed', 'cancelled'];

    public function mount()
    {
        $this->academic_year = date('Y');
    }

    public function updatedBiDocument()
    {
        $this->processBiDocument();
    }

    public function processBiDocument()
    {
        if (!$this->bi_document) {
            return;
        }

        $this->processing_bi = true;

        try {
            $tempPath = $this->bi_document->store('temp', 'public');
            $fullPath = storage_path('app/public/' . $tempPath);

            $ocrService = new GoogleVisionOCRService();
            $result = $ocrService->processDocument($fullPath);

            if ($result['success']) {
                $extractedData = $ocrService->extractBIData($result['text']);

                // Step 1: Dados Pessoais
                if (!empty($extractedData['full_name'])) {
                    $this->candidate_name = $extractedData['full_name'];
                }
                if (!empty($extractedData['dob'])) {
                    $this->candidate_dob = $extractedData['dob'];
                }
                if (!empty($extractedData['sex'])) {
                    $this->candidate_sex = $extractedData['sex'] === 'M' ? 'Masculino' :
                        ($extractedData['sex'] === 'F' ? 'Feminino' : $extractedData['sex']);
                }

                // Step 2: Documentos
                if (!empty($extractedData['document_id'])) {
                    $this->document_id = $extractedData['document_id'];
                }
                if (!empty($extractedData['document_type'])) {
                    $this->document_type = $extractedData['document_type'];
                }

                // Step 3: Naturalidade
                if (!empty($extractedData['birth_place'])) {
                    $this->birth_place = $extractedData['birth_place'];
                }
                if (!empty($extractedData['birth_district'])) {
                    $this->birth_district = $extractedData['birth_district'];
                }
                if (!empty($extractedData['birth_province'])) {
                    $this->birth_province = $extractedData['birth_province'];
                }

                // Step 4: Filiação
                if (!empty($extractedData['father_name'])) {
                    $this->father_name = $extractedData['father_name'];
                }
                if (!empty($extractedData['mother_name'])) {
                    $this->mother_name = $extractedData['mother_name'];
                }

                // Extrair foto do BI
                $facePath = $ocrService->extractFaceFromDocument($fullPath);
                if ($facePath && file_exists($facePath)) {
                    $photoContent = file_get_contents($facePath);
                    $photoStoragePath = 'candidate_photos/' . uniqid('photo_') . '.jpg';
                    Storage::disk('public')->put($photoStoragePath, $photoContent);
                    $this->extracted_photo_path = $photoStoragePath;
                    @unlink($facePath);
                }

                $this->bi_processed = true;

                if ($this->extracted_photo_path) {
                    $this->alert('success', 'Dados e foto extraídos com sucesso! Verifique os campos preenchidos.');
                } else {
                    $this->alert('success', 'Dados extraídos! Foto não detectada — pode fazer upload manualmente.');
                }
            } else {
                $this->alert('error', 'Erro ao processar BI: ' . $result['error']);
            }

            Storage::disk('public')->delete($tempPath);
        } catch (Exception $e) {
            $this->alert('error', 'Erro ao processar BI: ' . $e->getMessage());
        }

        $this->processing_bi = false;
    }

    protected function step1Rules()
    {
        return [
            'candidate_name' => ['required', 'string', 'max:255'],
            'candidate_sex' => ['required', 'string', 'in:Masculino,Feminino'],
            'candidate_dob' => ['required', 'date'],
            'candidate_phone' => ['required', 'string', 'max:20', 'unique:users,phone_number'],
            'candidate_email' => ['nullable', 'email', 'max:255', 'unique:users,email'],
            'marital_status' => ['nullable', 'string', 'in:Solteiro,Casado'],
            'nationality' => ['nullable', 'string', 'max:255'],
            'nuit' => ['nullable', 'string', 'max:50'],
            'chronic_disease' => ['nullable', 'boolean'],
            'chronic_disease_details' => ['nullable', 'string', 'max:500'],
        ];
    }

    protected function validationAttributes()
    {
        return [
            'candidate_phone' => 'telefone',
            'candidate_email' => 'email',
        ];
    }

    protected function messages()
    {
        return [
            'candidate_phone.unique' => 'Este número de telefone já está registado no sistema. A inscrição é apenas para novos candidatos.',
            'candidate_email.unique' => 'Este email já está registado no sistema. A inscrição é apenas para novos candidatos.',
        ];
    }

    protected function step2Rules()
    {
        return [
            'document_type' => ['nullable', 'string', 'in:BI,Cedula,Passaporte'],
            'document_id' => ['nullable', 'string', 'max:255'],
            'document_issue_place' => ['nullable', 'string', 'max:255'],
            'document_expiry_date' => ['nullable', 'date'],
            'entry_type' => ['required', 'string', 'in:Normal,Transferencia,Reingresso'],
            'previous_institution' => ['nullable', 'string', 'max:255'],
            'previous_course' => ['nullable', 'string', 'max:255'],
        ];
    }

    protected function step3Rules()
    {
        return [
            'birth_place' => ['nullable', 'string', 'max:255'],
            'birth_district' => ['nullable', 'string', 'max:255'],
            'birth_province' => ['nullable', 'string'],
            'address_province' => ['nullable', 'string'],
            'address_district' => ['nullable', 'string', 'max:255'],
            'address_bairro' => ['nullable', 'string', 'max:255'],
            'address_street' => ['nullable', 'string', 'max:255'],
            'address_house_number' => ['nullable', 'string', 'max:50'],
            'candidate_address' => ['nullable', 'string', 'max:500'],
            'candidate_whatsapp' => ['nullable', 'string', 'max:20'],
        ];
    }

    protected function step4Rules()
    {
        return [
            'father_name' => ['nullable', 'string', 'max:255'],
            'father_phone' => ['nullable', 'string', 'max:20'],
            'mother_name' => ['nullable', 'string', 'max:255'],
            'mother_phone' => ['nullable', 'string', 'max:20'],
        ];
    }

    protected function step5Rules()
    {
        return [
            'guardian_name' => ['nullable', 'string', 'max:255'],
            'guardian_relationship' => ['nullable', 'string', 'max:100'],
            'guardian_phone' => ['nullable', 'string', 'max:20'],
            'guardian_email' => ['nullable', 'email', 'max:255'],
            'guardian_address' => ['nullable', 'string', 'max:500'],
            'guardian_document_id' => ['nullable', 'string', 'max:255'],
            'guardian_document_issue_place' => ['nullable', 'string', 'max:255'],
            'guardian_document_issue_date' => ['nullable', 'date'],
            'guardian_bairro' => ['nullable', 'string', 'max:255'],
            'guardian_quarteirao' => ['nullable', 'string', 'max:255'],
            'guardian_workplace' => ['nullable', 'string', 'max:255'],
            'emergency_contact_name' => ['nullable', 'string', 'max:255'],
            'emergency_contact_phone' => ['nullable', 'string', 'max:20'],
        ];
    }

    protected function step6Rules()
    {
        return [
            'desired_class' => ['required', 'string'],
            'desired_classroom_id' => ['nullable', 'string'],
            'desired_shift' => ['nullable', 'string', 'in:Diurno,Tarde'],
            'academic_year' => ['required', 'string'],
            'observation' => ['nullable', 'string', 'max:1000'],
        ];
    }

    public function nextStep()
    {
        $method = 'step' . $this->currentStep . 'Rules';
        $this->validate($this->$method());
        $this->currentStep++;
    }

    public function previousStep()
    {
        $this->currentStep--;
    }

    public function submit()
    {
        $this->validate($this->step6Rules());

        $this->confirm('Confirmar submissão da pré-inscrição?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, submeter!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $code = PreEnrollment::generateCode();

        PreEnrollment::create([
            'code' => $code,
            'status' => 'pendente',
            'submitted_at' => now(),
            'candidate_name' => $this->candidate_name,
            'candidate_sex' => $this->candidate_sex,
            'candidate_dob' => $this->candidate_dob,
            'candidate_phone' => $this->candidate_phone,
            'candidate_email' => $this->candidate_email,
            'candidate_photo' => $this->getCandidatePhotoPath(),
            'candidate_address' => $this->candidate_address,
            'candidate_origin' => $this->entry_type === 'Normal' ? 'Interno' : 'Externo',
            'marital_status' => $this->marital_status,
            'nationality' => $this->nationality,
            'nuit' => $this->nuit,
            'chronic_disease' => $this->chronic_disease,
            'chronic_disease_details' => $this->chronic_disease ? $this->chronic_disease_details : null,
            'father_name' => $this->father_name,
            'mother_name' => $this->mother_name,
            'father_phone' => $this->father_phone,
            'mother_phone' => $this->mother_phone,
            'birth_place' => $this->birth_place,
            'birth_district' => $this->birth_district,
            'birth_province' => $this->birth_province,
            'document_type' => $this->document_type,
            'document_id' => $this->document_id,
            'document_issue_place' => $this->document_issue_place,
            'document_expiry_date' => $this->document_expiry_date,
            'entry_type' => $this->entry_type,
            'previous_institution' => $this->previous_institution,
            'previous_course' => $this->previous_course,
            'address_province' => $this->address_province,
            'address_district' => $this->address_district,
            'address_bairro' => $this->address_bairro,
            'address_street' => $this->address_street,
            'address_house_number' => $this->address_house_number,
            'candidate_whatsapp' => $this->candidate_whatsapp,
            'guardian_name' => $this->guardian_name,
            'guardian_relationship' => $this->guardian_relationship,
            'guardian_phone' => $this->guardian_phone,
            'guardian_email' => $this->guardian_email,
            'guardian_address' => $this->guardian_address,
            'guardian_document_id' => $this->guardian_document_id,
            'guardian_document_issue_place' => $this->guardian_document_issue_place,
            'guardian_document_issue_date' => $this->guardian_document_issue_date,
            'guardian_bairro' => $this->guardian_bairro,
            'guardian_quarteirao' => $this->guardian_quarteirao,
            'guardian_workplace' => $this->guardian_workplace,
            'emergency_contact_name' => $this->emergency_contact_name,
            'emergency_contact_phone' => $this->emergency_contact_phone,
            'desired_class' => $this->desired_class,
            'desired_classroom_id' => $this->desired_classroom_id,
            'desired_shift' => $this->desired_shift,
            'academic_year' => $this->academic_year,
            'observation' => $this->observation,
        ]);

        // Send SMS confirmation
        try {
            $phone = $this->guardian_phone ?: $this->candidate_phone;
            $message = "COPMOZ: Pre-inscricao recebida com sucesso! Codigo: {$code}. Aguarde confirmacao da secretaria.";
            (new SmsService())->send($phone, $message);
        } catch (Exception $e) {
            // Log but don't fail
        }

        $this->submittedCode = $code;
        $this->submittedId = PreEnrollment::where('code', $code)->value('id');
        $this->currentStep = 7;
    }

    private function getCandidatePhotoPath(): ?string
    {
        if ($this->candidate_photo_upload) {
            return $this->candidate_photo_upload->store('candidate_photos', 'public');
        }

        if ($this->extracted_photo_path && Storage::disk('public')->exists($this->extracted_photo_path)) {
            return $this->extracted_photo_path;
        }

        return null;
    }

    public function cancelled()
    {
        $this->alert('info', 'Submissão cancelada.');
    }

    public function render()
    {
        $classrooms = Classroom::query()
            ->when($this->desired_class, fn($q) => $q->where('class', $this->desired_class))
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $availableClasses = Classroom::where('is_active', true)
            ->select('class')
            ->distinct()
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->pluck('class');

        return view('livewire.secretary.public-pre-enrollment', [
            'classrooms' => $classrooms,
            'availableClasses' => $availableClasses,
        ])->layout('layouts.pre-enrollment');
    }
}
