<?php

namespace App\Models;

use App\Traits\Uuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class PreEnrollment extends Model
{
    use HasFactory, Uuids, SoftDeletes;

    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'code', 'status',
        'candidate_name', 'candidate_sex', 'candidate_dob',
        'candidate_phone', 'candidate_email', 'candidate_photo', 'candidate_address',
        'candidate_origin',
        'marital_status', 'nationality', 'nuit',
        'chronic_disease', 'chronic_disease_details',
        'father_name', 'mother_name',
        'father_phone', 'mother_phone',
        'birth_place', 'birth_district', 'birth_province',
        'document_type', 'document_id',
        'document_issue_place', 'document_expiry_date',
        'entry_type', 'previous_institution', 'previous_course',
        'address_province', 'address_district', 'address_bairro',
        'address_street', 'address_house_number',
        'candidate_whatsapp',
        'guardian_name', 'guardian_relationship', 'guardian_phone',
        'guardian_email', 'guardian_address',
        'guardian_document_id', 'guardian_document_issue_place',
        'guardian_document_issue_date',
        'guardian_bairro', 'guardian_quarteirao', 'guardian_workplace',
        'emergency_contact_name', 'emergency_contact_phone',
        'desired_class', 'desired_classroom_id', 'desired_shift',
        'academic_year',
        'submitted_at', 'confirmed_at', 'cancelled_at',
        'cancellation_reason',
        'registered_by', 'confirmed_by', 'enrolled_by',
        'resulting_user_id',
        'observation',
    ];

    protected $casts = [
        'candidate_dob' => 'date',
        'document_expiry_date' => 'date',
        'guardian_document_issue_date' => 'date',
        'chronic_disease' => 'boolean',
        'submitted_at' => 'datetime',
        'confirmed_at' => 'datetime',
        'cancelled_at' => 'datetime',
    ];

    // Relationships
    public function desiredClassroom(): BelongsTo
    {
        return $this->belongsTo(Classroom::class, 'desired_classroom_id');
    }

    public function resultingStudent(): BelongsTo
    {
        return $this->belongsTo(User::class, 'resulting_user_id');
    }

    public function confirmedByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'confirmed_by');
    }

    public function enrolledByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'enrolled_by');
    }

    // Code generation
    public static function generateCode(): string
    {
        $today = now()->format('ymd');
        $lastToday = self::where('code', 'like', "PI%-{$today}")
            ->orderBy('code', 'desc')
            ->first();

        $seq = 1;
        if ($lastToday) {
            $parts = explode('-', $lastToday->code);
            $seq = intval(substr($parts[0], 2)) + 1;
        }

        return sprintf('PI%03d-%s', $seq, $today);
    }

    // Scopes
    public function scopePendente($query)
    {
        return $query->where('status', 'pendente');
    }

    public function scopeConfirmada($query)
    {
        return $query->where('status', 'confirmada');
    }

    public function scopeAnulada($query)
    {
        return $query->where('status', 'anulada');
    }

    public function scopeMatriculada($query)
    {
        return $query->where('status', 'matriculada');
    }

    // Status helpers
    public function isPendente(): bool
    {
        return $this->status === 'pendente';
    }

    public function isConfirmada(): bool
    {
        return $this->status === 'confirmada';
    }

    public function isAnulada(): bool
    {
        return $this->status === 'anulada';
    }

    public function isMatriculada(): bool
    {
        return $this->status === 'matriculada';
    }

    public function getStatusBadgeClass(): string
    {
        return match ($this->status) {
            'pendente' => 'warning',
            'confirmada' => 'info',
            'anulada' => 'danger',
            'matriculada' => 'success',
            default => 'secondary',
        };
    }

    public function getStatusLabel(): string
    {
        return match ($this->status) {
            'pendente' => 'Pendente',
            'confirmada' => 'Confirmada',
            'anulada' => 'Anulada',
            'matriculada' => 'Matriculada',
            default => $this->status,
        };
    }
}
