<?php

namespace App\Observers;

use App\Models\PaymentReference;
use App\Models\POS\Product;
use App\Models\POS\Sale;
use App\Models\POS\SaleItem;
use Illuminate\Support\Facades\Log;

class PaymentReferenceObserver
{
    /**
     * Handle the PaymentReference "updated" event.
     */
    public function updated(PaymentReference $paymentReference): void
    {
        // Verificar se o status mudou para 'paid'
        if ($paymentReference->isDirty('status') && $paymentReference->status === 'paid') {
            $this->processPosSale($paymentReference);
        }
    }

    /**
     * Processar venda POS quando pagamento é confirmado
     */
    private function processPosSale(PaymentReference $paymentReference): void
    {
        // Verificar se tem produto POS vinculado
        if (!$paymentReference->metadata || !isset($paymentReference->metadata['pos_product_id'])) {
            return;
        }

        try {
            $productId = $paymentReference->metadata['pos_product_id'];
            $product = Product::find($productId);

            if (!$product) {
                Log::warning('Produto POS não encontrado ao processar pagamento', [
                    'reference_id' => $paymentReference->id,
                    'product_id' => $productId,
                ]);
                return;
            }

            // Verificar stock disponível
            if ($product->stock < 1) {
                Log::error('Stock insuficiente ao processar pagamento POS', [
                    'reference_id' => $paymentReference->id,
                    'product_id' => $product->id,
                    'current_stock' => $product->stock,
                ]);
                return;
            }

            // Gerar número de venda
            $saleNumber = 'POS-' . now()->format('YmdHis') . '-' . $paymentReference->id;

            // Criar venda POS
            $sale = Sale::create([
                'sale_number' => $saleNumber,
                'student_id' => $paymentReference->student_id,
                'payment_method' => 'payment_reference',
                'subtotal' => $product->price,
                'discount' => 0,
                'tax' => 0,
                'tax_rate' => 0,
                'total' => $product->price,
                'status' => 'completed', // Já está pago
                'notes' => 'Venda via Referência de Pagamento: ' . $paymentReference->reference_number,
            ]);

            // Criar item da venda
            SaleItem::create([
                'sale_id' => $sale->id,
                'product_id' => $product->id,
                'quantity' => 1,
                'unit_price' => $product->price,
                'subtotal' => $product->price,
                'tax_amount' => 0,
                'total' => $product->price,
            ]);

            // Abater stock AGORA que o pagamento foi confirmado
            $product->decrement('stock', 1);

            // Atualizar metadata com ID da venda criada
            $metadata = $paymentReference->metadata;
            $metadata['pos_sale_id'] = $sale->id;
            $metadata['pos_sale_number'] = $saleNumber;
            $metadata['pos_processed_at'] = now()->toDateTimeString();
            $paymentReference->update(['metadata' => $metadata]);

            Log::info('Venda POS criada e stock abatido após confirmação de pagamento', [
                'sale_id' => $sale->id,
                'sale_number' => $saleNumber,
                'product_id' => $product->id,
                'reference_id' => $paymentReference->id,
                'stock_remaining' => $product->fresh()->stock,
            ]);

        } catch (\Exception $e) {
            Log::error('Erro ao processar venda POS após confirmação de pagamento', [
                'error' => $e->getMessage(),
                'reference_id' => $paymentReference->id,
                'trace' => $e->getTraceAsString(),
            ]);
        }
    }
}
