<?php

namespace App\Services;

use App\Models\ExamConfiguration;
use App\Models\Subject;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class ExamConfigurationService
{
    /**
     * Salva/atualiza configurações em lote para uma classe
     *
     * @param int $classLevel
     * @param array $configurations ['subject_id' => 'exam_type', ...]
     * @param int|null $academicYear
     * @return bool
     */
    public function saveConfigurationsForClass(
        int $classLevel,
        array $configurations,
        ?int $academicYear = null
    ): bool {
        DB::beginTransaction();

        try {
            $userId = Auth::guard('admin')->id();

            foreach ($configurations as $subjectId => $examType) {
                ExamConfiguration::updateOrCreate(
                    [
                        'subject_id' => $subjectId,
                        'class_level' => $classLevel,
                        'academic_year' => $academicYear
                    ],
                    [
                        'exam_type' => $examType,
                        'is_active' => true,
                        'updated_by' => $userId
                    ]
                );
            }

            DB::commit();
            ExamConfiguration::clearCache();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erro ao salvar configurações de exames: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Carrega configurações existentes para uma classe
     *
     * @param int $classLevel
     * @param int|null $academicYear
     * @return array
     */
    public function loadConfigurationsForClass(
        int $classLevel,
        ?int $academicYear = null
    ): array {
        $subjects = Subject::orderBy('name')->get();
        $configurations = [];

        foreach ($subjects as $subject) {
            $config = ExamConfiguration::where('subject_id', $subject->id)
                ->where('class_level', $classLevel)
                ->where('academic_year', $academicYear)
                ->first();

            $configurations[$subject->id] = [
                'id' => $subject->id,
                'name' => $subject->name,
                'exam_type' => $config ? $config->exam_type : 'none',
                'config_id' => $config ? $config->id : null
            ];
        }

        return $configurations;
    }

    /**
     * Copia configurações de um ano para outro (todas as classes ou uma específica)
     *
     * @param int $fromYear
     * @param int $toYear
     * @param int|null $classLevel Se null, copia todas as classes
     * @return array ['success' => bool, 'copied' => int, 'skipped' => int]
     */
    public function copyConfigurationsToNewYear(int $fromYear, int $toYear, ?int $classLevel = null): array
    {
        DB::beginTransaction();

        try {
            $query = ExamConfiguration::where('academic_year', $fromYear)
                ->where('is_active', true);

            // Se especificou uma classe, filtrar
            if ($classLevel !== null) {
                $query->where('class_level', $classLevel);
            }

            $configs = $query->get();

            if ($configs->isEmpty()) {
                DB::rollBack();
                return ['success' => false, 'copied' => 0, 'skipped' => 0, 'message' => 'Nenhuma configuração encontrada no ano ' . $fromYear];
            }

            $userId = Auth::guard('admin')->id();
            $copied = 0;
            $skipped = 0;

            foreach ($configs as $config) {
                // Verificar se já existe configuração para este ano/classe/disciplina
                $existing = ExamConfiguration::where('subject_id', $config->subject_id)
                    ->where('class_level', $config->class_level)
                    ->where('academic_year', $toYear)
                    ->first();

                if ($existing) {
                    // Atualizar configuração existente
                    $existing->update([
                        'exam_type' => $config->exam_type,
                        'is_active' => true,
                        'updated_by' => $userId
                    ]);
                    $skipped++;
                } else {
                    // Criar nova configuração
                    ExamConfiguration::create([
                        'subject_id' => $config->subject_id,
                        'class_level' => $config->class_level,
                        'exam_type' => $config->exam_type,
                        'is_active' => true,
                        'academic_year' => $toYear,
                        'created_by' => $userId
                    ]);
                    $copied++;
                }
            }

            DB::commit();
            ExamConfiguration::clearCache();

            return [
                'success' => true,
                'copied' => $copied,
                'skipped' => $skipped,
                'message' => "Copiadas: {$copied}, Atualizadas: {$skipped}"
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erro ao copiar configurações: ' . $e->getMessage());
            return ['success' => false, 'copied' => 0, 'skipped' => 0, 'message' => 'Erro: ' . $e->getMessage()];
        }
    }

    /**
     * Verifica se existem configurações para um ano específico
     *
     * @param int $year
     * @param int|null $classLevel
     * @return int Número de configurações
     */
    public function countConfigurationsForYear(int $year, ?int $classLevel = null): int
    {
        $query = ExamConfiguration::where('academic_year', $year)
            ->where('is_active', true);

        if ($classLevel !== null) {
            $query->where('class_level', $classLevel);
        }

        return $query->count();
    }
}
