<?php

namespace App\Services;

use App\Models\SmsLog;
use App\Models\SystemSetting;
use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Twilio\Rest\Client as TwilioClient;

class SmsService
{
    protected string $gateway;
    protected array $config;
    protected string $countryCode;
    protected bool $logMessages;

    public function __construct(?string $gateway = null)
    {
        $this->loadConfiguration($gateway);
    }

    /**
     * Load configuration from database or fallback to config files
     */
    protected function loadConfiguration(?string $gateway = null): void
    {
        // Try to load from database first if table exists and has SMS settings
        if ($this->databaseSettingsExist() && $this->hasValidDbSettings()) {
            $this->gateway = $gateway ?? SystemSetting::get('gateway', 'movitel', 'sms');
            $this->countryCode = SystemSetting::get('country_code', '258', 'sms');
            $this->logMessages = SystemSetting::get('log_messages', true, 'sms');

            // Load gateway-specific config from database
            if ($this->gateway === 'movitel') {
                $this->config = [
                    'api_url' => SystemSetting::get('movitel_api_url', 'https://sms.itcore.app/api/http/sms/send', 'sms'),
                    'api_token' => SystemSetting::get('movitel_api_token', '', 'sms'),
                    'sender_id' => SystemSetting::get('movitel_sender_id', 'COPMOZ', 'sms'),
                ];
            } else {
                $this->config = [
                    'sid' => SystemSetting::get('twilio_sid', '', 'sms'),
                    'token' => SystemSetting::get('twilio_token', '', 'sms'),
                    'from' => SystemSetting::get('twilio_from', 'COPMOZ', 'sms'),
                ];
            }
        } else {
            // Fallback to config files / .env
            $this->gateway = $gateway ?? config('sms.default', 'movitel');
            $this->countryCode = config('sms.country_code', '258');
            $this->logMessages = config('sms.log_messages', true);

            // Load from config files
            if ($this->gateway === 'movitel') {
                $this->config = [
                    'api_url' => config('sms.gateways.movitel.api_url', 'https://sms.itcore.app/api/http/sms/send'),
                    'api_token' => config('sms.gateways.movitel.api_token', ''),
                    'sender_id' => config('sms.gateways.movitel.sender_id', 'COPMOZ'),
                ];
            } else {
                $this->config = [
                    'sid' => config('sms.gateways.twilio.sid', ''),
                    'token' => config('sms.gateways.twilio.token', ''),
                    'from' => config('sms.gateways.twilio.from', 'COPMOZ'),
                ];
            }
        }
    }

    /**
     * Check if database settings table exists
     */
    protected function databaseSettingsExist(): bool
    {
        try {
            return Schema::hasTable('system_settings');
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Check if database has valid SMS settings (token configured)
     */
    protected function hasValidDbSettings(): bool
    {
        try {
            $token = SystemSetting::get('movitel_api_token', '', 'sms');
            $twilioSid = SystemSetting::get('twilio_sid', '', 'sms');
            // Return true only if at least one gateway has credentials configured
            return !empty($token) || !empty($twilioSid);
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Send SMS to a single recipient
     */
    public function send(string $phoneNumber, string $message, ?string $senderId = null): array
    {
        $formattedPhone = $this->formatPhoneNumber($phoneNumber);

        try {
            $result = match ($this->gateway) {
                'twilio' => $this->sendViaTwilio($formattedPhone, $message, $senderId),
                'movitel' => $this->sendViaMovitel($formattedPhone, $message, $senderId),
                default => throw new Exception("Gateway '{$this->gateway}' não suportado"),
            };

            $this->logMessage($phoneNumber, $message, $result);

            return $result;
        } catch (Exception $e) {
            $errorResult = [
                'success' => false,
                'gateway' => $this->gateway,
                'phone' => $formattedPhone,
                'error' => $e->getMessage(),
            ];

            $this->logMessage($phoneNumber, $message, $errorResult);
            Log::channel('daily')->error("SMS Error [{$this->gateway}]: " . $e->getMessage());

            return $errorResult;
        }
    }

    /**
     * Send SMS to multiple recipients
     */
    public function sendBulk(array $phoneNumbers, string $message, ?string $senderId = null): array
    {
        $results = [
            'total' => count($phoneNumbers),
            'success' => 0,
            'failed' => 0,
            'details' => [],
        ];

        // For Movitel, we can send bulk in a single API call
        if ($this->gateway === 'movitel' && count($phoneNumbers) > 1) {
            $formattedNumbers = array_map([$this, 'formatPhoneNumber'], $phoneNumbers);
            $result = $this->sendViaMovitelBulk($formattedNumbers, $message, $senderId);

            if ($result['success']) {
                $results['success'] = count($phoneNumbers);
            } else {
                $results['failed'] = count($phoneNumbers);
            }
            $results['details'][] = $result;

            return $results;
        }

        // For other gateways, send individually
        foreach ($phoneNumbers as $phoneNumber) {
            $result = $this->send($phoneNumber, $message, $senderId);

            if ($result['success']) {
                $results['success']++;
            } else {
                $results['failed']++;
            }

            $results['details'][] = $result;
        }

        return $results;
    }

    /**
     * Send SMS via Movitel Gateway (Bulk SMS ITCORE)
     */
    protected function sendViaMovitel(string $phoneNumber, string $message, ?string $senderId = null): array
    {
        $apiUrl = $this->config['api_url'] ?? 'https://sms.itcore.app/api/http/sms/send';
        $apiToken = $this->config['api_token'] ?? '';
        $sender = $senderId ?? $this->config['sender_id'] ?? 'COPMOZ';

        if (empty($apiToken)) {
            throw new Exception('Movitel SMS API token não configurado');
        }

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ])->post($apiUrl, [
            'api_token' => $apiToken,
            'recipient' => $phoneNumber,
            'sender_id' => $sender,
            'type' => 'plain',
            'message' => $message,
        ]);

        $responseData = $response->json();

        // Log response para debug
        Log::channel('daily')->info("SMS Movitel Response: " . json_encode($responseData));

        if ($response->successful() && ($responseData['status'] ?? '') === 'success') {
            return [
                'success' => true,
                'gateway' => 'movitel',
                'phone' => $phoneNumber,
                'message_id' => $responseData['data']['uid'] ?? null,
                'response' => $responseData,
            ];
        }

        throw new Exception($responseData['message'] ?? 'Erro desconhecido ao enviar SMS via Movitel');
    }

    /**
     * Send bulk SMS via Movitel Gateway
     */
    protected function sendViaMovitelBulk(array $phoneNumbers, string $message, ?string $senderId = null): array
    {
        $apiUrl = $this->config['api_url'] ?? 'https://sms.itcore.app/api/http/sms/send';
        $apiToken = $this->config['api_token'] ?? '';
        $sender = $senderId ?? $this->config['sender_id'] ?? 'COPMOZ';

        if (empty($apiToken)) {
            throw new Exception('Movitel SMS API token não configurado');
        }

        // Join multiple numbers with comma as per API docs
        $recipients = implode(',', $phoneNumbers);

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ])->post($apiUrl, [
            'api_token' => $apiToken,
            'recipient' => $recipients,
            'sender_id' => $sender,
            'type' => 'plain',
            'message' => $message,
        ]);

        $responseData = $response->json();

        if ($response->successful() && ($responseData['status'] ?? '') === 'success') {
            return [
                'success' => true,
                'gateway' => 'movitel',
                'phones' => $phoneNumbers,
                'count' => count($phoneNumbers),
                'response' => $responseData,
            ];
        }

        throw new Exception($responseData['message'] ?? 'Erro desconhecido ao enviar SMS em massa via Movitel');
    }

    /**
     * Send SMS via Twilio
     */
    protected function sendViaTwilio(string $phoneNumber, string $message, ?string $senderId = null): array
    {
        $sid = $this->config['sid'] ?? '';
        $token = $this->config['token'] ?? '';
        $from = $senderId ?? $this->config['from'] ?? '';

        if (empty($sid) || empty($token)) {
            throw new Exception('Twilio credentials não configuradas');
        }

        $client = new TwilioClient($sid, $token);

        $twilioMessage = $client->messages->create('+' . $phoneNumber, [
            'from' => $from,
            'body' => $message,
            'riskCheck' => 'disable',
        ]);

        return [
            'success' => true,
            'gateway' => 'twilio',
            'phone' => $phoneNumber,
            'message_id' => $twilioMessage->sid,
            'status' => $twilioMessage->status,
        ];
    }

    /**
     * Schedule SMS for later delivery (Movitel only)
     */
    public function schedule(string $phoneNumber, string $message, string $scheduleTime, ?string $senderId = null): array
    {
        if ($this->gateway !== 'movitel') {
            throw new Exception('Agendamento de SMS só é suportado pelo gateway Movitel');
        }

        $formattedPhone = $this->formatPhoneNumber($phoneNumber);
        $apiUrl = $this->config['api_url'] ?? 'https://sms.itcore.app/api/http/sms/send';
        $apiToken = $this->config['api_token'] ?? '';
        $sender = $senderId ?? $this->config['sender_id'] ?? 'COPMOZ';

        if (empty($apiToken)) {
            throw new Exception('Movitel SMS API token não configurado');
        }

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ])->post($apiUrl, [
            'api_token' => $apiToken,
            'recipient' => $formattedPhone,
            'sender_id' => $sender,
            'type' => 'plain',
            'message' => $message,
            'schedule_time' => $scheduleTime, // Format: Y-m-d H:i
        ]);

        $responseData = $response->json();

        if ($response->successful() && ($responseData['status'] ?? '') === 'success') {
            return [
                'success' => true,
                'gateway' => 'movitel',
                'phone' => $formattedPhone,
                'scheduled_for' => $scheduleTime,
                'response' => $responseData,
            ];
        }

        throw new Exception($responseData['message'] ?? 'Erro ao agendar SMS');
    }

    /**
     * Get SMS status/details (Movitel only)
     */
    public function getStatus(string $messageId): array
    {
        if ($this->gateway !== 'movitel') {
            throw new Exception('Verificação de status só é suportada pelo gateway Movitel');
        }

        $apiUrl = rtrim($this->config['api_url'], '/send') . '/' . $messageId;
        $apiToken = $this->config['api_token'] ?? '';

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $apiToken,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ])->get($apiUrl);

        $responseData = $response->json();

        if ($response->successful() && ($responseData['status'] ?? '') === 'success') {
            return [
                'success' => true,
                'data' => $responseData['data'] ?? [],
            ];
        }

        throw new Exception($responseData['message'] ?? 'Erro ao obter status do SMS');
    }

    /**
     * Format phone number with country code
     */
    protected function formatPhoneNumber(string $phoneNumber): string
    {
        // Remove any non-numeric characters
        $cleaned = preg_replace('/[^0-9]/', '', $phoneNumber);

        // If already starts with country code, return as is
        if (str_starts_with($cleaned, $this->countryCode)) {
            return $cleaned;
        }

        // If starts with 0, remove it
        if (str_starts_with($cleaned, '0')) {
            $cleaned = substr($cleaned, 1);
        }

        return $this->countryCode . $cleaned;
    }

    /**
     * Log SMS message to database
     */
    protected function logMessage(string $phoneNumber, string $message, array $result): void
    {
        if (!$this->logMessages) {
            return;
        }

        try {
            SmsLog::create([
                'gateway' => $this->gateway,
                'phone_number' => $phoneNumber,
                'message' => $message,
                'message_id' => $result['message_id'] ?? null,
                'status' => $result['success'] ? 'sent' : 'failed',
                'error' => $result['error'] ?? null,
                'response' => json_encode($result['response'] ?? $result),
            ]);
        } catch (Exception $e) {
            Log::channel('daily')->warning("Failed to log SMS: " . $e->getMessage());
        }
    }

    /**
     * Switch to a different gateway
     */
    public function useGateway(string $gateway): self
    {
        $this->loadConfiguration($gateway);
        return $this;
    }

    /**
     * Get current gateway name
     */
    public function getGateway(): string
    {
        return $this->gateway;
    }

    /**
     * Set configuration directly (useful for testing)
     */
    public function setConfig(array $config): self
    {
        $this->config = array_merge($this->config, $config);
        return $this;
    }

    /**
     * Get current configuration
     */
    public function getConfig(): array
    {
        return $this->config;
    }
}
