# Componente: Payment References Grouped

## 📋 Descrição

Componente Blade reutilizável para exibir referências de pagamento agrupadas por mês com design profissional e responsivo.

## 🎯 Características

- ✅ **Agrupamento automático por mês/ano**
- ✅ **Design profissional com gradientes**
- ✅ **Suporta múltiplas referências por mês**
- ✅ **Estados visuais (pago, pendente, aprovado, expirado)**
- ✅ **Totais calculados automaticamente**
- ✅ **Ações personalizáveis**
- ✅ **Responsivo e mobile-friendly**

## 📦 Uso Básico

```blade
<x-payment-references-grouped
    :references="$payment_references"
    :showActions="true"
/>
```

## 🔧 Parâmetros

| Parâmetro | Tipo | Default | Descrição |
|-----------|------|---------|-----------|
| `references` | Collection/Array | **obrigatório** | Collection de PaymentReference |
| `showActions` | Boolean | `true` | Mostrar botões de ação (Pagar/Ver) |
| `showCheckboxes` | Boolean | `false` | Mostrar checkboxes para seleção |
| `selectedReferences` | Array | `[]` | IDs das referências selecionadas |

## 📝 Exemplos de Uso

### 1. View de Estudante (User)

```blade
<!-- livewire/user/fee/fee-collection.blade.php -->

<div class="page-content">
    <h2>Minhas Referências</h2>

    <x-payment-references-grouped
        :references="$payment_references"
        :showActions="true"
    />
</div>

<script>
function payReference(referenceId) {
    // Redirecionar para página de pagamento
    window.location.href = '/user/payments/' + referenceId;
}

function viewReference(referenceId) {
    // Abrir modal com detalhes
    Livewire.emit('viewReferenceDetails', referenceId);
}
</script>
```

### 2. View de Admin (Aprovação)

```blade
<!-- livewire/admin/fee/approve-references.blade.php -->

<x-payment-references-grouped
    :references="$references"
    :showActions="false"
    :showCheckboxes="true"
    :selectedReferences="$selectedReferences"
/>

<!-- Botões de ação em massa -->
<button wire:click="approveSelected">Aprovar Selecionadas</button>
```

### 3. Relatórios

```blade
<!-- livewire/admin/fee/references-report.blade.php -->

<x-payment-references-grouped
    :references="$filteredReferences"
    :showActions="false"
    :showCheckboxes="false"
/>
```

## 🎨 Estrutura Visual

### Cada Grupo de Mês Contém:
1. **Cabeçalho** (gradiente roxo)
   - Nome do mês/ano
   - Quantidade de pagamentos
   - Total do mês

2. **Lista de Referências**
   - Descrição do pagamento
   - Número da referência (formato monospace)
   - Valor
   - Data de expiração
   - Status (badge colorido)
   - Ações (se habilitado)

### Estados de Referências:

| Estado | Cor | Quando Aparece |
|--------|-----|----------------|
| **Pago** | Verde | `status === 'paid'` |
| **Aprovado** | Azul | `approval_status === 'approved'` |
| **Pendente** | Laranja | Aguardando aprovação/pagamento |
| **Expirado** | Vermelho | `expires_at < now()` |

## 💾 Estrutura de Dados Esperada

O componente espera uma collection de objetos `PaymentReference` com:

```php
PaymentReference {
    id: string (UUID)
    reference_number: string (13 dígitos)
    student_id: string
    amount: decimal
    fee_month: int (1-12)
    fee_year: int
    status: enum('pending', 'paid', 'expired')
    approval_status: enum('pending_approval', 'approved', 'rejected')
    expires_at: Carbon
    metadata: json {
        description?: string
        custom_fee_type?: string
        pos_product_name?: string
    }
}
```

## 🔄 Atualizar Componente Livewire

Para usar este componente com Livewire, certifique-se de retornar `payment_references` agrupadas:

```php
// app/Http/Livewire/User/Fee/FeeCollectionUser.php

public function render()
{
    $payment_references = PaymentReference::where('student_id', $this->studentId)
        ->where('fee_year', $this->selectedYear)
        ->where('approval_status', 'approved')
        ->orderBy('fee_month', 'desc')
        ->get();

    return view('livewire.user.fee.fee-collection', [
        'payment_references' => $payment_references,
    ]);
}
```

## 🎯 Casos de Uso

### ✅ Estudantes (User)
- Ver todas as referências do ano
- Pagar referências pendentes
- Baixar comprovantes de pagamento
- Acompanhar status

### ✅ Administradores (Admin)
- Aprovar referências em massa
- Visualizar relatórios por mês
- Filtrar por status/turma
- Exportar dados

### ✅ Relatórios
- Consolidar pagamentos por período
- Análise mensal de arrecadação
- Acompanhamento de inadimplência

## 🔧 Customização

### Alterar Cores

Edite o arquivo `payment-references-grouped.blade.php`:

```css
.month-group-header {
    background: linear-gradient(135deg, #SUA_COR_1 0%, #SUA_COR_2 100%);
}
```

### Adicionar Novos Estados

```css
.reference-item.seu-estado {
    background: linear-gradient(to right, #COR 0%, #ffffff 100%);
    border-left: 4px solid #COR;
}
```

### Customizar Ações

Modifique a seção de ações no componente:

```blade
@if($showActions)
<div class="col-md-2 text-end">
    <button onclick="suaAcao({{ $reference->id }})">
        Sua Ação
    </button>
</div>
@endif
```

## 📱 Responsividade

O componente é totalmente responsivo:
- **Desktop**: Layout em grid com todas as informações
- **Tablet**: Colunas ajustadas
- **Mobile**: Layout vertical empilhado

## 🐛 Troubleshooting

### Referências não aparecem agrupadas
- Verifique se `fee_month` e `fee_year` estão preenchidos
- Confirme que está passando uma Collection/Array válida

### Estilos não aplicados
- Confirme que o componente está em `resources/views/components/`
- Limpe cache de views: `php artisan view:clear`

### Ações não funcionam
- Implemente as funções `payReference()` e `viewReference()` no JavaScript
- Ou use Livewire: `wire:click="metodo({{ $reference->id }})"`

## 📚 Referências

- Componente criado em: 23/12/2025
- Localização: `/resources/views/components/payment-references-grouped.blade.php`
- View de exemplo: `/resources/views/livewire/user/fee/payment-references-view.blade.php`
