<?php

namespace App\Console\Commands;

use App\Models\PaymentReference;
use App\Services\SmsService;
use App\Services\ReferenceGenerator;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ApplyLatePenalties extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'references:apply-penalties {--dry-run : Run without actually creating penalty references}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Apply late penalties to overdue payment references by expiring them and generating new references with penalty';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $isDryRun = $this->option('dry-run');

        if ($isDryRun) {
            $this->warn('🔍 DRY RUN MODE - Nenhuma alteração será feita');
        }

        $this->info('🚀 Iniciando aplicação de multas...');

        // Buscar referências vencidas que precisam de multa
        $overdueReferences = PaymentReference::with('student')
            ->where('status', 'pending')
            ->where('approval_status', 'approved')
            ->where('has_penalty', false)
            ->whereNotNull('due_date')
            ->where('due_date', '<', now())
            ->whereNull('original_reference_id')
            ->get();

        if ($overdueReferences->isEmpty()) {
            $this->info('✅ Nenhuma referência vencida encontrada.');
            return 0;
        }

        $this->info("📋 Encontradas {$overdueReferences->count()} referências vencidas");

        $processedCount = 0;
        $failedCount = 0;
        $skippedCount = 0;

        foreach ($overdueReferences as $reference) {
            try {
                // Verificar se já existe referência com multa gerada
                if ($reference->hasPenaltyReferenceGenerated()) {
                    $this->warn("⏭️  Ref {$reference->reference_number}: Já tem multa gerada");
                    $skippedCount++;
                    continue;
                }

                // Calcular multa
                $penaltyAmount = $reference->calculatePenalty();

                if ($penaltyAmount <= 0) {
                    $this->warn("⏭️  Ref {$reference->reference_number}: Sem configuração de multa");
                    $skippedCount++;
                    continue;
                }

                $newAmount = $reference->amount + $penaltyAmount;

                $this->line("📌 Processando: {$reference->reference_number}");
                $this->line("   Estudante: {$reference->student->name}");
                $this->line("   Valor original: MT {$reference->amount}");
                $this->line("   Multa ({$reference->penalty_rate}%): MT {$penaltyAmount}");
                $this->line("   Novo valor: MT {$newAmount}");

                if (!$isDryRun) {
                    DB::beginTransaction();

                    try {
                        // 1. Marcar referência original como expirada
                        $reference->update([
                            'status' => 'expired',
                        ]);

                        // 2. Gerar nova referência com multa
                        $newReference = $this->generatePenaltyReference($reference, $newAmount, $penaltyAmount);

                        // 3. Enviar SMS
                        $smsSent = $this->sendPenaltySms($reference, $newReference, $penaltyAmount);

                        DB::commit();

                        $this->info("   ✅ Nova ref: {$newReference->reference_number} (SMS: " . ($smsSent ? 'Enviado' : 'Falhou') . ")");
                        $processedCount++;

                    } catch (\Exception $e) {
                        DB::rollback();
                        throw $e;
                    }
                } else {
                    $this->info("   ✅ [DRY RUN] Seria criada nova referência");
                    $processedCount++;
                }

            } catch (\Exception $e) {
                $failedCount++;
                $this->error("   ❌ Erro: {$e->getMessage()}");
                Log::error('Erro ao aplicar multa', [
                    'reference_id' => $reference->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);
            }
        }

        $this->newLine();
        $this->info('📊 Resumo:');
        $this->info("   ✅ Processadas: {$processedCount}");
        $this->info("   ⏭️  Ignoradas: {$skippedCount}");
        $this->info("   ❌ Falharam: {$failedCount}");

        Log::info('Apply penalties command completed', [
            'processed' => $processedCount,
            'skipped' => $skippedCount,
            'failed' => $failedCount,
            'dry_run' => $isDryRun,
        ]);

        return 0;
    }

    /**
     * Gera nova referência com multa
     */
    private function generatePenaltyReference(PaymentReference $originalRef, float $newAmount, float $penaltyAmount): PaymentReference
    {
        // Gerar novo número de referência
        $referenceGenerator = app(ReferenceGenerator::class);
        $newReferenceNumber = $referenceGenerator->makeV1FromStudent(
            $originalRef->fee_month,
            $originalRef->student,
            $newAmount
        );

        // Nova data de validade (mais 30 dias a partir de hoje)
        $validFrom = now()->startOfDay();
        $expiresAt = now()->addDays(30)->endOfDay();

        // Descrição atualizada
        $metadata = $originalRef->metadata ?? [];
        $baseDescription = $metadata['description'] ?? 'Pagamento de Mensalidade';

        $monthName = $this->getPortugueseMonth($originalRef->fee_month);
        $metadata['description'] = $baseDescription . " (com multa de atraso)";
        $metadata['original_amount'] = $originalRef->amount;
        $metadata['penalty_amount'] = $penaltyAmount;
        $metadata['penalty_applied_at'] = now()->toDateTimeString();

        // Criar nova referência
        $newReference = PaymentReference::create([
            'student_id' => $originalRef->student_id,
            'classe' => $originalRef->classe,
            'turma' => $originalRef->turma,
            'entity_code' => $originalRef->entity_code,
            'reference_number' => $newReferenceNumber,
            'amount' => $newAmount,
            'fee_month' => $originalRef->fee_month,
            'fee_year' => $originalRef->fee_year,
            'academic_year' => $originalRef->academic_year,
            'status' => 'pending',
            'approval_status' => 'approved', // Já aprova automaticamente
            'approved_by' => 1, // System user
            'approved_at' => now(),
            'valid_from' => $validFrom,
            'payment_starts_at' => now()->startOfDay(),
            'due_date' => now()->addDays(15)->endOfDay(), // 15 dias para pagar com multa
            'expires_at' => $expiresAt,
            'penalty_type' => $originalRef->penalty_type,
            'penalty_rate' => $originalRef->penalty_rate,
            'has_penalty' => true,
            'original_reference_id' => $originalRef->id,
            'metadata' => $metadata,
        ]);

        return $newReference;
    }

    /**
     * Envia SMS informando sobre multa e nova referência
     */
    private function sendPenaltySms(PaymentReference $originalRef, PaymentReference $newRef, float $penaltyAmount): bool
    {
        try {
            if (!$originalRef->student || !$originalRef->student->mobile) {
                return false;
            }

            $monthName = $this->getPortugueseMonth($originalRef->fee_month);
            $expiresDate = $newRef->expires_at->format('d/m/Y');

            $message = sprintf(
                "COPMOZ - Referencia Vencida\n\nEstudante: %s\n\nSua referencia de %s/%s venceu.\n\n⚠️ Multa aplicada: %.2f MT\n\nNOVA REFERENCIA:\nEntidade: %s\nReferencia: %s\nValor Total: %.2f MT\nExpira: %s\n\nPague urgente via eBanking, Balcões, ATMs da RedeSIMO e pelo USSD - BCI",
                $originalRef->student->name,
                $monthName,
                $originalRef->fee_year,
                $penaltyAmount,
                $newRef->entity_code,
                $newRef->reference_number,
                $newRef->amount,
                $expiresDate
            );

            $smsService = new SmsService();
            $result = $smsService->send($originalRef->student->mobile, $message);

            if ($result['success']) {
                Log::info('Penalty SMS sent', [
                    'original_reference' => $originalRef->reference_number,
                    'new_reference' => $newRef->reference_number,
                    'student_id' => $originalRef->student_id,
                    'penalty_amount' => $penaltyAmount,
                ]);
                return true;
            }

            return false;

        } catch (\Exception $e) {
            Log::error('Error sending penalty SMS', [
                'reference_id' => $originalRef->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Converte nome do mês em português
     */
    private function getPortugueseMonth($monthName): string
    {
        $portugueseMonths = [
            'janeiro' => 'Janeiro', 'fevereiro' => 'Fevereiro', 'março' => 'Março',
            'abril' => 'Abril', 'maio' => 'Maio', 'junho' => 'Junho',
            'julho' => 'Julho', 'agosto' => 'Agosto', 'setembro' => 'Setembro',
            'outubro' => 'Outubro', 'novembro' => 'Novembro', 'dezembro' => 'Dezembro',
        ];

        $englishToPortuguese = [
            'january' => 'Janeiro', 'february' => 'Fevereiro', 'march' => 'Março',
            'april' => 'Abril', 'may' => 'Maio', 'june' => 'Junho',
            'july' => 'Julho', 'august' => 'Agosto', 'september' => 'Setembro',
            'october' => 'Outubro', 'november' => 'Novembro', 'december' => 'Dezembro',
        ];

        $monthLower = strtolower($monthName);

        if (isset($portugueseMonths[$monthLower])) {
            return $portugueseMonths[$monthLower];
        }

        if (isset($englishToPortuguese[$monthLower])) {
            return $englishToPortuguese[$monthLower];
        }

        return ucfirst($monthName);
    }
}
