<?php

namespace App\Console\Commands;

use App\Models\Exam;
use App\Models\Subject;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class CleanDuplicateExams extends Command
{
    protected $signature = 'exams:clean-duplicates
                            {--year=2025 : Year to filter}
                            {--trimester= : Trimester ID (1, 2, 3)}
                            {--subject= : Subject name (partial match)}
                            {--dry-run : Preview without deleting}';

    protected $description = 'Clean duplicate exam records, keeping the one with highest marks';

    public function handle()
    {
        $year = $this->option('year');
        $trimester = $this->option('trimester');
        $subjectName = $this->option('subject');
        $dryRun = $this->option('dry-run');

        $this->info("=== Cleaning Duplicate Exams ===");
        $this->line("Year: {$year}");

        // Build query
        $query = Exam::where('year', $year);

        if ($trimester) {
            $query->where('trimester_id', $trimester);
            $this->line("Trimester: {$trimester}");
        }

        if ($subjectName) {
            $subject = Subject::where('name', 'like', "%{$subjectName}%")->first();
            if (!$subject) {
                $this->error("Subject '{$subjectName}' not found!");
                return 1;
            }
            $query->where('subject_id', $subject->id);
            $this->line("Subject: {$subject->name}");
        }

        if ($dryRun) {
            $this->warn("DRY RUN MODE - No records will be deleted");
        }

        $this->newLine();

        // Get all exams
        $exams = $query->with(['student', 'subject'])->get();
        $this->line("Total exams found: {$exams->count()}");

        // Group by student + subject + trimester
        $groups = $exams->groupBy(function ($exam) {
            return $exam->student_id . '_' . $exam->subject_id . '_' . $exam->trimester_id;
        });

        // Filter only duplicates
        $duplicates = $groups->filter(fn($group) => $group->count() > 1);

        if ($duplicates->isEmpty()) {
            $this->info("No duplicates found!");
            return 0;
        }

        $this->line("Duplicate groups found: {$duplicates->count()}");
        $this->newLine();

        $deleted = 0;
        $kept = 0;

        foreach ($duplicates as $group) {
            // Sort by total marks (descending) - keep the one with most marks
            $sorted = $group->sortByDesc(function ($exam) {
                return floatval($exam->ACS1a) + floatval($exam->ACS2a) + floatval($exam->AT) + floatval($exam->NE);
            });

            $toKeep = $sorted->first();
            $studentName = $toKeep->student->name ?? 'Unknown';
            $subjectName = $toKeep->subject->name ?? 'Unknown';

            $this->line("Student: <fg=cyan>{$studentName}</>");
            $this->line("Subject: <fg=yellow>{$subjectName}</> | Trimester: {$toKeep->trimester_id}");

            foreach ($group as $exam) {
                $marks = "ACS1a={$exam->ACS1a}, ACS2a={$exam->ACS2a}, AT={$exam->AT}, NE={$exam->NE}";
                $total = floatval($exam->ACS1a) + floatval($exam->ACS2a) + floatval($exam->AT) + floatval($exam->NE);

                if ($exam->id === $toKeep->id) {
                    $this->line("  <fg=green>[KEEP]</> {$marks} (Total: {$total})");
                    $kept++;
                } else {
                    $this->line("  <fg=red>[DELETE]</> {$marks} (Total: {$total})");

                    if (!$dryRun) {
                        $exam->delete();
                    }
                    $deleted++;
                }
            }
            $this->line("---");
        }

        $this->newLine();
        $this->info("=== Summary ===");
        $this->line("Records kept: <fg=green>{$kept}</>");

        if ($dryRun) {
            $this->line("Records to delete: <fg=yellow>{$deleted}</>");
            $this->warn("Run without --dry-run to actually delete");
        } else {
            $this->line("Records deleted: <fg=red>{$deleted}</>");
        }

        return 0;
    }
}
