<?php

namespace App\Console\Commands;

use App\Models\PaymentReference;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ExpireOldReferences extends Command
{
    /**
     * The name and signature of the console command.
     *
     * --dry-run: Simula sem actualizar referências
     * --days: Dias de tolerância após expiração (default: 0)
     */
    protected $signature = 'references:expire
                            {--dry-run : Simular sem actualizar referências}
                            {--days=0 : Dias de tolerância após data de expiração}';

    protected $description = 'Marca referências pendentes como expiradas baseado na data de vencimento';

    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        $toleranceDays = (int) $this->option('days');

        $this->info('=== Expiração de Referências ===');
        $this->info('Data/Hora: ' . now()->format('d/m/Y H:i:s'));

        if ($isDryRun) {
            $this->warn('MODO SIMULAÇÃO - Nenhuma referência será actualizada');
        }

        if ($toleranceDays > 0) {
            $this->info("Tolerância: {$toleranceDays} dia(s) após expiração");
        }

        $this->newLine();

        // Buscar referências pendentes que já expiraram
        $cutoffDate = now()->subDays($toleranceDays);

        $expiredReferences = PaymentReference::where('status', 'pending')
            ->whereNotNull('expires_at')
            ->where('expires_at', '<', $cutoffDate)
            ->get();

        if ($expiredReferences->isEmpty()) {
            $this->info('Nenhuma referência para expirar.');
            return 0;
        }

        $this->info("Encontradas {$expiredReferences->count()} referência(s) expirada(s)");
        $this->newLine();

        $this->table(
            ['ID', 'Estudante', 'Mês/Ano', 'Valor', 'Expirou em', 'Dias atraso'],
            $expiredReferences->map(function ($ref) {
                $student = $ref->student;
                $daysOverdue = Carbon::parse($ref->expires_at)->diffInDays(now());
                return [
                    $ref->id,
                    $student ? $student->name : 'N/A',
                    "{$ref->fee_month}/{$ref->fee_year}",
                    number_format($ref->amount, 2) . ' MT',
                    Carbon::parse($ref->expires_at)->format('d/m/Y'),
                    $daysOverdue
                ];
            })
        );

        if ($isDryRun) {
            $this->newLine();
            $this->warn('Modo simulação - nenhuma alteração feita.');
            return 0;
        }

        // Confirmar antes de prosseguir (apenas em modo interactivo)
        if (!$this->option('no-interaction')) {
            if (!$this->confirm('Deseja marcar estas referências como expiradas?', true)) {
                $this->info('Operação cancelada.');
                return 0;
            }
        }

        $updated = 0;
        $errors = 0;

        foreach ($expiredReferences as $reference) {
            try {
                $reference->update([
                    'status' => 'expired',
                    'metadata' => array_merge($reference->metadata ?? [], [
                        'expired_at' => now()->toDateTimeString(),
                        'expired_by_system' => true,
                        'days_overdue' => Carbon::parse($reference->expires_at)->diffInDays(now())
                    ])
                ]);

                $updated++;

                Log::info('Reference expired by system', [
                    'reference_id' => $reference->id,
                    'student_id' => $reference->student_id,
                    'expired_at' => $reference->expires_at,
                    'amount' => $reference->amount
                ]);

            } catch (\Exception $e) {
                $errors++;
                $this->error("Erro ao expirar referência {$reference->id}: {$e->getMessage()}");
                Log::error('Failed to expire reference', [
                    'reference_id' => $reference->id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        $this->newLine();
        $this->info('=== Resumo ===');
        $this->line("Referências expiradas: {$updated}");
        $this->line("Erros: {$errors}");

        return 0;
    }
}
