<?php

namespace App\Console\Commands;

use App\Models\FeeStructure;
use App\Models\PaymentReference;
use App\Models\User;
use App\Services\ReferenceGenerator;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class GenerateMonthlyReferences extends Command
{
    /**
     * The name and signature of the console command.
     *
     * --dry-run: Simula sem criar referências
     * --month: Mês específico para gerar referências (ex: Janeiro, Fevereiro)
     * --year: Ano específico
     *
     * IMPORTANTE: Este comando gera referências para o MÊS SEGUINTE
     * Exemplo: Executado no dia 25 de Dezembro, gera referências de Janeiro
     */
    protected $signature = 'references:generate
                            {--dry-run : Simular sem criar referências}
                            {--month= : Mês para gerar referências (ex: Janeiro)}
                            {--year= : Ano específico}';

    protected $description = 'Gera referências de pagamento automaticamente para o MÊS SEGUINTE, baseado nas FeeStructures configuradas';

    /**
     * Turmas excluídas da geração automática
     * Estudantes nestas turmas não recebem referências via cron job
     * Usa LIKE para capturar variações (ex: "Finalistas 2024", "FINALISTAS 2025")
     */
    private array $excludedClassroomPatterns = [
        'finalista',           // Finalistas 2024, FINALISTAS 2025, etc.
        'transferido',         // TRANSFERIDOS, Transferidos, etc.
        'não matriculado',     // Não matriculados
        'nao matriculado',     // Variação sem acento
        'recem matriculad',    // Recem matriculados, Recem matriculada
        'recém matriculad',    // Com acento
        'administração',       // Administração Escolar
        'administracao',       // Variação sem acento
    ];

    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        $specificMonth = $this->option('month');
        $specificYear = $this->option('year');

        $this->info('=== Geração Automática de Referências ===');
        $this->info('Data/Hora: ' . now()->format('d/m/Y H:i:s'));

        if ($isDryRun) {
            $this->warn('MODO SIMULAÇÃO - Nenhuma referência será criada');
        }

        // Meses em português
        $monthsPortuguese = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];

        // LÓGICA PRINCIPAL:
        // No dia payment_start_day (ex: 25), geramos referências para o MÊS SEGUINTE
        // Exemplo: Dia 25 de Dezembro -> Gera referências de Janeiro do ano seguinte

        if ($specificMonth) {
            // Se especificou mês manualmente, usar esse mês
            $targetMonth = $specificMonth;
            $targetYear = (int) ($specificYear ?? now()->year);
        } else {
            // Calcular mês seguinte automaticamente
            $nextMonth = now()->addMonth();
            $targetMonth = $monthsPortuguese[$nextMonth->month];
            $targetYear = $nextMonth->year;

            // Se ano foi especificado manualmente, respeitar
            if ($specificYear) {
                $targetYear = (int) $specificYear;
            }
        }

        $today = now()->day;

        $this->info("Mês alvo: {$targetMonth} / Ano: {$targetYear}");
        $this->info("Dia actual: {$today}");
        $this->newLine();

        // Buscar FeeStructures activas que incluem o mês alvo
        // E que o dia actual corresponde ao payment_start_day (dia de geração)
        $feeStructures = FeeStructure::where('active', 1)
            ->where('academic_year', $targetYear)
            ->get()
            ->filter(function ($fs) use ($targetMonth, $today, $specificMonth) {
                // Verificar se o mês está incluído na FeeStructure
                if (empty($fs->months)) {
                    return false;
                }
                $months = array_map('trim', explode(',', $fs->months));
                if (!in_array($targetMonth, $months)) {
                    return false;
                }

                // Se foi especificado um mês manualmente, ignorar a verificação do dia
                if ($specificMonth) {
                    return true;
                }

                // Verificar se hoje é o dia de geração configurado
                $paymentStartDay = $fs->payment_start_day ?? 5;
                return $today == $paymentStartDay;
            });

        if ($feeStructures->isEmpty()) {
            $this->warn("Nenhuma FeeStructure encontrada para {$targetMonth}/{$targetYear}");
            if (!$specificMonth) {
                $this->line("(Dica: Hoje é dia {$today}. Verifique se payment_start_day corresponde.)");
            }
            return 0;
        }

        $this->info("Encontradas {$feeStructures->count()} FeeStructure(s) para processar");
        $this->newLine();

        // Mostrar turmas excluídas
        $this->warn('Turmas EXCLUÍDAS (Finalistas, Transferidos, Não Matriculados):');
        $this->line('   - ' . implode(', ', $this->excludedClassroomPatterns));
        $this->newLine();

        $totalGenerated = 0;
        $totalSkipped = 0;
        $totalExcluded = 0;
        $totalErrors = 0;

        foreach ($feeStructures as $feeStructure) {
            $this->info("--- {$feeStructure->fee_name} (ID: {$feeStructure->id}) ---");
            $this->line("   Valor: {$feeStructure->monthly_fee} MT");
            $this->line("   Classes: {$feeStructure->grades}");
            $this->line("   Dia Geração: {$feeStructure->payment_start_day}");
            $this->line("   Dia Vencimento: {$feeStructure->payment_due_day}");

            // Buscar estudantes das classes aplicáveis
            $grades = array_map('trim', explode(',', $feeStructure->grades));

            // A classe do estudante está na tabela classrooms (campo 'class')
            // EXCLUIR turmas de Finalistas, Transferidos e Não Matriculados
            $students = User::where('is_active', 1)
                ->whereHas('classroom', function ($query) use ($grades) {
                    $query->whereIn('class', $grades)
                        ->where('is_active', 1); // Apenas turmas activas
                })
                ->with('classroom') // Eager load para verificar nome da turma
                ->get();

            // Filtrar estudantes excluídos
            $eligibleStudents = $students->filter(function ($student) use (&$totalExcluded) {
                if (!$student->classroom) {
                    return false;
                }

                $classroomName = strtolower($student->classroom->name);

                foreach ($this->excludedClassroomPatterns as $pattern) {
                    if (str_contains($classroomName, $pattern)) {
                        $totalExcluded++;
                        return false; // Excluir este estudante
                    }
                }

                return true; // Incluir este estudante
            });

            $this->line("   Estudantes encontrados: {$students->count()}");
            $this->line("   Estudantes elegíveis: {$eligibleStudents->count()}");
            $this->line("   Excluídos (Finalistas/Transf./Não Matr.): " . ($students->count() - $eligibleStudents->count()));

            foreach ($eligibleStudents as $student) {
                // Verificar se já existe referência para este estudante/mês/taxa
                $existingRef = PaymentReference::where('student_id', $student->id)
                    ->where('fee_month', $targetMonth)
                    ->where('fee_year', $targetYear)
                    ->where('status', '!=', 'paid')
                    ->where(function($q) use ($feeStructure) {
                        $q->whereJsonContains('metadata->fee_structure_id', $feeStructure->id)
                          ->orWhereJsonContains('metadata->fee_structure_id', (string)$feeStructure->id);
                    })
                    ->first();

                if ($existingRef) {
                    $totalSkipped++;
                    continue; // Já tem referência
                }

                if ($isDryRun) {
                    $this->line("   [SIMULAÇÃO] Geraria referência para: {$student->name} ({$student->classroom->name})");
                    $totalGenerated++;
                    continue;
                }

                try {
                    // Gerar referência
                    $reference = $this->generateReference($student, $feeStructure, $targetMonth, $targetYear);
                    $totalGenerated++;

                    Log::info('Auto-generated reference', [
                        'student_id' => $student->id,
                        'student_name' => $student->name,
                        'classroom' => $student->classroom->name ?? 'N/A',
                        'fee_structure_id' => $feeStructure->id,
                        'reference' => $reference->reference_number,
                        'amount' => $reference->amount,
                        'target_month' => $targetMonth,
                        'target_year' => $targetYear
                    ]);

                } catch (\Exception $e) {
                    $totalErrors++;
                    $this->error("   Erro para {$student->name}: {$e->getMessage()}");
                    Log::error('Auto-generate reference failed', [
                        'student_id' => $student->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            $this->newLine();
        }

        $this->newLine();
        $this->info('=== Resumo ===');
        $this->line("Mês/Ano gerado: {$targetMonth}/{$targetYear}");
        $this->line("Referências geradas: {$totalGenerated}");
        $this->line("Referências ignoradas (já existiam): {$totalSkipped}");
        $this->line("Estudantes excluídos (Finalistas/Transf./Não Matr.): {$totalExcluded}");
        $this->line("Erros: {$totalErrors}");

        return 0;
    }

    private function generateReference(User $student, FeeStructure $feeStructure, string $month, int $year): PaymentReference
    {
        $referenceGenerator = new ReferenceGenerator();

        // Calcular data de vencimento (no mês ALVO, não no mês actual)
        $monthNumber = $this->getMonthNumber($month);
        $dueDay = $feeStructure->payment_due_day ?? 25;
        $lastDayOfMonth = Carbon::create($year, $monthNumber, 1)->endOfMonth()->day;
        $expiresAt = Carbon::create($year, $monthNumber, min($dueDay, $lastDayOfMonth), 23, 59, 59);

        // Para referências geradas antecipadamente, não há multa
        // A multa só seria aplicada se estivéssemos após a data de vencimento
        $baseAmount = (float) $feeStructure->monthly_fee;
        $fineAmount = 0;

        // Só calcular multa se a data de vencimento já passou
        if ($expiresAt->isPast()) {
            $penaltyPercentage = (float) ($feeStructure->late_penalty_percentage ?? 0);
            $fineAmount = round($baseAmount * ($penaltyPercentage / 100), 2);
        }

        $totalAmount = $baseAmount + $fineAmount;

        // Gerar número de referência usando formato V1 (11 dígitos) - compatível com BCI
        $entity = config('payments.entity', '90013');
        $reference = $referenceGenerator->makeV1FromStudent($month, $student, $totalAmount);

        // Criar referência
        $paymentReference = PaymentReference::create([
            'student_id' => $student->id,
            'entity_code' => $entity,
            'reference_number' => $reference,
            'amount' => $totalAmount,
            'fine_amount' => $fineAmount,
            'fee_month' => $month,
            'fee_year' => $year,
            'expires_at' => $expiresAt,
            'status' => 'pending',
            'metadata' => [
                'fee_structure_id' => $feeStructure->id,
                'fee_name' => $feeStructure->fee_name,
                'description' => $feeStructure->fee_name,
                'base_amount' => $baseAmount,
                'fine_amount' => $fineAmount,
                'auto_generated' => true,
                'generated_at' => now()->toDateTimeString(),
                'generated_for_next_month' => true
            ]
        ]);

        return $paymentReference;
    }

    private function getMonthNumber(string $monthName): int
    {
        $months = [
            'janeiro' => 1, 'fevereiro' => 2, 'março' => 3, 'abril' => 4,
            'maio' => 5, 'junho' => 6, 'julho' => 7, 'agosto' => 8,
            'setembro' => 9, 'outubro' => 10, 'novembro' => 11, 'dezembro' => 12
        ];
        return $months[strtolower($monthName)] ?? 1;
    }
}
