<?php

namespace App\Console\Commands;

use App\Models\Exam;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ImprovedCleanupDuplicateExams extends Command
{
    protected $signature = 'exams:cleanup-smart {--year=} {--student-id=} {--subject-id=} {--trimester-id=} {--dry-run} {--force} {--keep-strategy=complete}';
    protected $description = 'Smart cleanup of duplicate exam records, keeping the most complete or latest record';

    public function handle()
    {
        $year = $this->option('year') ?: date('Y');
        $studentId = $this->option('student-id');
        $subjectId = $this->option('subject-id');
        $trimesterId = $this->option('trimester-id');
        $dryRun = $this->option('dry-run');
        $force = $this->option('force');
        $keepStrategy = $this->option('keep-strategy'); // 'complete', 'latest', or 'ask'

        $this->info("Smart cleanup of duplicate exam records for year: {$year}");
        $this->line("Keep strategy: <fg=cyan>{$keepStrategy}</>");

        // Display filters
        if ($studentId) $this->line("Filter by student ID: <fg=yellow>{$studentId}</>");
        if ($subjectId) $this->line("Filter by subject ID: <fg=yellow>{$subjectId}</>");
        if ($trimesterId) $this->line("Filter by trimester ID: <fg=yellow>{$trimesterId}</>");

        if ($dryRun) {
            $this->warn("DRY RUN MODE - No records will be deleted");
        }

        // Debug: Show raw count first
        $rawCount = DB::table('exams')->where('year', $year)->whereNull('deleted_at')->count();
        $this->line("Debug: Raw exam count in DB for {$year}: <fg=cyan>{$rawCount}</>");

        if (!$force && !$dryRun) {
            if (!$this->confirm('This will permanently delete duplicate exam records. Continue?')) {
                $this->info('Operation cancelled.');
                return 0;
            }
        }

        // Build query with filters - SoftDeletes trait already filters deleted_at
        $query = Exam::where('year', $year)
            ->with(['student', 'subject']);

        if ($studentId) {
            $query->where('student_id', $studentId);
        }

        if ($subjectId) {
            $query->where('subject_id', $subjectId);
        }

        if ($trimesterId) {
            $query->where('trimester_id', $trimesterId);
        }

        $exams = $query->get();

        if ($exams->isEmpty()) {
            $this->warn('No exam records found matching the specified criteria.');

            // Debug: show raw count
            $rawCount = DB::table('exams')->where('year', $year)->whereNull('deleted_at')->count();
            $this->line("Raw count in database: {$rawCount}");

            return 0;
        }

        $this->line("Found <fg=cyan>{$exams->count()}</> exam records to analyze.");

        // Group by student + subject + trimester + year (ignore classroom)
        $duplicateGroups = $exams->groupBy(function ($exam) {
            return $exam->student_id . '_' . $exam->subject_id . '_' . $exam->trimester_id . '_' . $exam->year;
        });

        // Count actual duplicates
        $duplicateCount = $duplicateGroups->filter(fn($group) => $group->count() > 1)->count();

        if ($duplicateCount === 0) {
            $this->info("✓ No duplicate exam records found!");
            return 0;
        }

        $this->line("Found <fg=yellow>{$duplicateCount}</> duplicate groups to process.");
        $this->newLine();

        $cleanedCount = 0;
        $reviewCount = 0;
        $totalGroups = 0;

        foreach ($duplicateGroups as $groupKey => $group) {
            if ($group->count() > 1) {
                $totalGroups++;

                $result = $this->processGroup($group, $keepStrategy, $dryRun);

                if ($result['action'] == 'cleaned') {
                    $cleanedCount += $result['deleted_count'];
                } elseif ($result['action'] == 'review') {
                    $reviewCount++;
                }
            }
        }

        $this->displayResults($totalGroups, $cleanedCount, $reviewCount, $dryRun);

        return 0;
    }

    private function processGroup($group, $keepStrategy, $dryRun)
    {
        $firstExam = $group->first();
        $groupInfo = "{$firstExam->student->name} | {$firstExam->subject->name} | T{$firstExam->trimester_id} | {$firstExam->year}";

        // Analyze each record
        $recordsAnalysis = [];
        foreach ($group as $exam) {
            $markCount = 0;
            $marks = [];

            if ($exam->ACS1a > 0) { $markCount++; $marks[] = "ACS1a:{$exam->ACS1a}"; }
            if ($exam->ACS2a > 0) { $markCount++; $marks[] = "ACS2a:{$exam->ACS2a}"; }
            if ($exam->AT > 0) { $markCount++; $marks[] = "AT:{$exam->AT}"; }
            if ($exam->NE > 0) { $markCount++; $marks[] = "NE:{$exam->NE}"; }

            $recordsAnalysis[] = [
                'exam' => $exam,
                'mark_count' => $markCount,
                'marks_detail' => implode(', ', $marks),
                'score' => $this->calculateScore($exam, $markCount)
            ];
        }

        // Determine which record to keep
        $recordToKeep = $this->selectRecordToKeep($recordsAnalysis, $keepStrategy);

        if (!$recordToKeep) {
            $this->warn("❓ Manual review needed: {$groupInfo}");
            $this->displayRecordDetails($recordsAnalysis);
            return ['action' => 'review', 'deleted_count' => 0];
        }

        // Delete other records
        $recordsToDelete = collect($recordsAnalysis)->reject(function($record) use ($recordToKeep) {
            return $record['exam']->id === $recordToKeep['exam']->id;
        });

        $this->line("🔍 Processing: <fg=cyan>{$groupInfo}</>");
        $this->line("   Keeping: ID {$recordToKeep['exam']->id} (Classroom: {$recordToKeep['exam']->classroom_id}, Marks: {$recordToKeep['mark_count']}, Score: {$recordToKeep['score']})");

        $deletedCount = 0;
        foreach ($recordsToDelete as $record) {
            $this->line("   Deleting: ID {$record['exam']->id} (Classroom: {$record['exam']->classroom_id}, Marks: {$record['mark_count']}, Score: {$record['score']})");

            if (!$dryRun) {
                $record['exam']->delete();
            }
            $deletedCount++;
        }

        return ['action' => 'cleaned', 'deleted_count' => $deletedCount];
    }

    private function calculateScore($exam, $markCount)
    {
        // Create a score that prioritizes:
        // 1. Number of marks (most important)
        // 2. Recency (updated_at)
        // 3. Creation time

        $score = $markCount * 1000; // Mark count is most important

        // Add points for recency (max 100 points)
        $daysSinceUpdate = now()->diffInDays($exam->updated_at);
        $recencyScore = max(0, 100 - $daysSinceUpdate);
        $score += $recencyScore;

        // Add small points for creation time (max 10 points)
        $daysSinceCreation = now()->diffInDays($exam->created_at);
        $creationScore = max(0, 10 - ($daysSinceCreation / 10));
        $score += $creationScore;

        return round($score, 2);
    }

    private function selectRecordToKeep($recordsAnalysis, $keepStrategy)
    {
        // Sort by score (highest first)
        usort($recordsAnalysis, function($a, $b) {
            return $b['score'] <=> $a['score'];
        });

        $topRecords = collect($recordsAnalysis)->where('score', $recordsAnalysis[0]['score']);

        if ($topRecords->count() == 1) {
            return $recordsAnalysis[0];
        }

        // Handle ties based on strategy
        switch ($keepStrategy) {
            case 'complete':
                // Already handled by score calculation
                return $recordsAnalysis[0];

            case 'latest':
                return $topRecords->sortByDesc(function($record) {
                    return $record['exam']->updated_at;
                })->first();

            case 'ask':
                // For manual review
                return null;

            default:
                return $recordsAnalysis[0];
        }
    }

    private function displayRecordDetails($recordsAnalysis)
    {
        $headers = ['ID', 'Classroom', 'Marks', 'Details', 'Score', 'Updated'];
        $rows = [];

        foreach ($recordsAnalysis as $record) {
            $rows[] = [
                $record['exam']->id,
                $record['exam']->classroom_id,
                $record['mark_count'],
                $record['marks_detail'] ?: 'No marks',
                $record['score'],
                $record['exam']->updated_at->format('Y-m-d H:i')
            ];
        }

        $this->table($headers, $rows);
    }

    private function displayResults($totalGroups, $cleanedCount, $reviewCount, $dryRun)
    {
        $this->newLine();
        $this->line("<fg=green>━━━ CLEANUP RESULTS ━━━</>");
        $this->line("Total duplicate groups found: <fg=yellow>{$totalGroups}</>");

        if ($dryRun) {
            $this->line("Records that would be deleted: <fg=red>{$cleanedCount}</>");
        } else {
            $this->line("Records deleted: <fg=red>{$cleanedCount}</>");
        }

        if ($reviewCount > 0) {
            $this->line("Groups needing manual review: <fg=yellow>{$reviewCount}</>");
        }

        $this->newLine();

        if ($reviewCount > 0) {
            $this->line("<fg=blue>Next Steps:</>");
            $this->line("• Review the groups marked for manual review above");
            $this->line("• Use --keep-strategy=ask for interactive cleanup");
            $this->line("• Use specific filters to handle problematic groups individually");
        }

        if ($dryRun) {
            $this->line("<fg=cyan>Run without --dry-run to perform actual cleanup</>");
        }
    }
}
