<?php

namespace App\Http\Livewire\Admin\Admins;

use App\Models\Admin;
use App\Traits\SysPermissions;
use App\Services\SmsService;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;
use Exception;

class EditAdmin extends Component
{
    use FindGuard, SysPermissions, LivewireAlert;

    public $user;
    public $name, $role;
    public $phone_number;
    public $email, $is_active;
    public $is_dormant;
    public $validatedData;
    public $access = [];
    public $generateNewPassword = false;

    protected $listeners = [
        'confirmed',
        'cancelled',
        'confirmedPasswordGeneration'
    ];

    public function mount(string $slug)
    {
        $this->user = Admin::query()->firstWhere('slug', $slug);
        if (!$this->user) {
            return redirect()->route('admin.list.admins');
        } else {
            $this->name = $this->user->name;
            $this->phone_number = $this->user->phone_number;
            $this->email = $this->user->email;
            $this->role = $this->user->role;
            $this->is_active = $this->user->is_active;
        }
        foreach ($this->user->getPermissionNames() as $permissionName) {
            $this->access[] = $permissionName;
        }

    }

    protected function rules(): array
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'role' => ['required', 'integer', 'min:0', 'max:6'],
            'email' => ['required', 'string', 'email', 'max:255'],
            'phone_number' => ['required', 'numeric'],
            'is_active' => ['required'],
        ];
    }

    /**
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    /**
     * update user password here
     */
    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function linkOrRemovePrivilege(string $privilege)
    {
        // check if it exists in array
        if (in_array($privilege, $this->access)) {
            unset($this->access[array_search($privilege, $this->access)]);
            if ($this->user->hasPermissionTo($privilege)) {
                $this->user->revokePermissionTo($privilege);
                $this->user->refresh();
                $this->emit('noteAdded');
                $this->alert('warning', 'Privilégio revogado ' . $privilege);
            }
        } else {
            $this->access[] = $privilege;
            $this->user->givePermissionTo($privilege);
            $this->user->refresh();
            $this->emit('noteAdded');
            $this->alert('info', 'Privilégio concedido ' . $privilege);
        }
    }

    public function confirmed()
    {
        $this->user->fill($this->validatedData);
        if ($this->user->isClean() && !$this->generateNewPassword) {
            $this->alert('warning', 'Pelo menos um valor deve mudar.');
            return redirect()->back();
        }

        // Preparar dados para atualização
        $updateData = [
            'name' => $this->name,
            'email' => $this->email,
            'role' => $this->role,
            'phone_number' => $this->phone_number,
            'is_active' => $this->is_active,
        ];

        // Apenas gerar nova senha se a opção estiver ativada
        if ($this->generateNewPassword) {
            $password = rand(100000, 500000);
            $updateData['password'] = bcrypt($password);

            // Enviar SMS com nova senha
            try {
                $callbackUrl = route('admin.login');
                $message = "Olá " . $this->name . ", a sua senha foi alterada no sistema COPMOZ.\n\nEmail: " . $this->email . "\nNova Senha: " . $password . "\n\nLink para login: " . $callbackUrl;

                $smsService = new SmsService();
                $result = $smsService->send($this->phone_number, $message);

                if (!$result['success']) {
                    $this->alert('warning', 'Dados atualizados, mas houve erro ao enviar SMS');
                }
            } catch (Exception $e) {
                $this->alert('warning', 'Dados atualizados, mas houve erro ao enviar SMS: ' . $e->getMessage());
            }
        }

        Admin::query()->where('id', $this->user->id)->update($updateData);

        Note::createSystemNotification(Admin::class, 'Atualização da conta Admin', 'Você atualizou a conta com sucesso ' . $this->name);
        $this->emit('noteAdded');
        $this->alert('success', 'Atualizado com sucesso Admin ' . $this->name);
        return redirect()->route('admin.list.admins');
    }

    /**
     * Gerar e enviar nova senha imediatamente
     */
    public function generateAndSendNewPassword()
    {
        $this->confirm('Tem certeza de que deseja gerar uma nova senha para este staff?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, gerar nova senha!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmedPasswordGeneration',
            'onDismissed' => 'cancelled'
        ]);
    }

    /**
     * Confirmação para gerar nova senha
     */
    public function confirmedPasswordGeneration()
    {
        try {
            $password = rand(100000, 500000);

            Admin::query()->where('id', $this->user->id)->update([
                'password' => bcrypt($password),
            ]);

            // Enviar SMS com nova senha
            try {
                $callbackUrl = route('admin.login');
                $message = "Olá " . $this->name . ", a sua senha foi alterada no sistema COPMOZ.\n\nEmail: " . $this->email . "\nNova Senha: " . $password . "\n\nLink para login: " . $callbackUrl;

                $smsService = new SmsService();
                $result = $smsService->send($this->phone_number, $message);

                if ($result['success']) {
                    $this->alert('success', 'Nova senha gerada e enviada por SMS com sucesso!');
                } else {
                    $this->alert('warning', 'Nova senha gerada, mas houve erro ao enviar SMS');
                }
            } catch (Exception $e) {
                $this->alert('warning', 'Nova senha gerada, mas houve erro ao enviar SMS: ' . $e->getMessage());
            }

            Note::createSystemNotification(Admin::class, 'Senha Alterada', 'Nova senha gerada para ' . $this->name);
            $this->emit('noteAdded');

        } catch (Exception $e) {
            $this->alert('error', 'Erro ao gerar nova senha: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.admins.edit-admin', [
            'permissions' => $this->permissions()
        ]);
    }
}
