<?php

namespace App\Http\Livewire\Admin\Documents;

use App\Models\Classroom;
use App\Models\User;
use App\Services\StudentCardService;
use Livewire\Component;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ManageStudentCards extends Component
{
    use WithPagination, LivewireAlert;

    protected $paginationTheme = 'bootstrap';

    // Tab control
    public $activeTab = 'generate'; // generate, list

    // Filters for generation
    public $selectedClass = '';
    public $selectedTurma = null;
    public $academicYear;
    public $availableTurmas = [];
    public $availableClasses = [];

    // Students data
    public $studentsWithCards = [];
    public $studentsWithoutCards = [];
    public $showStudentsList = false;

    // Individual generation
    public $selectedStudentId = null;

    // Search/filter for listing
    public $searchTerm = '';

    // Processing state
    public $isProcessing = false;
    public $processingProgress = 0;
    public $processingTotal = 0;

    // Statistics
    public $stats = [];

    // Listeners
    protected $listeners = [
        'confirmGenerateCard',
        'confirmGenerateBatch',
        'refreshComponent' => '$refresh',
    ];

    public function mount()
    {
        $this->academicYear = (int) date('Y');
        $this->loadAvailableClasses();
        $this->loadStats();
    }

    public function loadAvailableClasses()
    {
        // Get all distinct class levels
        $this->availableClasses = Classroom::where('is_active', 1)
            ->where('name', 'NOT LIKE', '%Recem%')
            ->where('name', 'NOT LIKE', '%Recém%')
            ->where('name', 'NOT LIKE', '%Não Matriculado%')
            ->where('name', 'NOT LIKE', '%Finalista%')
            ->select('class')
            ->distinct()
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->pluck('class')
            ->toArray();
    }

    public function loadTurmas()
    {
        $query = Classroom::where('is_active', 1)
            ->where('name', 'NOT LIKE', '%Recem%')
            ->where('name', 'NOT LIKE', '%Recém%')
            ->where('name', 'NOT LIKE', '%Não Matriculado%')
            ->where('name', 'NOT LIKE', '%Finalista%');

        if ($this->selectedClass) {
            $query->where('class', $this->selectedClass);
        }

        $this->availableTurmas = $query->orderByRaw('CAST(class AS UNSIGNED)')->orderBy('name')->get();
    }

    public function updatedSelectedClass()
    {
        $this->selectedTurma = null;
        $this->showStudentsList = false;
        $this->studentsWithCards = [];
        $this->studentsWithoutCards = [];
        $this->loadTurmas();
    }

    public function updatedSelectedTurma()
    {
        $this->showStudentsList = false;
        $this->studentsWithCards = [];
        $this->studentsWithoutCards = [];
    }

    public function loadStats()
    {
        $service = app(StudentCardService::class);
        $this->stats = $service->getStatistics($this->academicYear);
    }

    public function checkStudents()
    {
        if (!$this->selectedTurma) {
            $this->alert('warning', 'Seleccione uma turma');
            return;
        }

        $service = app(StudentCardService::class);
        $result = $service->getStudentsWithoutCards($this->selectedTurma);

        $this->studentsWithCards = $result['with_cards'];
        $this->studentsWithoutCards = $result['without_cards'];
        $this->showStudentsList = true;
    }

    public function generateIndividualCard($studentId)
    {
        try {
            $student = User::findOrFail($studentId);
            $service = app(StudentCardService::class);

            $path = $service->generateCard($student);

            $this->alert('success', 'Cartão gerado com sucesso!');
            $this->loadStats();
            $this->checkStudents(); // Refresh list

            // Open in new window via JS
            $url = Storage::disk('public')->url($path);
            $this->emit('openUrl', $url);
        } catch (\Exception $e) {
            Log::error('Erro ao gerar cartão', ['student_id' => $studentId, 'error' => $e->getMessage()]);
            $this->alert('error', 'Erro: ' . $e->getMessage());
        }
    }

    public function downloadCard($studentId)
    {
        try {
            $student = User::findOrFail($studentId);
            $year = date('Y');
            $fileName = Str::slug($student->name) . '_cartao_' . $student->student_id . '.pdf';
            $filePath = "student_cards/{$year}/{$fileName}";

            if (!Storage::disk('public')->exists($filePath)) {
                // Generate if doesn't exist
                $service = app(StudentCardService::class);
                $filePath = $service->generateCard($student);
            }

            // Open in new window via JS
            $url = Storage::disk('public')->url($filePath);
            $this->emit('openUrl', $url);
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao baixar cartão: ' . $e->getMessage());
        }
    }

    public function requestBatchGeneration()
    {
        if (!$this->selectedTurma) {
            $this->alert('warning', 'Seleccione uma turma');
            return;
        }

        $count = count($this->studentsWithoutCards);
        if ($count === 0) {
            $this->alert('info', 'Todos os estudantes desta turma já têm cartão');
            return;
        }

        $classroom = Classroom::find($this->selectedTurma);

        $this->alert('question', "Gerar {$count} cartões para a turma {$classroom->name}?", [
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, gerar!',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmGenerateBatch',
        ]);
    }

    public function confirmGenerateBatch()
    {
        if (!$this->selectedTurma) {
            return;
        }

        try {
            $this->isProcessing = true;
            $service = app(StudentCardService::class);

            $result = $service->generateCardsForClassroom($this->selectedTurma);

            $successCount = count($result['success']);
            $failedCount = count($result['failed']);

            $this->isProcessing = false;
            $this->loadStats();
            $this->checkStudents();

            if ($failedCount > 0) {
                $this->alert('warning', "Gerados {$successCount} cartões. {$failedCount} falharam.");
            } else {
                $this->alert('success', "{$successCount} cartões gerados com sucesso!");
            }
        } catch (\Exception $e) {
            $this->isProcessing = false;
            Log::error('Erro ao gerar cartões em lote', ['error' => $e->getMessage()]);
            $this->alert('error', 'Erro: ' . $e->getMessage());
        }
    }

    public function downloadBatchZip()
    {
        if (!$this->selectedTurma) {
            $this->alert('warning', 'Seleccione uma turma');
            return;
        }

        try {
            $service = app(StudentCardService::class);
            $zipPath = $service->generateClassroomZip($this->selectedTurma);

            $url = Storage::disk('public')->url($zipPath);
            $this->emit('openUrl', $url);
        } catch (\Exception $e) {
            $this->alert('error', 'Erro ao gerar ZIP: ' . $e->getMessage());
        }
    }

    public function regenerateCard($studentId)
    {
        try {
            $student = User::findOrFail($studentId);
            $service = app(StudentCardService::class);

            // Delete existing card
            $year = date('Y');
            $fileName = Str::slug($student->name) . '_cartao_' . $student->student_id . '.pdf';
            $filePath = "student_cards/{$year}/{$fileName}";
            if (Storage::disk('public')->exists($filePath)) {
                Storage::disk('public')->delete($filePath);
            }

            // Generate new
            $path = $service->generateCard($student);

            $this->alert('success', 'Cartão regenerado com sucesso!');
            $this->loadStats();

            // Only refresh students list if a turma is selected
            if ($this->selectedTurma) {
                $this->checkStudents();
            }

            $url = Storage::disk('public')->url($path);
            $this->emit('openUrl', $url);
        } catch (\Exception $e) {
            $this->alert('error', 'Erro: ' . $e->getMessage());
        }
    }

    public function render()
    {
        // Get paginated list of all cards for the list tab
        $cardsQuery = User::whereHas('classroom', function ($q) {
            $q->where('is_active', 1);
        })->where('is_active', 1);

        if ($this->searchTerm) {
            $cardsQuery->where(function ($q) {
                $q->where('name', 'like', "%{$this->searchTerm}%")
                  ->orWhere('student_id', 'like', "%{$this->searchTerm}%");
            });
        }

        $students = $cardsQuery->orderBy('name')->paginate(20);

        // Check which students have cards (use through() to preserve pagination)
        $year = date('Y');
        $students->through(function ($student) use ($year) {
            $fileName = Str::slug($student->name) . '_cartao_' . $student->student_id . '.pdf';
            $filePath = "student_cards/{$year}/{$fileName}";
            $student->has_card = Storage::disk('public')->exists($filePath);
            $student->card_path = $filePath;
            return $student;
        });

        return view('livewire.admin.documents.manage-student-cards', [
            'students' => $students,
        ]);
    }
}
