<?php

namespace App\Http\Livewire\Admin\Downloadcenter;

use App\Models\Downloadcenter;
use App\Models\Downloadsetings;
use App\Models\User;
use App\Models\Classroom;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithPagination;
use Note\Note;
use Auth;
use DB;

class Downloads extends Component
{
    use WithPagination, FindGuard, LivewireAlert;

    public $search = '', $slug;
    public $filter = '';
    public $selectedIds = []; // Selected IDs
    public $selectAll = false; // Select All state
    public bool $readyToLoad = false;
    public $selectedClass = '';
    public $selectedTurma = '';
    public $turmas = [];

    // Bulk delete properties
    public $showBulkDeleteModal = false;
    public $bulkDeleteType = '';
    public $bulkDeleteFromDate = '';
    public $bulkDeleteToDate = '';
    public $bulkDeleteSelectedClass = '';
    public $bulkDeleteSelectedTurma = '';

    protected $queryString = ['search', 'filter', 'selectedClass', 'selectedTurma'];
    protected string $paginationTheme = 'bootstrap';

    public function exportPDF()
    {
        return redirect()->route('admin.matrizes.export.pdf', [
            'class' => $this->selectedClass,
            'turma' => $this->selectedTurma,
            'search' => $this->search,
            'filter' => $this->filter
        ]);
    }

    public function exportExcel()
    {
        return redirect()->route('admin.matrizes.export.excel', [
            'class' => $this->selectedClass,
            'turma' => $this->selectedTurma,
            'search' => $this->search,
            'filter' => $this->filter
        ]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingFilter()
    {
        $this->resetPage();
    }

    public function updatedSelectedClass($value)
    {
        $this->selectedTurma = '';
        $this->resetPage();

        if ($value) {
            $this->turmas = Classroom::where('class', $value)->get();
        } else {
            $this->turmas = [];
        }
    }

    public function updatedSelectedTurma()
    {
        $this->resetPage();
    }

    public function loadData()
    {
        $this->readyToLoad = true;
    }

    protected $listeners = ['confirmed', 'cancelled'];

    public function updatedSelectAll($value)
    {
        if ($value) {
            $this->selectedIds = Downloadsetings::pluck('id')->toArray();
        } else {
            $this->selectedIds = [];
        }
    }

    public function updatedSelectedIds()
    {
        $this->selectAll = count($this->selectedIds) === Downloadsetings::count();
    }

    public function submit($id)
    {
        $this->resetComponentState();

        // Apenas armazena o slug, não o modelo inteiro (evita erro de hydration)
        $download = Downloadcenter::query()->firstWhere('slug', $id);

        if (!$download) {
            $this->alert('error', 'Matriz não encontrada.');
            return;
        }

        $this->slug = $download->slug;

        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => "confirmed",
            'onDismissed' => 'cancelled',
        ]);
    }

    public function confirmed()
    {
        if (!$this->slug) {
            $this->alert('error', 'Erro: Matriz não identificada.');
            return;
        }

        Downloadcenter::where('slug', $this->slug)->delete();
        Downloadsetings::where('download_id', $this->slug)->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Matriz', 'Deletado com sucesso');
        $this->emit('noteAdded');
        $this->alert('success', 'Matriz deletada com sucesso!');

        // Limpar slug após deleção
        $this->slug = null;
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function bulkDelete()
    {
        if (empty($this->selectedIds)) {
            $this->emit('error', 'No items selected!');
            return;
        }

        Downloadsetings::whereIn('id', $this->selectedIds)->delete();

        $this->selectedIds = [];
        $this->selectAll = false;
        $this->emit('noteAdded');
        $this->alert('success', 'Itens selecionados deletado com sucesso!');
        $this->emit('success', 'Itens selecionados deletado com sucesso!');
    }

    public function openBulkDeleteModal($type)
    {
        $this->bulkDeleteType = $type;
        $this->showBulkDeleteModal = true;
    }

    public function closeBulkDeleteModal()
    {
        $this->reset(['showBulkDeleteModal', 'bulkDeleteType', 'bulkDeleteFromDate', 'bulkDeleteToDate', 'bulkDeleteSelectedClass', 'bulkDeleteSelectedTurma']);
    }

    public function confirmBulkDelete()
    {
        switch ($this->bulkDeleteType) {
            case 'all':
                $this->deleteAll();
                break;
            case 'byTurma':
                $this->deleteByTurma();
                break;
            case 'byClasse':
                $this->deleteByClasse();
                break;
            case 'byDateRange':
                $this->deleteByDateRange();
                break;
        }

        $this->closeBulkDeleteModal();
    }

    public function deleteAll()
    {
        $count = Downloadsetings::count();
        Downloadsetings::query()->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Matriz', "Todas as matrizes deletadas ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Todas as matrizes foram deletadas com sucesso ({$count} itens)!");
    }

    public function deleteByTurma()
    {
        if (!$this->bulkDeleteSelectedTurma) {
            $this->alert('error', 'Por favor selecione uma turma!');
            return;
        }

        $count = Downloadsetings::where('class_id', $this->bulkDeleteSelectedTurma)->count();
        Downloadsetings::where('class_id', $this->bulkDeleteSelectedTurma)->delete();

        $turma = Classroom::find($this->bulkDeleteSelectedTurma);
        $turmaName = $turma ? $turma->class . ' Classe - ' . $turma->name : 'Turma';

        Note::createSystemNotification(\App\Models\Admin::class, 'Matriz', "Matrizes da {$turmaName} deletadas ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Matrizes da {$turmaName} deletadas com sucesso ({$count} itens)!");
    }

    public function deleteByClasse()
    {
        if (!$this->bulkDeleteSelectedClass) {
            $this->alert('error', 'Por favor selecione uma classe!');
            return;
        }

        $classroomIds = Classroom::where('class', $this->bulkDeleteSelectedClass)->pluck('id');
        $count = Downloadsetings::whereIn('class_id', $classroomIds)->count();
        Downloadsetings::whereIn('class_id', $classroomIds)->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Matriz', "Matrizes da {$this->bulkDeleteSelectedClass}ª Classe deletadas ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Matrizes da {$this->bulkDeleteSelectedClass}ª Classe deletadas com sucesso ({$count} itens)!");
    }

    public function deleteByDateRange()
    {
        if (!$this->bulkDeleteFromDate || !$this->bulkDeleteToDate) {
            $this->alert('error', 'Por favor selecione as datas de início e fim!');
            return;
        }

        $count = Downloadsetings::whereBetween('created_at', [
            $this->bulkDeleteFromDate . ' 00:00:00',
            $this->bulkDeleteToDate . ' 23:59:59'
        ])->count();

        Downloadsetings::whereBetween('created_at', [
            $this->bulkDeleteFromDate . ' 00:00:00',
            $this->bulkDeleteToDate . ' 23:59:59'
        ])->delete();

        Note::createSystemNotification(\App\Models\Admin::class, 'Matriz', "Matrizes entre {$this->bulkDeleteFromDate} e {$this->bulkDeleteToDate} deletadas ({$count} itens)");
        $this->emit('noteAdded');
        $this->alert('success', "Matrizes entre {$this->bulkDeleteFromDate} e {$this->bulkDeleteToDate} deletadas com sucesso ({$count} itens)!");
    }

    public function resetComponentState()
    {
        $this->reset(['search', 'delete', 'slug', 'filter', 'page', 'selectedIds', 'selectAll']);
    }
    
    public function render()
    {
        $user = Auth::user();

        $models = $this->readyToLoad
            ? Downloadsetings::with('download') // Carrega apenas a relação download
                ->whereHas('download', function ($query) {
                    $query->whereNull('deleted_at'); // Garante que o ficheiro não está deletado
                })
                // Aplicar filtros de pesquisa
                ->when($this->search && !$this->filter, function ($query) {
                    // Pesquisa geral em múltiplos campos
                    $query->where(function ($subQuery) {
                        $subQuery->whereHas('download', function ($downloadQuery) {
                            $downloadQuery->where('title', 'like', '%' . $this->search . '%')
                                    ->orWhere('description', 'like', '%' . $this->search . '%');
                        })
                        ->orWhere('type', 'like', '%' . $this->search . '%')
                        ->orWhereIn('class_id', function ($classQuery) {
                            $classQuery->select('id')
                                ->from('classrooms')
                                ->where(function ($whereClause) {
                                    $whereClause->where('name', 'like', '%' . $this->search . '%')
                                        ->orWhere('class', 'like', '%' . $this->search . '%')
                                        ->orWhere(\DB::raw("CONCAT(class, ' Classe - ', name)"), 'like', '%' . $this->search . '%')
                                        ->orWhere(\DB::raw("CONCAT(class, ' classe')"), 'like', '%' . $this->search . '%');
                                });
                        })
                        ->orWhereIn('student_id', function ($studentQuery) {
                            $studentQuery->select('id')
                                ->from('users')
                                ->where('name', 'like', '%' . $this->search . '%');
                        });
                    });
                })
                // Aplicar filtros específicos
                ->when($this->filter === 'title' && $this->search, function ($query) {
                    $query->whereHas('download', function ($subQuery) {
                        $subQuery->where('title', 'like', '%' . $this->search . '%');
                    });
                })
                ->when($this->filter === 'description' && $this->search, function ($query) {
                    $query->whereHas('download', function ($subQuery) {
                        $subQuery->where('description', 'like', '%' . $this->search . '%');
                    });
                })
                ->when($this->filter === 'class' && $this->search, function ($query) {
                    $query->whereIn('class_id', function ($subQuery) {
                        $subQuery->select('id')
                            ->from('classrooms')
                            ->where(function ($whereClause) {
                                $whereClause->where('name', 'like', '%' . $this->search . '%')
                                    ->orWhere('class', 'like', '%' . $this->search . '%')
                                    ->orWhere(\DB::raw("CONCAT(class, ' Classe - ', name)"), 'like', '%' . $this->search . '%')
                                    ->orWhere(\DB::raw("CONCAT(class, ' classe')"), 'like', '%' . $this->search . '%');
                            });
                    });
                })
                ->when($this->filter === 'student' && $this->search, function ($query) {
                    $query->whereIn('student_id', function ($subQuery) {
                        $subQuery->select('id')
                            ->from('users')
                            ->where('name', 'like', '%' . $this->search . '%');
                    });
                })
                ->when($this->filter === 'type' && $this->search, function ($query) {
                    $searchLower = strtolower($this->search);
                    $query->where(function ($subQuery) use ($searchLower) {
                        $subQuery->where('type', 'like', '%' . $this->search . '%');
                        
                        // Permitir buscar por termos em português
                        if (strpos($searchLower, 'todos') !== false) {
                            $subQuery->orWhere('type', 'all');
                        }
                        if (strpos($searchLower, 'estudante') !== false) {
                            $subQuery->orWhere('type', 'student');
                        }
                        if (strpos($searchLower, 'classe') !== false) {
                            $subQuery->orWhere('type', 'class');
                        }
                    });
                })
                ->when($this->filter === 'year' && $this->search, function ($query) {
                    if (is_numeric($this->search)) {
                        $query->whereYear('created_at', $this->search);
                    }
                })
                // Filtro por Classe
                ->when($this->selectedClass, function ($query) {
                    $query->whereIn('class_id', function ($subQuery) {
                        $subQuery->select('id')
                            ->from('classrooms')
                            ->where('class', $this->selectedClass);
                    });
                })
                // Filtro por Turma específica
                ->when($this->selectedTurma, function ($query) {
                    $query->where('class_id', $this->selectedTurma);
                })
                // Aplicar permissões de usuário
                ->when(Auth::guard('admin')->check(), function ($query) {
                    // Admin: vê todas as matrizes
                    $query->orderByDesc('created_at');
                })
                ->when(Auth::guard('web')->check(), function ($query) use ($user) {
                    // Estudante: vê apenas matrizes para todos (type='all'),
                    // ou para sua turma (class_id), ou específicas para ele (student_id)
                    $query->where(function ($subQuery) use ($user) {
                        $subQuery->where("type", "all")
                            ->orWhere("class_id", $user->classroom_id)
                            ->orWhere("student_id", $user->id);
                    })->orderByDesc('created_at');
                })
                ->paginate(10)
            : [];

        // Buscar todas as classes para o filtro
        $classes = Classroom::select('class')->distinct()->orderByRaw('CAST(class AS UNSIGNED)')->get();

        // Detectar se é estudante (guard web) ou admin (guard admin)
        $student = Auth::guard('web')->check() ? Auth::guard('web')->user() : null;

        return view('livewire.admin.downloadcenter.downloads', compact('models', 'classes', 'student'));
    }
}