<?php

namespace App\Http\Livewire\Admin\Exams;

use App\Models\ExamConfiguration;
use App\Models\Subject;
use App\Services\ExamConfigurationService;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;

class ConfigureExams extends Component
{
    use FindGuard, LivewireAlert;

    public $class_level;
    public $academic_year;
    public $configurations = [];
    public $showFilters = true;
    public $totalSubjects = 0;
    public $subjectsWithNE = 0;
    public $subjectsWithAF = 0;

    protected $listeners = [
        'confirmed',
        'cancelled',
        'confirmedCopyAll',
    ];

    public function mount()
    {
        // Ano atual por padrão
        $this->academic_year = date('Y');
    }

    public function updatedClassLevel()
    {
        $this->loadConfigurations();
    }

    public function updatedAcademicYear()
    {
        if ($this->class_level) {
            $this->loadConfigurations();
        }
    }

    public function loadConfigurations()
    {
        if (!$this->class_level) {
            $this->configurations = [];
            return;
        }

        $service = new ExamConfigurationService();
        $this->configurations = $service->loadConfigurationsForClass(
            $this->class_level,
            $this->academic_year
        );

        $this->calculateStatistics();
    }

    public function calculateStatistics()
    {
        $this->totalSubjects = count($this->configurations);
        $this->subjectsWithNE = collect($this->configurations)
            ->where('exam_type', 'NE')
            ->count();
        $this->subjectsWithAF = collect($this->configurations)
            ->where('exam_type', 'AF')
            ->count();
    }

    public function submit()
    {
        if (empty($this->configurations)) {
            $this->alert('warning', 'Selecione uma classe primeiro!');
            return;
        }

        $this->confirm('Tem certeza de que deseja salvar estas configurações?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, salvar!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $service = new ExamConfigurationService();

        $configsToSave = collect($this->configurations)
            ->mapWithKeys(function ($config, $subjectId) {
                return [$subjectId => $config['exam_type']];
            })
            ->toArray();

        $success = $service->saveConfigurationsForClass(
            $this->class_level,
            $configsToSave,
            $this->academic_year
        );

        if ($success) {
            $this->alert('success', 'Configurações salvas com sucesso!', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
            ]);

            // Recarregar para mostrar dados atualizados
            $this->loadConfigurations();
        } else {
            $this->alert('error', 'Erro ao salvar configurações. Tente novamente.');
        }
    }

    public function cancelled()
    {
        $this->alert('info', 'Operação cancelada.');
    }

    public function resetAll()
    {
        $this->confirm('Deseja remover TODOS os exames desta classe?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, remover todos!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedReset',
        ]);
    }

    public function confirmedReset()
    {
        foreach ($this->configurations as $subjectId => &$config) {
            $config['exam_type'] = 'none';
        }

        $this->calculateStatistics();
        $this->alert('success', 'Todas as configurações foram removidas. Clique em "Salvar" para confirmar.');
    }

    public function copyFromPreviousYear()
    {
        $service = new ExamConfigurationService();
        $previousYear = $this->academic_year - 1;

        // Verificar se há configurações no ano anterior para esta classe
        $countPrevious = $service->countConfigurationsForYear($previousYear, $this->class_level);

        if ($countPrevious == 0) {
            $this->alert('warning', "Não há configurações de {$this->class_level}ª classe no ano {$previousYear}.");
            return;
        }

        // Copiar apenas a classe selecionada
        $result = $service->copyConfigurationsToNewYear($previousYear, $this->academic_year, $this->class_level);

        if ($result['success']) {
            $this->alert('success', "Configurações da {$this->class_level}ª classe copiadas de {$previousYear}! {$result['message']}");
            $this->loadConfigurations();
        } else {
            $this->alert('error', $result['message'] ?? 'Erro ao copiar configurações.');
        }
    }

    public function copyAllClassesFromPreviousYear()
    {
        $service = new ExamConfigurationService();
        $previousYear = $this->academic_year - 1;

        // Verificar se há configurações no ano anterior
        $countPrevious = $service->countConfigurationsForYear($previousYear);

        if ($countPrevious == 0) {
            $this->alert('warning', "Não há configurações no ano {$previousYear}.");
            return;
        }

        $this->confirm("Copiar TODAS as configurações de {$previousYear} para {$this->academic_year}? ({$countPrevious} configurações)", [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, copiar tudo!',
            'cancelButtonText' => 'Cancelar',
            'onConfirmed' => 'confirmedCopyAll',
        ]);
    }

    public function confirmedCopyAll()
    {
        $service = new ExamConfigurationService();
        $previousYear = $this->academic_year - 1;

        // Copiar todas as classes
        $result = $service->copyConfigurationsToNewYear($previousYear, $this->academic_year, null);

        if ($result['success']) {
            $this->alert('success', "Todas as configurações copiadas de {$previousYear}! {$result['message']}");
            if ($this->class_level) {
                $this->loadConfigurations();
            }
        } else {
            $this->alert('error', $result['message'] ?? 'Erro ao copiar configurações.');
        }
    }

    public function render()
    {
        return view('livewire.admin.exams.configure-exams');
    }
}
