<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\PaymentReference;
use App\Models\User;
use App\Models\Classroom;
use App\Models\FeeStructure;
use App\Models\FeeMonthDeadline;
use App\Models\POS\Product;
use App\Models\POS\Sale;
use App\Models\POS\SaleItem;
use App\Exports\PaymentReferencesExport;
use App\Exports\PaymentReferencesTemplateExport;
use App\Imports\PaymentReferencesImport;
use App\Mail\PaymentReferenceCreated;
use App\Models\Admin;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Maatwebsite\Excel\Facades\Excel;
use App\Services\SmsService;
use Exception;

class GenerateReferences extends Component
{
    use WithPagination, WithFileUploads;

    protected $paginationTheme = 'bootstrap';

    // Modo de geração
    public $generationMode = 'individual'; // individual, class, bulk, import

    // Filtros para busca individual
    public $searchTerm = '';
    public $selectedStudentId = null;

    // Filtros para seleção em cascata (classe → turma)
    public $selectedClass = null;
    public $selectedTurma = null;
    public $availableTurmas = [];
    public $selectAllTurmas = false; // Checkbox para selecionar todas as turmas da classe

    // Filtros para turma (modo antigo)
    public $selectedClassroom = null;
    public $classroomStudents = [];

    // Importação via Excel
    public $importFile = null;
    public $importProgress = 0;
    public $importResults = null;

    // Dados da referência
    public $selectedFeeType = null;
    public $feeTypeMode = 'predefined'; // 'predefined' ou 'custom'
    public $customFeeTypeName = '';
    public $selectedPosProduct = null; // Produto POS selecionado
    public $posProducts = []; // Lista de produtos POS
    public $feeMonth = null;
    public $feeYear = null;
    public $amount = null;
    public $paymentDescription = '';

    // Penalty/Fine fields (auto-populated from FeeStructure/FeeMaster1, not user-editable)
    // Estas propriedades são preenchidas automaticamente quando uma taxa é selecionada
    // e podem ser usadas para informação/display, mas não são editáveis pelo usuário
    public $fine_type = 'percentage';
    public $percentage = null;
    public $fixed_amount = null;
    public $grace_days = 0;

    // Período de validade da referência (quanto tempo ela fica válida para pagamento)
    public $validityMode = 'days'; // 'dates' ou 'days'
    public $startDate; // Data de início da validade
    public $endDate; // Data de fim da validade
    public $validityDays = 30; // Dias de validade
    public $calculatedDays = null; // Dias calculados

    // Prazo para geração (até quando pode gerar referências para este mês)
    public $generationDeadline = null; // Deadline do FeeMonthDeadline
    public $canGenerate = true; // Se ainda pode gerar para este mês
    public $deadlineMessage = ''; // Mensagem sobre o deadline

    // Entidade padrão (lida do .env)
    public $entityCode;

    // Filtros para tabela de referências
    public $filterStatus = ''; // all, pending, paid, expired
    public $filterSearch = ''; // Busca por nome/código/referência
    public $filterAcademicYear = ''; // Filtro por ano acadêmico
    public $filterClasse = ''; // Filtro por classe
    public $filterTurma = ''; // Filtro por turma
    public $showAllReferences = false; // Toggle para mostrar tabela completa

    // Estatísticas
    public $stats = [
        'total_generated' => 0,
        'total_pending' => 0,
        'total_paid' => 0,
        'total_expired' => 0,
    ];

    protected $rules = [
        'selectedFeeType' => 'required', // Can be FeeGroup ID, 'pos', or custom
        'feeMonth' => 'required|string',
        'feeYear' => 'required|integer|min:2020|max:2100',
        'amount' => 'required|numeric|min:1',
        'startDate' => 'required|date',
        'paymentDescription' => 'required|string|max:500',
        // Nota: Regras de multa removidas - valores vêm automaticamente do Plano de Taxas
    ];

    public function mount()
    {
        $this->entityCode = config('payments.entity', '90013');
        $this->feeMonth = $this->getCurrentMonthInPortuguese();
        $this->feeYear = now()->year;
        $this->updateStartDateFromFeeMonth();
        $this->loadPosProducts();
        $this->loadStats();
        $this->checkGenerationDeadline();
    }

    /**
     * Retorna o mês atual em português
     */
    private function getCurrentMonthInPortuguese()
    {
        $monthsPortuguese = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];
        return $monthsPortuguese[now()->month];
    }

    public function loadPosProducts()
    {
        $this->posProducts = Product::where('is_active', true)
            ->orderBy('name')
            ->get();
    }

    public function updatedSelectedFeeType($value)
    {
        // Se selecionou POS, limpar produto selecionado
        if ($value === 'pos') {
            $this->amount = null;
            $this->paymentDescription = '';
            $this->fine_type = 'percentage';
            $this->percentage = null;
            $this->fixed_amount = null;
            $this->grace_days = 0;
            return;
        }

        // Limpar seleção de produto POS se não for POS
        $this->selectedPosProduct = null;

        // Auto-preencher valor, descrição e multas baseado no Plano de Taxas selecionado
        if ($value) {
            // Verificar se é FeeStructure (prefixo 'fs_') ou FeeMaster1
            if (str_starts_with($value, 'fs_')) {
                // É FeeStructure - extrair ID real
                $structureId = (int) str_replace('fs_', '', $value);
                $feeStructure = FeeStructure::find($structureId);

                if ($feeStructure) {
                    // Preencher valor
                    $this->amount = $feeStructure->monthly_fee;

                    // Extrair ano do academic_year (formato: 2025/2026)
                    if (!empty($feeStructure->academic_year)) {
                        $years = explode('/', $feeStructure->academic_year);
                        $this->feeYear = isset($years[1]) ? (int)$years[1] : (int)$years[0];
                    }

                    // Preencher descrição
                    $this->paymentDescription = $feeStructure->fee_name;
                    if ($this->feeMonth) {
                        $this->paymentDescription .= ' - ' . $this->feeMonth . '/' . $this->feeYear;
                    }

                    // Preencher multas
                    $this->fine_type = $feeStructure->penalty_type ?? 'percentage';
                    $this->percentage = $feeStructure->late_penalty_percentage;
                    $this->fixed_amount = null;
                    $this->grace_days = 0;

                    // Atualizar data de início para corresponder ao mês/ano selecionado
                    $this->updateStartDateFromFeeMonth();

                    // Recarregar estudantes se já houver turma selecionada (para aplicar o novo ano)
                    if ($this->selectedTurma) {
                        $this->updatedSelectedTurma($this->selectedTurma);
                    } elseif ($this->selectedClassroom) {
                        $this->updatedSelectedClassroom($this->selectedClassroom);
                    }
                }
            }
        }
    }

    /**
     * Atualiza descrição quando o mês é alterado
     */
    public function updatedSelectedPosProduct($productId)
    {
        if ($productId) {
            $product = Product::find($productId);
            if ($product) {
                // Auto-preencher valor e descrição
                $this->amount = $product->price;
                $this->paymentDescription = "POS - " . $product->name;
            }
        }
    }

    /* ==================== WATCHERS ==================== */

    public function updatedFeeMonth($value)
    {
        // Atualizar descrição quando o mês é alterado
        if ($value && $this->selectedFeeType && str_starts_with($this->selectedFeeType, 'fs_')) {
            $structureId = (int) str_replace('fs_', '', $this->selectedFeeType);
            $feeStructure = FeeStructure::find($structureId);

            if ($feeStructure) {
                $monthName = $this->getPortugueseMonth($value);
                $this->paymentDescription = $feeStructure->fee_name;

                if ($this->feeYear) {
                    $this->paymentDescription .= ' - ' . $monthName . '/' . $this->feeYear;
                }
            }
        }

        $this->checkGenerationDeadline();
        $this->updateStartDateFromFeeMonth();
    }

    public function updatedFeeYear($value)
    {
        // Atualizar descrição quando o ano é alterado
        if ($value && $this->selectedFeeType && str_starts_with($this->selectedFeeType, 'fs_') && $this->feeMonth) {
            $structureId = (int) str_replace('fs_', '', $this->selectedFeeType);
            $feeStructure = FeeStructure::find($structureId);

            if ($feeStructure) {
                $monthName = $this->getPortugueseMonth($this->feeMonth);
                $this->paymentDescription = $feeStructure->fee_name;
                $this->paymentDescription .= ' - ' . $monthName . '/' . $value;
            }
        }

        $this->checkGenerationDeadline();
        $this->updateStartDateFromFeeMonth();
    }

    /**
     * Atualiza a data de início com base no mês/ano selecionado
     */
    private function updateStartDateFromFeeMonth()
    {
        if (!$this->feeMonth || !$this->feeYear) {
            return;
        }

        $monthNumber = $this->getMonthNumber($this->feeMonth);
        if ($monthNumber) {
            $this->startDate = Carbon::create($this->feeYear, $monthNumber, 1)->format('Y-m-d');
            $this->calculatePeriod();
        }
    }

    /**
     * Converte nome do mês para número
     */
    private function getMonthNumber($monthName): ?int
    {
        $months = [
            'janeiro' => 1, 'january' => 1,
            'fevereiro' => 2, 'february' => 2,
            'março' => 3, 'march' => 3,
            'abril' => 4, 'april' => 4,
            'maio' => 5, 'may' => 5,
            'junho' => 6, 'june' => 6,
            'julho' => 7, 'july' => 7,
            'agosto' => 8, 'august' => 8,
            'setembro' => 9, 'september' => 9,
            'outubro' => 10, 'october' => 10,
            'novembro' => 11, 'november' => 11,
            'dezembro' => 12, 'december' => 12,
        ];

        return $months[strtolower($monthName)] ?? null;
    }

    /**
     * Verifica se a data de início corresponde ao mês/ano selecionado
     */
    public function getStartDateMismatchProperty(): bool
    {
        if (!$this->startDate || !$this->feeMonth || !$this->feeYear) {
            return false;
        }

        $startDateParsed = Carbon::parse($this->startDate);
        $expectedMonth = $this->getMonthNumber($this->feeMonth);

        return $startDateParsed->month !== $expectedMonth || $startDateParsed->year !== (int) $this->feeYear;
    }

    public function updatedStartDate()
    {
        $this->calculatePeriod();
    }

    public function updatedEndDate()
    {
        $this->calculatePeriod();
    }

    public function updatedValidityDays()
    {
        $this->calculatePeriod();
    }

    public function updatedValidityMode()
    {
        $this->calculatePeriod();
    }

    /**
     * Verifica o prazo para GERAR referências (não confundir com validade da referência)
     */
    private function checkGenerationDeadline()
    {
        if (!$this->feeMonth || !$this->feeYear) {
            return;
        }

        $deadline = FeeMonthDeadline::getDeadline($this->feeMonth, $this->feeYear);

        if ($deadline && $deadline->deadline) {
            $this->generationDeadline = Carbon::parse($deadline->deadline);

            // Verifica se já passou o prazo para gerar
            if (now()->isAfter($this->generationDeadline)) {
                $this->canGenerate = false;
                $this->deadlineMessage = 'Prazo para gerar referências de ' . ucfirst($this->feeMonth) . '/' . $this->feeYear .
                                        ' expirou em ' . $this->generationDeadline->format('d/m/Y');
            } else {
                $this->canGenerate = true;
                $this->deadlineMessage = 'Você pode gerar referências para ' . ucfirst($this->feeMonth) . '/' . $this->feeYear .
                                        ' até ' . $this->generationDeadline->format('d/m/Y');
            }
        } else {
            // Sem deadline configurado = pode gerar sempre
            $this->canGenerate = true;
            $this->generationDeadline = null;
            $this->deadlineMessage = 'Sem prazo configurado - pode gerar referências a qualquer momento';
        }
    }

    /**
     * Calcula período de VALIDADE da referência (quanto tempo ela fica válida para pagamento)
     */
    private function calculatePeriod()
    {
        if (!$this->startDate) {
            return;
        }

        switch ($this->validityMode) {
            case 'dates':
                // Modo 1: Por Datas (usuário escolhe início e fim)
                if ($this->startDate && $this->endDate) {
                    $start = Carbon::parse($this->startDate);
                    $end = Carbon::parse($this->endDate);
                    if ($end->gte($start)) {
                        $this->calculatedDays = $start->diffInDays($end);
                        $this->validityDays = $this->calculatedDays;
                    }
                }
                break;

            case 'days':
                // Modo 2: Por Dias (usuário escolhe início + dias)
                if ($this->startDate && $this->validityDays) {
                    $this->endDate = Carbon::parse($this->startDate)
                        ->addDays($this->validityDays)
                        ->format('Y-m-d');
                    $this->calculatedDays = $this->validityDays;
                }
                break;
        }
    }

    /* ==================== GERAÇÃO INDIVIDUAL ==================== */

    public function searchStudents()
    {
        // A busca é feita automaticamente via computed property
        $this->resetPage();
    }

    public function selectStudent($studentId)
    {
        $this->selectedStudentId = $studentId;
        $student = User::find($studentId);
        if ($student) {
            $this->searchTerm = $student->name;
        }
    }

    public function generateIndividualReference()
    {
        Log::info('=== MÉTODO CHAMADO: generateIndividualReference ===', [
            'selectedStudentId' => $this->selectedStudentId,
            'selectedFeeType' => $this->selectedFeeType,
            'amount' => $this->amount,
            'feeMonth' => $this->feeMonth,
            'feeYear' => $this->feeYear,
        ]);

        // Verifica se ainda pode gerar referências para este mês
        if (!$this->canGenerate) {
            session()->flash('error', $this->deadlineMessage);
            return;
        }

        // Validação: verifica se estudante foi selecionado
        if (!$this->selectedStudentId) {
            session()->flash('error', 'Por favor, selecione um estudante.');
            return;
        }

        // Validação: verifica campos obrigatórios antes de validar com Laravel
        if ($this->feeTypeMode === 'predefined') {
            if (!$this->selectedFeeType) {
                session()->flash('error', 'Por favor, selecione o tipo de taxa.');
                return;
            }
        } else {
            // Modo custom
            if (!$this->customFeeTypeName) {
                session()->flash('error', 'Por favor, preencha o nome do tipo de taxa.');
                return;
            }
        }

        if (!$this->amount || $this->amount <= 0) {
            session()->flash('error', 'Por favor, preencha o valor da taxa.');
            return;
        }

        if (!$this->paymentDescription) {
            session()->flash('error', 'Por favor, preencha a descrição do pagamento.');
            return;
        }

        // Validação dinâmica baseada no modo
        $rules = [
            'feeMonth' => 'required|string',
            'feeYear' => 'required|integer|min:2020|max:2100',
            'amount' => 'required|numeric|min:1',
            'startDate' => 'required|date',
            'paymentDescription' => 'required|string|max:500',
        ];

        // Adiciona validação específica baseada no modo
        if ($this->feeTypeMode === 'predefined') {
            $rules['selectedFeeType'] = 'required';
        } else {
            $rules['customFeeTypeName'] = 'required|string|max:255';
        }

        $this->validate($rules);

        DB::beginTransaction();
        try {
            // Busca o estudante selecionado
            $student = User::find($this->selectedStudentId);

            if (!$student) {
                session()->flash('error', 'Estudante não encontrado.');
                return;
            }

            // Verificar se já existe referência para este estudante/mês/ano
            $existingRef = PaymentReference::where('student_id', $student->id)
                ->where('fee_month', $this->feeMonth)
                ->where('fee_year', $this->feeYear)
                ->whereIn('status', ['pending', 'generated'])
                ->first();

            if ($existingRef) {
                session()->flash('error', "Já existe uma referência pendente para {$student->name} em {$this->feeMonth}/{$this->feeYear}.");
                return;
            }

            $reference = $this->createPaymentReference($student);
            $this->sendReferenceNotification($student, $reference);
            $this->sendPcaEmailNotification($reference);

            DB::commit();

            session()->flash('success', "Referência gerada com sucesso para {$student->name}!");

            $this->reset(['selectedStudentId', 'searchTerm']);
            $this->loadStats();

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Erro ao gerar referência individual', [
                'student_id' => $this->selectedStudentId,
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao gerar referência: ' . $e->getMessage());
        }
    }

    /* ==================== SELEÇÃO EM CASCATA (CLASSE → TURMA) ==================== */

    public function updatedSelectedClass($class)
    {
        $this->selectedTurma = null;
        $this->selectAllTurmas = false;
        $this->classroomStudents = [];

        if ($class) {
            $this->availableTurmas = Classroom::where('class', $class)
                ->orderBy('name')
                ->get();
        } else {
            $this->availableTurmas = [];
        }
    }

    public function updatedSelectAllTurmas($value)
    {
        if ($value) {
            // Desmarcar turma específica se "Todas" for marcado
            $this->selectedTurma = null;
            // Carregar todos os estudantes de todas as turmas da classe
            $this->loadAllStudentsFromClass();
        } else {
            // Limpar estudantes se desmarcar
            $this->classroomStudents = [];
        }
    }

    public function updatedSelectedTurma($turmaId)
    {
        // Desmarcar "Todas as turmas" se uma turma específica for selecionada
        if ($turmaId) {
            $this->selectAllTurmas = false;

            // Usar o ano selecionado (feeYear) ou o ano atual como fallback
            $academicYear = $this->feeYear ?? date('Y');

            $this->classroomStudents = User::where('classroom_id', $turmaId)
                ->where('is_active', 1)
                ->where('year_edu', $academicYear)
                ->orderBy('name')
                ->get();

            // Log para debug
            Log::info('Students loaded for turma', [
                'turma_id' => $turmaId,
                'academic_year' => $academicYear,
                'total_students' => $this->classroomStudents->count()
            ]);
        } else {
            $this->classroomStudents = [];
        }
    }

    /**
     * Carrega todos os estudantes de todas as turmas da classe selecionada
     */
    private function loadAllStudentsFromClass()
    {
        if (!$this->selectedClass) {
            $this->classroomStudents = [];
            return;
        }

        // Usar o ano selecionado (feeYear) ou o ano atual como fallback
        $academicYear = $this->feeYear ?? date('Y');

        // Buscar todos os IDs das turmas da classe selecionada
        $turmaIds = Classroom::where('class', $this->selectedClass)
            ->pluck('id')
            ->toArray();

        if (empty($turmaIds)) {
            $this->classroomStudents = [];
            return;
        }

        // Carregar todos os estudantes dessas turmas
        $this->classroomStudents = User::whereIn('classroom_id', $turmaIds)
            ->where('is_active', 1)
            ->where('year_edu', $academicYear)
            ->orderBy('name')
            ->get();

        // Log para debug
        Log::info('Students loaded for all turmas of class', [
            'class' => $this->selectedClass,
            'turma_ids' => $turmaIds,
            'academic_year' => $academicYear,
            'total_students' => $this->classroomStudents->count()
        ]);
    }

    /* ==================== GERAÇÃO POR TURMA (MODO ANTIGO) ==================== */

    public function updatedSelectedClassroom($classroomId)
    {
        if ($classroomId) {
            // Usar o ano selecionado (feeYear) ou o ano atual como fallback
            $academicYear = $this->feeYear ?? date('Y');

            $this->classroomStudents = User::where('classroom_id', $classroomId)
                ->where('is_active', 1)
                ->where('year_edu', $academicYear)
                ->orderBy('name')
                ->get();

            // Log para debug
            Log::info('Students loaded for classroom', [
                'classroom_id' => $classroomId,
                'academic_year' => $academicYear,
                'total_students' => $this->classroomStudents->count()
            ]);
        } else {
            $this->classroomStudents = [];
        }
    }

    public function generateClassReferences()
    {
        // Verifica se ainda pode gerar referências para este mês
        if (!$this->canGenerate) {
            session()->flash('error', $this->deadlineMessage);
            return;
        }

        // Validação: aceita tanto o novo sistema (selectedTurma) quanto o antigo (selectedClassroom)
        $validationRules = $this->rules;
        $validationRules['feeMonth'] = 'required|string';

        // Verifica qual sistema está sendo usado
        if ($this->selectedTurma) {
            // Novo sistema de cascata (classe → turma específica)
            $validationRules['selectedTurma'] = 'required|exists:classrooms,id';
        } elseif ($this->selectAllTurmas && $this->selectedClass) {
            // Novo sistema de cascata (classe → todas as turmas)
            $validationRules['selectedClass'] = 'required';
        } elseif ($this->selectedClassroom) {
            // Sistema antigo (turma direta)
            $validationRules['selectedClassroom'] = 'required|exists:classrooms,id';
        } else {
            session()->flash('error', 'Por favor, selecione uma turma ou marque "Todas as turmas desta classe".');
            return;
        }

        $this->validate($validationRules);

        if (empty($this->classroomStudents) || count($this->classroomStudents) == 0) {
            session()->flash('error', 'Nenhum estudante encontrado na turma selecionada.');
            return;
        }

        DB::beginTransaction();
        try {
            $successCount = 0;
            $failedCount = 0;
            $duplicateCount = 0;
            $lastReference = null;

            foreach ($this->classroomStudents as $student) {
                try {
                    // Verifica se já existe referência pendente para este estudante/mês/ano
                    $existingRef = PaymentReference::where('student_id', $student->id)
                        ->where('fee_month', $this->feeMonth)
                        ->where('fee_year', $this->feeYear)
                        ->whereIn('status', ['pending', 'generated'])
                        ->first();

                    if ($existingRef) {
                        $duplicateCount++;
                        continue;
                    }

                    $reference = $this->createPaymentReference($student);
                    $this->sendReferenceNotification($student, $reference);
                    $lastReference = $reference;
                    $successCount++;

                } catch (\Exception $e) {
                    $failedCount++;
                    Log::error('Erro ao gerar referência para estudante', [
                        'student_id' => $student->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            // Enviar email ao PCA com total de referências criadas
            if ($lastReference && $successCount > 0) {
                $this->sendPcaEmailNotification($lastReference, $successCount);
            }

            DB::commit();

            $message = "{$successCount} referência(s) gerada(s) com sucesso";
            if ($duplicateCount > 0) {
                $message .= ", {$duplicateCount} duplicada(s) ignorada(s)";
            }
            if ($failedCount > 0) {
                $message .= ", {$failedCount} falharam";
            }

            session()->flash('success', $message);

            // Resetar ambos os sistemas (novo e antigo)
            $this->reset(['selectedClass', 'selectedTurma', 'availableTurmas', 'selectAllTurmas', 'selectedClassroom', 'classroomStudents']);
            $this->loadStats();

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Erro ao gerar referências por turma', [
                'classroom_id' => $this->selectedClassroom,
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao gerar referências: ' . $e->getMessage());
        }
    }

    /* ==================== MÉTODOS AUXILIARES ==================== */

    private function createPaymentReference($student)
    {
        // Gera número de referência usando algoritmo BCI oficial (formato V1 - 11 dígitos)
        $referenceGenerator = app(\App\Services\ReferenceGenerator::class);
        $referenceNumber = $referenceGenerator->makeV1FromStudent(
            $this->feeMonth,
            $student,
            $this->amount
        );

        // Calcula datas de validade
        $validFrom = Carbon::parse($this->startDate)->startOfDay();
        $expiresAt = Carbon::parse($this->startDate)->addDays($this->validityDays)->endOfDay();

        // Busca informações da turma do estudante
        $classe = $student->classroom?->class;
        $turma = $student->classroom?->name;

        // Determina o ano acadêmico com base no período de validade
        $startYear = $validFrom->year;
        $endYear = $expiresAt->year;

        if ($startYear === $endYear) {
            $academicYear = (string) $startYear;
        } else {
            $academicYear = $startYear . '-' . $endYear;
        }

        // Prepara metadata com descrição do pagamento
        $metadata = [
            'description' => $this->paymentDescription,
        ];

        // Se for modo custom, adiciona o nome personalizado
        if ($this->feeTypeMode === 'custom') {
            $metadata['custom_fee_type'] = $this->customFeeTypeName;
        }

        // Buscar FeeStructure para calcular datas de pagamento e multa
        $feeStructure = null;
        $paymentStartsAt = null;
        $dueDate = null;
        $penaltyType = null;
        $penaltyRate = null;

        if ($this->selectedFeeType && str_starts_with($this->selectedFeeType, 'fs_')) {
            $structureId = (int) str_replace('fs_', '', $this->selectedFeeType);
            $feeStructure = FeeStructure::find($structureId);

            if ($feeStructure) {
                // Converter mês para número
                $monthNumber = $this->getMonthNumber($this->feeMonth);

                if ($monthNumber) {
                    // Calcular payment_starts_at com offset
                    $startOffsetMonths = $feeStructure->payment_start_offset_months ?? -1;
                    $paymentStartsAt = Carbon::create($this->feeYear, $monthNumber, $feeStructure->payment_start_day)
                        ->addMonths($startOffsetMonths)
                        ->startOfDay();

                    // Calcular due_date com offset
                    $dueOffsetMonths = $feeStructure->payment_due_offset_months ?? 0;
                    $dueDate = Carbon::create($this->feeYear, $monthNumber, $feeStructure->payment_due_day)
                        ->addMonths($dueOffsetMonths)
                        ->endOfDay();
                }

                // Guardar informações de multa
                $penaltyType = $feeStructure->penalty_type;
                $penaltyRate = $feeStructure->late_penalty_percentage;
            }
        }

        $reference = PaymentReference::create([
            'student_id' => $student->id,
            'classe' => $classe,
            'turma' => $turma,
            'entity_code' => $this->entityCode,
            'reference_number' => $referenceNumber,
            'amount' => $this->amount,
            'fee_month' => $this->feeMonth,
            'fee_year' => $this->feeYear,
            'academic_year' => $academicYear,
            'status' => 'pending',
            'approval_status' => 'pending_approval', // Requer aprovação antes de ficar disponível
            'valid_from' => $validFrom,
            'payment_starts_at' => $paymentStartsAt,
            'due_date' => $dueDate,
            'expires_at' => $expiresAt,
            'penalty_type' => $penaltyType,
            'penalty_rate' => $penaltyRate,
            'has_penalty' => false,
            'metadata' => $metadata,
        ]);

        // Se é produto POS, guardar informação do produto na referência
        if ($this->selectedFeeType === 'pos' && $this->selectedPosProduct) {
            $this->attachPosProductToReference($reference);
        }

        return $reference;
    }

    private function attachPosProductToReference($reference)
    {
        // Apenas guardar o ID do produto na referência
        // A venda POS e o abate de stock só serão feitos quando o pagamento for confirmado
        try {
            $product = Product::find($this->selectedPosProduct);

            if ($product) {
                // Atualizar metadados mantendo a descrição existente
                $currentMetadata = is_array($reference->metadata) ? $reference->metadata : [];
                $updatedMetadata = array_merge($currentMetadata, [
                    'pos_product_id' => $product->id,
                    'pos_product_name' => $product->name,
                    'pos_product_price' => $product->price,
                ]);

                $reference->update([
                    'metadata' => $updatedMetadata
                ]);

                Log::info('Produto POS vinculado à referência (aguardando pagamento)', [
                    'reference_id' => $reference->id,
                    'product_id' => $product->id,
                    'product_name' => $product->name,
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Erro ao vincular produto POS à referência', [
                'error' => $e->getMessage(),
                'reference_id' => $reference->id,
                'product_id' => $this->selectedPosProduct,
            ]);
        }
    }

    /**
     * Envia SMS quando a referência é gerada
     */
    private function sendReferenceNotification($student, $reference)
    {
        try {
            if (!$student->mobile) {
                Log::warning('Cannot send SMS - missing mobile number', [
                    'reference_id' => $reference->id,
                    'student_id' => $student->id
                ]);
                return false;
            }

            $monthName = $this->getPortugueseMonth($this->feeMonth);
            $expiresDate = $reference->expires_at ? $reference->expires_at->format('d/m/Y') : 'N/A';

            // Buscar descrição de pagamento
            $description = $this->paymentDescription ?: 'Pagamento de Mensalidade';

            // Mensagem com Entidade, Referência, Valor e Descrição
            $message = sprintf(
                "COPMOZ - Referencia Gerada!\n\nEstudante: %s\nDescricao: %s\nMes: %s/%s\n\nEntidade: %s\nReferencia: %s\nValor: %.2f MT\nExpira: %s\n\nAguarda aprovacao. Pague via eBanking, Balcões, ATMs da RedeSIMO e pelo USSD - BCI",
                $student->name,
                $description,
                $monthName,
                $reference->fee_year,
                $reference->entity_code,
                $reference->reference_number,
                $reference->amount,
                $expiresDate
            );

            // Usar SmsService para enviar SMS
            $smsService = new SmsService();
            $result = $smsService->send($student->mobile, $message);

            if ($result['success']) {
                Log::info('Generation SMS sent', [
                    'reference_id' => $reference->id,
                    'student_id' => $student->id,
                    'mobile' => $student->mobile,
                    'gateway' => $result['gateway'],
                    'message_id' => $result['message_id'] ?? null,
                    'entity' => $reference->entity_code,
                    'reference' => $reference->reference_number,
                    'amount' => $reference->amount,
                    'description' => $description,
                ]);

                return true;
            } else {
                Log::error('Error sending generation SMS', [
                    'reference_id' => $reference->id,
                    'error' => $result['error'] ?? 'Unknown error',
                    'gateway' => $result['gateway'],
                ]);
                return false;
            }

        } catch (\Exception $e) {
            Log::error('Error sending generation SMS', [
                'reference_id' => $reference->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Envia email para todos os PCA (role 0) quando uma referência é gerada
     */
    private function sendPcaEmailNotification($reference, $totalCount = 1)
    {
        try {
            // Buscar todos os administradores com role 0 (PCA/Super Admin)
            $pcaAdmins = Admin::where('role', 0)
                ->where('is_active', 1)
                ->whereNotNull('email')
                ->get();

            if ($pcaAdmins->isEmpty()) {
                Log::warning('No PCA admins found to send email notification', [
                    'reference_id' => $reference->id,
                ]);
                return false;
            }

            foreach ($pcaAdmins as $pca) {
                try {
                    Mail::to($pca->email)->send(new PaymentReferenceCreated($reference, $totalCount));

                    Log::info('Email sent to PCA', [
                        'reference_id' => $reference->id,
                        'reference_number' => $reference->reference_number,
                        'pca_email' => $pca->email,
                        'pca_name' => $pca->name,
                    ]);
                } catch (\Exception $e) {
                    Log::error('Error sending email to PCA', [
                        'reference_id' => $reference->id,
                        'pca_email' => $pca->email,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            return true;

        } catch (\Exception $e) {
            Log::error('Error in sendPcaEmailNotification', [
                'reference_id' => $reference->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Converte nome ou número do mês para português
     */
    private function getPortugueseMonth($monthIdentifier)
    {
        // Se for número, converter diretamente
        if (is_numeric($monthIdentifier)) {
            $monthsNumeric = [
                1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
                5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
                9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
            ];
            return $monthsNumeric[(int)$monthIdentifier] ?? $monthIdentifier;
        }

        // Mapeamento português → português (capitalizado)
        $portugueseMonths = [
            'janeiro' => 'Janeiro', 'fevereiro' => 'Fevereiro', 'março' => 'Março',
            'abril' => 'Abril', 'maio' => 'Maio', 'junho' => 'Junho',
            'julho' => 'Julho', 'agosto' => 'Agosto', 'setembro' => 'Setembro',
            'outubro' => 'Outubro', 'novembro' => 'Novembro', 'dezembro' => 'Dezembro',
        ];

        // Mapeamento inglês → português
        $englishToPortuguese = [
            'january' => 'Janeiro', 'february' => 'Fevereiro', 'march' => 'Março',
            'april' => 'Abril', 'may' => 'Maio', 'june' => 'Junho',
            'july' => 'Julho', 'august' => 'Agosto', 'september' => 'Setembro',
            'october' => 'Outubro', 'november' => 'Novembro', 'december' => 'Dezembro',
        ];

        $monthLower = strtolower($monthIdentifier);

        // Tentar português primeiro
        if (isset($portugueseMonths[$monthLower])) {
            return $portugueseMonths[$monthLower];
        }

        // Tentar inglês
        if (isset($englishToPortuguese[$monthLower])) {
            return $englishToPortuguese[$monthLower];
        }

        // Fallback: capitalizar primeira letra
        return ucfirst($monthIdentifier);
    }

    private function loadStats()
    {
        try {
            $this->stats = [
                'total_generated' => PaymentReference::count(),
                'total_pending' => PaymentReference::where('status', 'pending')->count(),
                'total_paid' => PaymentReference::where('status', 'paid')->count(),
                'total_expired' => PaymentReference::where('status', 'pending')
                    ->where('expires_at', '<', now())
                    ->count(),
            ];
        } catch (\Exception $e) {
            Log::error('Erro ao carregar estatísticas', ['error' => $e->getMessage()]);
            $this->stats = [
                'total_generated' => 0,
                'total_pending' => 0,
                'total_paid' => 0,
                'total_expired' => 0,
            ];
        }
    }

    /* ==================== COMPUTED PROPERTIES ==================== */

    public function getSelectedStudentProperty()
    {
        if (!$this->selectedStudentId) {
            return null;
        }
        return User::find($this->selectedStudentId);
    }

    public function getSearchResultsProperty()
    {
        if (strlen($this->searchTerm) < 2) {
            return collect([]);
        }

        return User::where('is_active', 1)
            ->where(function ($query) {
                $query->where('name', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('student_id', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('email', 'like', '%' . $this->searchTerm . '%');
            })
            ->limit(10)
            ->get();
    }

    public function getRecentReferencesProperty()
    {
        return PaymentReference::with('student')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
    }

    public function getAllReferencesProperty()
    {
        $query = PaymentReference::with('student')
            ->orderBy('created_at', 'desc');

        // Filtro por status
        if ($this->filterStatus) {
            if ($this->filterStatus === 'expired') {
                $query->where('status', 'pending')
                      ->where('expires_at', '<', now());
            } else {
                $query->where('status', $this->filterStatus);
            }
        }

        // Filtro por ano acadêmico
        if ($this->filterAcademicYear) {
            $query->where('academic_year', $this->filterAcademicYear);
        }

        // Filtro por classe
        if ($this->filterClasse) {
            $query->where('classe', $this->filterClasse);
        }

        // Filtro por turma
        if ($this->filterTurma) {
            $query->where('turma', $this->filterTurma);
        }

        // Filtro por busca
        if ($this->filterSearch) {
            $query->where(function($q) {
                $q->where('reference_number', 'like', '%' . $this->filterSearch . '%')
                  ->orWhere('entity_code', 'like', '%' . $this->filterSearch . '%')
                  ->orWhereHas('student', function($sq) {
                      $sq->where('name', 'like', '%' . $this->filterSearch . '%')
                        ->orWhere('student_id', 'like', '%' . $this->filterSearch . '%');
                  });
            });
        }

        return $query->paginate(20);
    }

    // Retorna lista de anos acadêmicos disponíveis
    public function getAcademicYearsProperty()
    {
        return PaymentReference::select('academic_year')
            ->whereNotNull('academic_year')
            ->distinct()
            ->orderBy('academic_year', 'desc')
            ->pluck('academic_year');
    }

    // Retorna lista de classes disponíveis
    public function getClassesProperty()
    {
        return PaymentReference::select('classe')
            ->whereNotNull('classe')
            ->distinct()
            ->orderBy('classe')
            ->pluck('classe');
    }

    // Retorna lista de turmas disponíveis
    public function getTurmasProperty()
    {
        return PaymentReference::select('turma')
            ->whereNotNull('turma')
            ->distinct()
            ->orderBy('turma')
            ->pluck('turma');
    }

    public function clearFilters()
    {
        $this->filterStatus = '';
        $this->filterSearch = '';
        $this->filterAcademicYear = '';
        $this->filterClasse = '';
        $this->filterTurma = '';
    }

    /**
     * Cancela/Anula uma referência pendente
     */
    public function cancelReference($referenceId)
    {
        try {
            $reference = PaymentReference::find($referenceId);

            if (!$reference) {
                session()->flash('error', 'Referência não encontrada.');
                return;
            }

            if ($reference->status === 'paid') {
                session()->flash('error', 'Não é possível cancelar uma referência já paga.');
                return;
            }

            $reference->update([
                'status' => 'cancelled',
                'cancelled_at' => now(),
            ]);

            Log::info('Referência cancelada', [
                'reference_id' => $reference->id,
                'reference_number' => $reference->reference_number,
                'cancelled_by' => auth()->id(),
            ]);

            session()->flash('success', 'Referência ' . $reference->reference_number . ' cancelada com sucesso.');
            $this->loadStats();

        } catch (\Exception $e) {
            Log::error('Erro ao cancelar referência', [
                'reference_id' => $referenceId,
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao cancelar referência: ' . $e->getMessage());
        }
    }

    /**
     * Elimina permanentemente uma referência (apenas pendentes/canceladas)
     */
    public function deleteReference($referenceId)
    {
        try {
            $reference = PaymentReference::find($referenceId);

            if (!$reference) {
                session()->flash('error', 'Referência não encontrada.');
                return;
            }

            if ($reference->status === 'paid') {
                session()->flash('error', 'Não é possível eliminar uma referência já paga.');
                return;
            }

            $refNumber = $reference->reference_number;
            $reference->delete();

            Log::info('Referência eliminada', [
                'reference_number' => $refNumber,
                'deleted_by' => auth()->id(),
            ]);

            session()->flash('success', 'Referência ' . $refNumber . ' eliminada permanentemente.');
            $this->loadStats();

        } catch (\Exception $e) {
            Log::error('Erro ao eliminar referência', [
                'reference_id' => $referenceId,
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao eliminar referência: ' . $e->getMessage());
        }
    }

    public function exportToExcel()
    {
        try {
            $filters = [
                'status' => $this->filterStatus,
                'search' => $this->filterSearch,
                'academic_year' => $this->filterAcademicYear,
                'classe' => $this->filterClasse,
                'turma' => $this->filterTurma,
            ];

            $fileName = 'referencias_pagamento_' . now()->format('Y-m-d_His') . '.xlsx';

            return Excel::download(new PaymentReferencesExport($filters), $fileName);

        } catch (\Exception $e) {
            Log::error('Erro ao exportar referências', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            session()->flash('error', 'Erro ao exportar: ' . $e->getMessage());
        }
    }

    /* ==================== IMPORTAÇÃO VIA EXCEL ==================== */

    public function importFromExcel()
    {
        $this->validate([
            'importFile' => 'required|mimes:xlsx,xls,csv|max:10240', // Max 10MB
        ]);

        try {
            DB::beginTransaction();

            $import = new PaymentReferencesImport($this->entityCode);

            Excel::import($import, $this->importFile->getRealPath());

            $errors = $import->errors();
            $successCount = PaymentReference::where('created_at', '>=', now()->subMinutes(1))->count();

            DB::commit();

            $this->importResults = [
                'success' => $successCount,
                'errors' => count($errors),
                'error_details' => $errors,
            ];

            session()->flash('success',
                "Importação concluída! {$successCount} referências criadas com sucesso." .
                (count($errors) > 0 ? " " . count($errors) . " erros encontrados." : "")
            );

            $this->reset(['importFile']);
            $this->loadStats();

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Erro ao importar referências', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            session()->flash('error', 'Erro ao importar arquivo: ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $fileName = 'template_importacao_referencias_' . now()->format('Y-m-d') . '.xlsx';

        return Excel::download(new PaymentReferencesTemplateExport(), $fileName);
    }

    public function render()
    {
        // Lista de classes disponíveis
        $availableClasses = Classroom::select('class')
            ->distinct()
            ->orderByRaw('CAST(class AS UNSIGNED)')
            ->pluck('class');

        // Buscar apenas taxas manuais (is_manual = true) do FeeStructure
        $manualFeeStructures = FeeStructure::where('is_manual', true)
            ->where('active', true)
            ->get()
            ->map(function($structure) {
                return (object)[
                    'id' => 'fs_' . $structure->id,
                    'name' => $structure->fee_name,
                    'description' => $structure->fee_name,
                    'amount' => $structure->monthly_fee,
                    'fine_type' => $structure->penalty_type,
                    'percentage' => $structure->late_penalty_percentage,
                    'fixed_amount' => null,
                    'type' => 'fee_structure',
                    'original' => $structure,
                ];
            });

        // Agrupar por categoria
        $feeGroupsGrouped = collect([
            'Plano de Taxas' => $manualFeeStructures,
        ]);

        return view('livewire.admin.fee.generate-references', [
            'classrooms' => Classroom::orderBy('name')->get(),
            'availableClasses' => $availableClasses,
            'feeGroupsGrouped' => $feeGroupsGrouped,
            'searchResults' => $this->search_results,
            'recentReferences' => $this->recent_references,
            'allReferences' => $this->all_references,
            'academicYears' => $this->academic_years,
            'classes' => $this->classes,
            'turmas' => $this->turmas,
            'stats' => $this->stats,
        ]);
    }
}
