<?php

namespace App\Http\Livewire\Admin\Fee;

use App\Models\User;
use App\Models\Fee_assign;
use App\Services\FeeCalculationService;
use App\Exports\FeePaymentsTemplateExport;
use Livewire\Component;
use Livewire\WithFileUploads;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ImportPayments extends Component
{
    use WithFileUploads;

    public $step = 1;
    public $excel_file;
    public $preview_data = [];
    public $valid_count = 0;
    public $invalid_count = 0;
    public $duplicate_count = 0;
    public $import_result = null;

    protected $feeCalculationService;

    protected $monthNames = [
        1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
        5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
        9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
    ];

    protected $paymentModes = [
        'cash' => 'Cash', 'dinheiro' => 'Cash',
        'bank' => 'Bank', 'banco' => 'Bank', 'transferencia' => 'Bank',
        'mobile' => 'Mobile', 'mpesa' => 'Mobile', 'emola' => 'Mobile',
        'pos' => 'POS',
    ];

    public function boot(FeeCalculationService $feeCalculationService)
    {
        $this->feeCalculationService = $feeCalculationService;
    }

    public function downloadTemplate()
    {
        return Excel::download(
            new FeePaymentsTemplateExport(),
            'template_importacao_pagamentos.xlsx'
        );
    }

    public function updatedExcelFile()
    {
        $this->validate(['excel_file' => 'required|file|mimes:xlsx,xls|max:10240']);
        $this->parseAndPreview();
    }

    private function parseAndPreview()
    {
        $filePath = $this->excel_file->getRealPath();
        $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($filePath);
        $worksheet = $spreadsheet->getActiveSheet();
        $rows = $worksheet->toArray(null, true, true, true);

        $this->preview_data = [];
        $this->valid_count = 0;
        $this->invalid_count = 0;
        $this->duplicate_count = 0;

        // Pre-load students for performance
        $studentIds = [];
        $isFirst = true;
        foreach ($rows as $row) {
            if ($isFirst) { $isFirst = false; continue; }
            $values = array_values($row);
            if (!empty(trim($values[0] ?? ''))) {
                $studentIds[] = trim($values[0]);
            }
        }
        $students = User::whereIn('student_id', array_unique($studentIds))->get()->keyBy('student_id');

        $isFirst = true;
        foreach ($rows as $rowIndex => $row) {
            if ($isFirst) { $isFirst = false; continue; }

            $parsed = $this->parseRow($row);
            if (!$parsed) continue;

            $student = $students->get($parsed['numero_aluno']);
            $validation = $this->validateRow($parsed, $student);

            $parsed['row_number'] = $rowIndex;
            $parsed['status'] = $validation['status'];
            $parsed['errors'] = $validation['errors'];
            $parsed['student_name'] = $validation['student_name'] ?? '-';

            $this->preview_data[] = $parsed;

            if ($validation['status'] === 'valid') $this->valid_count++;
            elseif ($validation['status'] === 'duplicate') $this->duplicate_count++;
            else $this->invalid_count++;
        }

        $this->step = 2;
    }

    private function parseRow(array $row): ?array
    {
        $values = array_values($row);
        if (empty(array_filter($values))) return null;
        if (empty(trim($values[0] ?? ''))) return null;

        return [
            'numero_aluno' => trim($values[0] ?? ''),
            'mes' => intval($values[1] ?? 0),
            'ano' => intval($values[2] ?? 0),
            'valor' => floatval(str_replace(',', '.', $values[3] ?? 0)),
            'data_pagamento' => trim($values[4] ?? ''),
            'modo_pagamento' => trim($values[5] ?? 'Cash'),
            'observacao' => trim($values[6] ?? ''),
        ];
    }

    private function validateRow(array $row, ?User $student): array
    {
        $errors = [];
        $status = 'valid';
        $studentName = null;

        if (!$student) {
            $errors[] = 'Estudante não encontrado: ' . $row['numero_aluno'];
            $status = 'invalid';
        } else {
            $studentName = $student->name;
        }

        if ($row['mes'] < 1 || $row['mes'] > 12) {
            $errors[] = 'Mês inválido (deve ser 01-12)';
            $status = 'invalid';
        }

        if ($row['ano'] < 2020 || $row['ano'] > 2030) {
            $errors[] = 'Ano inválido';
            $status = 'invalid';
        }

        if ($row['valor'] <= 0) {
            $errors[] = 'Valor deve ser positivo';
            $status = 'invalid';
        }

        try {
            Carbon::createFromFormat('d/m/Y', $row['data_pagamento']);
        } catch (\Exception $e) {
            $errors[] = 'Data inválida (usar dd/mm/aaaa)';
            $status = 'invalid';
        }

        // Check duplicate only if other validations passed
        if ($status === 'valid' && $student && isset($this->monthNames[$row['mes']])) {
            $monthName = $this->monthNames[$row['mes']];
            if ($this->feeCalculationService->hasPaymentForPeriod($student, $monthName, $row['ano'])) {
                $errors[] = 'Já existe pagamento para ' . $monthName . '/' . $row['ano'];
                $status = 'duplicate';
            }
        }

        return ['status' => $status, 'errors' => $errors, 'student_name' => $studentName];
    }

    public function confirmImport()
    {
        $imported = 0;
        $skipped = 0;

        DB::beginTransaction();
        try {
            foreach ($this->preview_data as $row) {
                if ($row['status'] !== 'valid') {
                    $skipped++;
                    continue;
                }

                $student = User::where('student_id', $row['numero_aluno'])->first();
                if (!$student) {
                    $skipped++;
                    continue;
                }

                $monthName = $this->monthNames[$row['mes']];

                // Re-check duplicate at import time
                if ($this->feeCalculationService->hasPaymentForPeriod($student, $monthName, $row['ano'])) {
                    $skipped++;
                    continue;
                }

                $paymentDate = Carbon::createFromFormat('d/m/Y', $row['data_pagamento']);
                $paymentMode = $this->normalizePaymentMode($row['modo_pagamento']);

                $this->feeCalculationService->createFeePayment($student, [
                    'month' => $monthName,
                    'year' => $row['ano'],
                    'amount' => $row['valor'],
                    'fine' => 0,
                    'discount' => 0,
                    'payment_mode' => $paymentMode,
                    'pay_type' => strtolower($paymentMode),
                    'transaction_id' => null,
                    'reference_number' => null,
                    'payment_date' => $paymentDate,
                ]);

                $imported++;
            }

            DB::commit();

            $this->import_result = [
                'success' => true,
                'imported' => $imported,
                'skipped' => $skipped,
                'total' => count($this->preview_data),
            ];

            Log::info('Bulk payment import completed', $this->import_result);
            $this->step = 3;

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Bulk payment import failed', ['error' => $e->getMessage()]);
            session()->flash('error', 'Erro na importação: ' . $e->getMessage());
        }
    }

    public function resetImport()
    {
        $this->step = 1;
        $this->excel_file = null;
        $this->preview_data = [];
        $this->import_result = null;
        $this->valid_count = 0;
        $this->invalid_count = 0;
        $this->duplicate_count = 0;
    }

    private function normalizePaymentMode(string $input): string
    {
        return $this->paymentModes[strtolower(trim($input))] ?? 'Cash';
    }

    public function render()
    {
        return view('livewire.admin.fee.import-payments');
    }
}
