<?php

//namespace App\Http\Livewire\Admin\Reports;
//
//use App\Models\FeeAssign;
//use App\Models\FeeStructure;
//use App\Models\User;
//use Carbon\Carbon;
//use Livewire\Component;
//use Livewire\WithPagination;
//use Maatwebsite\Excel\Facades\Excel;
//use Barryvdh\DomPDF\Facade\Pdf;
//use Illuminate\Support\Facades\DB;
//use App\Exports\FeeCollectionExport;
//
//class FeeReportController extends Component
//{
//    use WithPagination;
//
//    protected $paginationTheme = 'bootstrap';
//
//    // Filtros
//    public $start_date;
//    public $end_date;
//    public $year;
//    public $grade;
//    public $fee_type;
//    public $status;
//    public $pay_type;
//    public $searchStudent;
//    public $quickPeriod;
//    public $tableSearch;
//
//    // Dados
//    public $feesCollection = null;
//    public $feeTypes;
//    public $grades;
//    
//    // Estatísticas
//    public $totalCollected = 0;
//    public $totalPending = 0;
//    public $totalDiscounts = 0;
//    public $totalFines = 0;
//    public $percentageIncrease = 0;
//    public $pendingCount = 0;
//    public $discountCount = 0;
//    public $fineCount = 0;
//
//    // Seleção
//    public $selectedFees = [];
//    public $selectAll = false;
//    public $selectedFeeDetails = null;
//
//    protected $listeners = [
//        'refreshComponent' => '$refresh',
//        'resetSelection' => 'resetSelection'
//    ];
//
//    protected $rules = [
//        'start_date' => 'nullable|date',
//        'end_date' => 'nullable|date|after_or_equal:start_date',
//        'year' => 'nullable|integer|min:2020|max:2030',
//        'grade' => 'nullable|integer|min:1|max:12',
//    ];
//
//    public function mount()
//    {
//        $this->initializeFilters();
//        $this->loadFeeTypes();
//        $this->loadGrades();
//        $this->calculateStatistics();
//    }
//
//    public function initializeFilters()
//    {
//        $this->start_date = Carbon::now()->startOfMonth()->format('Y-m-d');
//        $this->end_date = Carbon::now()->format('Y-m-d');
//        $this->year = date('Y');
//        $this->quickPeriod = 'this_month';
//    }
//
//    public function loadFeeTypes()
//    {
//        $this->feeTypes = FeeStructure::where('active', 1)
//            ->select('fee_name')
//            ->distinct()
//            ->get();
//    }
//
//    public function loadGrades()
//    {
//        // Carregar as classes disponíveis da estrutura de taxas
//        $gradesString = FeeStructure::where('active', 1)
//            ->pluck('grades')
//            ->first();
//            
//        $this->grades = $gradesString ? explode(',', $gradesString) : range(1, 12);
//    }
//
//    public function updatedQuickPeriod($value)
//    {
//        if (!$value) return;
//
//        $now = Carbon::now();
//        
//        switch($value) {
//            case 'today':
//                $this->start_date = $now->format('Y-m-d');
//                $this->end_date = $now->format('Y-m-d');
//                break;
//            case 'yesterday':
//                $this->start_date = $now->subDay()->format('Y-m-d');
//                $this->end_date = $now->format('Y-m-d');
//                break;
//            case 'this_week':
//                $this->start_date = $now->startOfWeek()->format('Y-m-d');
//                $this->end_date = $now->endOfWeek()->format('Y-m-d');
//                break;
//            case 'last_week':
//                $this->start_date = $now->subWeek()->startOfWeek()->format('Y-m-d');
//                $this->end_date = $now->subWeek()->endOfWeek()->format('Y-m-d');
//                break;
//            case 'this_month':
//                $this->start_date = $now->startOfMonth()->format('Y-m-d');
//                $this->end_date = $now->endOfMonth()->format('Y-m-d');
//                break;
//            case 'last_month':
//                $this->start_date = $now->subMonth()->startOfMonth()->format('Y-m-d');
//                $this->end_date = $now->subMonth()->endOfMonth()->format('Y-m-d');
//                break;
//            case 'this_year':
//                $this->start_date = $now->startOfYear()->format('Y-m-d');
//                $this->end_date = $now->endOfYear()->format('Y-m-d');
//                break;
//            case 'last_year':
//                $this->start_date = $now->subYear()->startOfYear()->format('Y-m-d');
//                $this->end_date = $now->subYear()->endOfYear()->format('Y-m-d');
//                break;
//        }
//        
//        $this->generateReport();
//    }
//
//    public function generateReport()
//    {
//        $this->validate();
//        
//        $query = FeeAssign::with(['user', 'feeStructure']);
//
//        // Aplicar filtros
//        if ($this->start_date && $this->end_date) {
//            $query->whereBetween('pay_day', [$this->start_date, $this->end_date]);
//        }
//
//        if ($this->year) {
//            $query->where('year', $this->year);
//        }
//
//        if ($this->grade) {
//            $query->whereHas('feeStructure', function($q) {
//                $q->where('grade', $this->grade);
//            });
//        }
//
//        if ($this->fee_type) {
//            $query->whereHas('feeStructure', function($q) {
//                $q->where('fee_name', $this->fee_type);
//            });
//        }
//
//        if ($this->status) {
//            $query->where('status', $this->status);
//        }
//
//        if ($this->pay_type) {
//            $query->where('pay_type', $this->pay_type);
//        }
//
//        if ($this->searchStudent) {
//            $query->whereHas('user', function($q) {
//                $q->where('name', 'like', '%' . $this->searchStudent . '%')
//                  ->orWhere('id', $this->searchStudent);
//            });
//        }
//
//        // Aplicar busca na tabela
//        if ($this->tableSearch) {
//            $query->where(function($q) {
//                $q->where('id', 'like', '%' . $this->tableSearch . '%')
//                  ->orWhere('student_id', 'like', '%' . $this->tableSearch . '%')
//                  ->orWhereHas('user', function($subQ) {
//                      $subQ->where('name', 'like', '%' . $this->tableSearch . '%');
//                  });
//            });
//        }
//
//        $this->feesCollection = $query->orderBy('pay_day', 'desc')->paginate(20);
//        
//        $this->calculateStatistics();
//        $this->resetSelection();
//        
//        $this->dispatch('show-notification', [
//            'type' => 'success',
//            'title' => 'Relatório Gerado',
//            'message' => 'O relatório foi gerado com sucesso!'
//        ]);
//    }
//
//    public function calculateStatistics()
//    {
//        // Total coletado no período
//        $query = FeeAssign::where('status', 'paid');
//        
//        if ($this->start_date && $this->end_date) {
//            $query->whereBetween('pay_day', [$this->start_date, $this->end_date]);
//        }
//        
//        $this->totalCollected = $query->sum(DB::raw('amount - discount + fine'));
//        
//        // Total pendente
//        $pendingQuery = FeeAssign::whereIn('status', ['unpaid', 'partial']);
//        
//        if ($this->start_date && $this->end_date) {
//            $pendingQuery->whereBetween('created_at', [$this->start_date, $this->end_date]);
//        }
//        
//        $this->totalPending = $pendingQuery->sum('amount');
//        $this->pendingCount = $pendingQuery->count();
//        
//        // Total de descontos
//        $discountQuery = FeeAssign::where('discount', '>', 0);
//        
//        if ($this->start_date && $this->end_date) {
//            $discountQuery->whereBetween('pay_day', [$this->start_date, $this->end_date]);
//        }
//        
//        $this->totalDiscounts = $discountQuery->sum('discount');
//        $this->discountCount = $discountQuery->count();
//        
//        // Total de multas
//        $fineQuery = FeeAssign::where('fine', '>', 0);
//        
//        if ($this->start_date && $this->end_date) {
//            $fineQuery->whereBetween('pay_day', [$this->start_date, $this->end_date]);
//        }
//        
//        $this->totalFines = $fineQuery->sum('fine');
//        $this->fineCount = $fineQuery->count();
//        
//        // Calcular percentual de aumento
//        $lastMonthTotal = FeeAssign::where('status', 'paid')
//            ->whereBetween('pay_day', [
//                Carbon::now()->subMonth()->startOfMonth(),
//                Carbon::now()->subMonth()->endOfMonth()
//            ])
//            ->sum(DB::raw('amount - discount + fine'));
//            
//        if ($lastMonthTotal > 0) {
//            $this->percentageIncrease = round((($this->totalCollected - $lastMonthTotal) / $lastMonthTotal) * 100, 2);
//        } else {
//            $this->percentageIncrease = 0;
//        }
//    }
//
//    public function resetFilters()
//    {
//        $this->reset([
//            'start_date', 'end_date', 'year', 'grade',
//            'fee_type', 'status', 'pay_type', 'searchStudent',
//            'quickPeriod', 'tableSearch'
//        ]);
//        
//        $this->initializeFilters();
//        $this->feesCollection = null;
//        $this->calculateStatistics();
//    }
//
//    public function updatedSelectAll($value)
//    {
//        if ($value) {
//            $this->selectedFees = $this->feesCollection->pluck('id')->toArray();
//        } else {
//            $this->selectedFees = [];
//        }
//    }
//
//    public function resetSelection()
//    {
//        $this->selectedFees = [];
//        $this->selectAll = false;
//    }
//
//    public function viewDetails($feeId)
//    {
//        $this->selectedFeeDetails = FeeAssign::with(['user', 'feeStructure'])->find($feeId);
//        $this->dispatch('show-details-modal');
//    }
//
//    public function printReceipt($feeId)
//    {
//        $fee = FeeAssign::with(['user', 'feeStructure'])->find($feeId);
//        
//        if (!$fee) {
//            $this->dispatch('show-notification', [
//                'type' => 'error',
//                'title' => 'Erro',
//                'message' => 'Pagamento não encontrado!'
//            ]);
//            return;
//        }
//
//        // Gerar PDF do recibo
//        $pdf = Pdf::loadView('reports.fee-receipt', compact('fee'));
//        
//        return response()->streamDownload(function() use ($pdf) {
//            echo $pdf->output();
//        }, 'recibo_' . $fee->id . '.pdf');
//    }
//
//    public function exportPDF()
//    {
//        if (!$this->feesCollection || $this->feesCollection->isEmpty()) {
//            $this->dispatch('show-notification', [
//                'type' => 'warning',
//                'title' => 'Atenção',
//                'message' => 'Nenhum dado para exportar!'
//            ]);
//            return;
//        }
//
//        $data = [
//            'fees' => $this->feesCollection,
//            'filters' => [
//                'start_date' => $this->start_date,
//                'end_date' => $this->end_date,
//                'grade' => $this->grade,
//                'status' => $this->status
//            ],
//            'statistics' => [
//                'total_collected' => $this->totalCollected,
//                'total_pending' => $this->totalPending,
//                'total_discounts' => $this->totalDiscounts,
//                'total_fines' => $this->totalFines
//            ]
//        ];
//
//        $pdf = Pdf::loadView('reports.fee-collection-pdf', $data);
//        $pdf->setPaper('A4', 'landscape');
//        
//        return response()->streamDownload(function() use ($pdf) {
//            echo $pdf->output();
//        }, 'relatorio_taxas_' . date('Y-m-d') . '.pdf');
//    }
//
//    public function exportExcel()
//    {
//        if (!$this->feesCollection || $this->feesCollection->isEmpty()) {
//            $this->dispatch('show-notification', [
//                'type' => 'warning',
//                'title' => 'Atenção',
//                'message' => 'Nenhum dado para exportar!'
//            ]);
//            return;
//        }
//
//        return Excel::download(new FeeCollectionExport($this->feesCollection),
//            'relatorio_taxas_' . date('Y-m-d') . '.xlsx');
//    }
//
//    public function exportSelected()
//    {
//        if (empty($this->selectedFees)) {
//            $this->dispatch('show-notification', [
//                'type' => 'warning',
//                'title' => 'Atenção',
//                'message' => 'Selecione pelo menos um registro!'
//            ]);
//            return;
//        }
//
//        $selected = FeeAssign::with(['user', 'feeStructure'])
//            ->whereIn('id', $this->selectedFees)
//            ->get();
//
//        return Excel::download(new FeeCollectionExport($selected),
//            'taxas_selecionadas_' . date('Y-m-d') . '.xlsx');
//    }
//
//    public function printSelected()
//    {
//        if (empty($this->selectedFees)) {
//            $this->dispatch('show-notification', [
//                'type' => 'warning',
//                'title' => 'Atenção',
//                'message' => 'Selecione pelo menos um registro!'
//            ]);
//            return;
//        }
//
//        $selected = FeeAssign::with(['user', 'feeStructure'])
//            ->whereIn('id', $this->selectedFees)
//            ->get();
//
//        $pdf = Pdf::loadView('reports.fee-collection-selected', compact('selected'));
//        
//        return response()->streamDownload(function() use ($pdf) {
//            echo $pdf->output();
//        }, 'taxas_selecionadas_' . date('Y-m-d') . '.pdf');
//    }
//
//    public function sendReminder($feeId)
//    {
//        $fee = FeeAssign::with('user')->find($feeId);
//        
//        if (!$fee || !$fee->user) {
//            $this->dispatch('show-notification', [
//                'type' => 'error',
//                'title' => 'Erro',
//                'message' => 'Dados do estudante não encontrados!'
//            ]);
//            return;
//        }
//
//        // Aqui você implementaria o envio de e-mail/SMS
//        // Por exemplo:
//        // Mail::to($fee->user->email)->send(new FeeReminderMail($fee));
//        
//        $this->dispatch('show-notification', [
//            'type' => 'success',
//            'title' => 'Lembrete Enviado',
//            'message' => 'Lembrete enviado para ' . $fee->user->name
//        ]);
//    }
//
//    public function sendBulkReminders()
//    {
//        if (empty($this->selectedFees)) {
//            $this->dispatch('show-notification', [
//                'type' => 'warning',
//                'title' => 'Atenção',
//                'message' => 'Selecione pelo menos um registro!'
//            ]);
//            return;
//        }
//
//        $fees = FeeAssign::with('user')
//            ->whereIn('id', $this->selectedFees)
//            ->where('status', '!=', 'paid')
//            ->get();
//
//        $count = 0;
//        foreach ($fees as $fee) {
//            if ($fee->user && $fee->user->email) {
//                // Enviar lembrete
//                // Mail::to($fee->user->email)->queue(new FeeReminderMail($fee));
//                $count++;
//            }
//        }
//
//        $this->dispatch('show-notification', [
//            'type' => 'success',
//            'title' => 'Lembretes Enviados',
//            'message' => $count . ' lembretes foram enviados com sucesso!'
//        ]);
//        
//        $this->resetSelection();
//    }
//
//    public function print()
//    {
//        $this->dispatch('print-page');
//    }


namespace App\Http\Livewire\Admin\Reports;

use App\Models\FeeAssign;
use App\Models\FeeStructure;
use App\Models\User;
use Carbon\Carbon;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class FeeReportController extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    // Filtros
    public $start_date;
    public $end_date;
    public $year;
    public $grade;
    public $fee_type;
    public $status;
    public $pay_type;
    public $searchStudent;
    public $quickPeriod;
    public $tableSearch;

    // Dados
    public $feesCollection = null;
    public $feeTypes = [];
    public $grades = [];
    
    // Estatísticas
    public $totalCollected = 0;
    public $totalPending = 0;
    public $totalDiscounts = 0;
    public $totalFines = 0;
    public $percentageIncrease = 0;
    public $pendingCount = 0;
    public $discountCount = 0;
    public $fineCount = 0;

    // Seleção
    public $selectedFees = [];
    public $selectAll = false;
    public $selectedFeeDetails = null;

    // Debug
    public $isLoading = false;
    public $debugMessage = '';

    protected $listeners = [
        'refreshComponent' => '$refresh',
        'resetSelection' => 'resetSelection'
    ];

    protected $rules = [
        'start_date' => 'nullable|date',
        'end_date' => 'nullable|date|after_or_equal:start_date',
        'year' => 'nullable|integer|min:2020|max:2030',
        'grade' => 'nullable|integer|min:1|max:12',
    ];

    public function mount()
    {
        try {
            $this->debugMessage = "Iniciando componente...";
            $this->initializeFilters();
            $this->loadFeeTypes();
            $this->loadGrades();
            // Removido calculateStatistics() do mount para evitar query pesada inicial
            $this->debugMessage = "Componente iniciado com sucesso!";
        } catch (\Exception $e) {
            Log::error('Erro no mount: ' . $e->getMessage());
            $this->debugMessage = "Erro: " . $e->getMessage();
        }
    }

    public function initializeFilters()
    {
        $this->start_date = Carbon::now()->startOfMonth()->format('Y-m-d');
        $this->end_date = Carbon::now()->format('Y-m-d');
        $this->year = date('Y');
        $this->quickPeriod = 'this_month';
    }

    public function loadFeeTypes()
    {
        try {
            // Verificar se a tabela existe
            if (!DB::getSchemaBuilder()->hasTable('fee_structures')) {
                $this->feeTypes = [];
                $this->debugMessage = "Tabela fee_structures não encontrada";
                return;
            }

            $this->feeTypes = FeeStructure::where('active', 1)
                ->select('fee_name')
                ->distinct()
                ->limit(50) // Limitar resultados
                ->get();
        } catch (\Exception $e) {
            Log::error('Erro ao carregar tipos de taxa: ' . $e->getMessage());
            $this->feeTypes = [];
            $this->debugMessage = "Erro ao carregar tipos: " . $e->getMessage();
        }
    }

    public function loadGrades()
    {
        try {
            // Se não tiver dados da tabela, usar grades padrão
            $this->grades = range(1, 12);
            
            // Tentar carregar da tabela se existir
            if (DB::getSchemaBuilder()->hasTable('fee_structures')) {
                $gradesString = FeeStructure::where('active', 1)
                    ->pluck('grades')
                    ->first();
                    
                if ($gradesString) {
                    $this->grades = explode(',', $gradesString);
                }
            }
        } catch (\Exception $e) {
            Log::error('Erro ao carregar classes: ' . $e->getMessage());
            $this->grades = range(1, 12);
        }
    }

    public function updatedQuickPeriod($value)
    {
        if (!$value) return;

        $now = Carbon::now();
        
        switch($value) {
            case 'today':
                $this->start_date = $now->format('Y-m-d');
                $this->end_date = $now->format('Y-m-d');
                break;
            case 'yesterday':
                $yesterday = Carbon::yesterday();
                $this->start_date = $yesterday->format('Y-m-d');
                $this->end_date = $yesterday->format('Y-m-d');
                break;
            case 'this_week':
                $this->start_date = $now->startOfWeek()->format('Y-m-d');
                $this->end_date = $now->endOfWeek()->format('Y-m-d');
                break;
            case 'last_week':
                $lastWeek = $now->copy()->subWeek();
                $this->start_date = $lastWeek->startOfWeek()->format('Y-m-d');
                $this->end_date = $lastWeek->endOfWeek()->format('Y-m-d');
                break;
            case 'this_month':
                $this->start_date = $now->startOfMonth()->format('Y-m-d');
                $this->end_date = $now->endOfMonth()->format('Y-m-d');
                break;
            case 'last_month':
                $lastMonth = $now->copy()->subMonth();
                $this->start_date = $lastMonth->startOfMonth()->format('Y-m-d');
                $this->end_date = $lastMonth->endOfMonth()->format('Y-m-d');
                break;
            case 'this_year':
                $this->start_date = $now->startOfYear()->format('Y-m-d');
                $this->end_date = $now->endOfYear()->format('Y-m-d');
                break;
            case 'last_year':
                $lastYear = $now->copy()->subYear();
                $this->start_date = $lastYear->startOfYear()->format('Y-m-d');
                $this->end_date = $lastYear->endOfYear()->format('Y-m-d');
                break;
        }
        
        // Não chamar generateReport automaticamente para evitar travamento
        // $this->generateReport();
    }

    public function generateReport()
    {
        try {
            $this->isLoading = true;
            $this->debugMessage = "Gerando relatório...";
            
            // Validar dados
            $this->validate();
            
            // Verificar se a tabela existe
            if (!DB::getSchemaBuilder()->hasTable('fee_assign')) {
                session()->flash('error', 'Tabela fee_assign não encontrada!');
                $this->isLoading = false;
                return;
            }
            
            // Query otimizada
            $query = FeeAssign::query();
            
            // Adicionar relacionamentos apenas se existirem
            if (DB::getSchemaBuilder()->hasTable('users')) {
                $query->with('user:id,name');
            }
            
            if (DB::getSchemaBuilder()->hasTable('fee_structures')) {
                $query->with('feeStructure:id,fee_name,grade');
            }

            // Aplicar filtros
            if ($this->start_date && $this->end_date) {
                $query->whereBetween('pay_day', [$this->start_date, $this->end_date]);
            }

            if ($this->year) {
                $query->where('year', $this->year);
            }

            if ($this->grade && DB::getSchemaBuilder()->hasTable('fee_structures')) {
                $query->whereHas('feeStructure', function($q) {
                    $q->where('grade', $this->grade);
                });
            }

            if ($this->fee_type && DB::getSchemaBuilder()->hasTable('fee_structures')) {
                $query->whereHas('feeStructure', function($q) {
                    $q->where('fee_name', $this->fee_type);
                });
            }

            if ($this->status) {
                $query->where('status', $this->status);
            }

            if ($this->pay_type) {
                $query->where('pay_type', $this->pay_type);
            }

            if ($this->searchStudent && DB::getSchemaBuilder()->hasTable('users')) {
                $query->whereHas('user', function($q) {
                    $q->where('name', 'like', '%' . $this->searchStudent . '%');
                })->orWhere('student_id', 'like', '%' . $this->searchStudent . '%');
            }

            // Limitar resultados e paginar
            $this->feesCollection = $query
                ->orderBy('pay_day', 'desc')
                ->limit(1000) // Limitar para evitar sobrecarga
                ->paginate(20);
            
            // Calcular estatísticas de forma otimizada
            $this->calculateStatisticsOptimized();
            $this->resetSelection();
            
            $this->debugMessage = "Relatório gerado com sucesso!";
            
            // Emitir evento de sucesso
            $this->dispatch('show-notification', [
                'type' => 'success',
                'title' => 'Sucesso',
                'message' => 'Relatório gerado com sucesso!'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Erro ao gerar relatório: ' . $e->getMessage());
            $this->debugMessage = "Erro: " . $e->getMessage();
            
            session()->flash('error', 'Erro ao gerar relatório: ' . $e->getMessage());
            
            $this->dispatch('show-notification', [
                'type' => 'error',
                'title' => 'Erro',
                'message' => 'Erro ao gerar relatório: ' . $e->getMessage()
            ]);
        } finally {
            $this->isLoading = false;
        }
    }

    public function calculateStatisticsOptimized()
    {
        try {
            // Usar queries agregadas para melhor performance
            $statsQuery = FeeAssign::query();
            
            if ($this->start_date && $this->end_date) {
                $statsQuery->whereBetween('pay_day', [$this->start_date, $this->end_date]);
            }
            
            // Uma única query para múltiplas estatísticas
            $stats = $statsQuery->selectRaw('
                SUM(CASE WHEN status = "paid" THEN (COALESCE(amount, 0) - COALESCE(discount, 0) + COALESCE(fine, 0)) ELSE 0 END) as total_collected,
                SUM(CASE WHEN status IN ("unpaid", "partial") THEN COALESCE(amount, 0) ELSE 0 END) as total_pending,
                COUNT(CASE WHEN status IN ("unpaid", "partial") THEN 1 END) as pending_count,
                SUM(COALESCE(discount, 0)) as total_discounts,
                COUNT(CASE WHEN discount > 0 THEN 1 END) as discount_count,
                SUM(COALESCE(fine, 0)) as total_fines,
                COUNT(CASE WHEN fine > 0 THEN 1 END) as fine_count
            ')->first();
            
            $this->totalCollected = $stats->total_collected ?? 0;
            $this->totalPending = $stats->total_pending ?? 0;
            $this->pendingCount = $stats->pending_count ?? 0;
            $this->totalDiscounts = $stats->total_discounts ?? 0;
            $this->discountCount = $stats->discount_count ?? 0;
            $this->totalFines = $stats->total_fines ?? 0;
            $this->fineCount = $stats->fine_count ?? 0;
            
            // Calcular percentual de aumento (simplificado)
            $this->percentageIncrease = 0;
            
        } catch (\Exception $e) {
            Log::error('Erro ao calcular estatísticas: ' . $e->getMessage());
            // Definir valores padrão em caso de erro
            $this->totalCollected = 0;
            $this->totalPending = 0;
            $this->totalDiscounts = 0;
            $this->totalFines = 0;
            $this->percentageIncrease = 0;
            $this->pendingCount = 0;
            $this->discountCount = 0;
            $this->fineCount = 0;
        }
    }

    public function resetFilters()
    {
        $this->reset([
            'start_date', 'end_date', 'year', 'grade',
            'fee_type', 'status', 'pay_type', 'searchStudent',
            'quickPeriod', 'tableSearch'
        ]);
        
        $this->initializeFilters();
        $this->feesCollection = null;
        $this->calculateStatisticsOptimized();
    }

    public function updatedSelectAll($value)
    {
        if ($value && $this->feesCollection) {
            $this->selectedFees = $this->feesCollection->pluck('id')->toArray();
        } else {
            $this->selectedFees = [];
        }
    }

    public function resetSelection()
    {
        $this->selectedFees = [];
        $this->selectAll = false;
    }

    public function viewDetails($feeId)
    {
        try {
            $this->selectedFeeDetails = FeeAssign::find($feeId);
            
            if ($this->selectedFeeDetails) {
                // Carregar relacionamentos se existirem
                if (DB::getSchemaBuilder()->hasTable('users')) {
                    $this->selectedFeeDetails->load('user');
                }
                if (DB::getSchemaBuilder()->hasTable('fee_structures')) {
                    $this->selectedFeeDetails->load('feeStructure');
                }
            }
            
            $this->dispatch('show-details-modal');
        } catch (\Exception $e) {
            Log::error('Erro ao visualizar detalhes: ' . $e->getMessage());
        }
    }

    // Stubs para métodos de exportação (implementar depois)
    public function exportPDF()
    {
        session()->flash('info', 'Funcionalidade de exportação PDF em desenvolvimento');
    }

    public function exportExcel()
    {
        session()->flash('info', 'Funcionalidade de exportação Excel em desenvolvimento');
    }

    public function printReceipt($feeId)
    {
        session()->flash('info', 'Funcionalidade de impressão em desenvolvimento');
    }

    public function sendReminder($feeId)
    {
        session()->flash('info', 'Funcionalidade de lembrete em desenvolvimento');
    }

    public function sendBulkReminders()
    {
        session()->flash('info', 'Funcionalidade de lembretes em massa em desenvolvimento');
    }

    public function exportSelected()
    {
        session()->flash('info', 'Funcionalidade de exportação selecionada em desenvolvimento');
    }

    public function printSelected()
    {
        session()->flash('info', 'Funcionalidade de impressão selecionada em desenvolvimento');
    }

    public function print()
    {
        $this->dispatch('print-page');
    }


    public function render()
    {
        return view('livewire.admin.reports.fees_collection_report', [
                'feesCollection' => $this->feesCollection,
            ]);
    }
}
