<?php

namespace App\Http\Livewire\Admin\Settings;

use App\Models\PaymentReminderSetting;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PaymentReminderSettings extends Component
{
    public $settings = [];

    // Form fields
    public $editingId = null;
    public $name = '';
    public $days_before = 2;
    public $message_template = '';
    public $enabled = true;
    public $priority = 1;

    // Global settings
    public $sendFrequency = 1; // 1 ou 2 vezes por dia
    public $sendTime = '09:00';
    public $sendTime2 = '14:00'; // Segundo horário (se frequency = 2)

    // Modal control
    public $showModal = false;
    public $showVariablesModal = false;

    protected $rules = [
        'name' => 'required|string|max:255',
        'days_before' => 'required|integer|min:0',
        'message_template' => 'required|string',
        'enabled' => 'boolean',
        'priority' => 'required|integer|min:1',
    ];

    public function mount()
    {
        $this->loadSettings();
        $this->loadGlobalSettings();
    }

    public function loadGlobalSettings()
    {
        // Carregar configurações de envio do cache ou usar padrão
        $this->sendFrequency = cache('payment_reminders_frequency', 1);
        $this->sendTime = cache('payment_reminders_send_time', '09:00');
        $this->sendTime2 = cache('payment_reminders_send_time_2', '14:00');
    }

    public function loadSettings()
    {
        $this->settings = PaymentReminderSetting::orderBy('priority')->get();
    }

    public function openCreateModal()
    {
        $this->reset(['editingId', 'name', 'days_before', 'message_template', 'enabled', 'priority']);
        $this->enabled = true;
        $this->priority = $this->settings->count() + 1;

        // Template padrão
        $this->message_template = "⚠️ COPMOZ - Lembrete de Pagamento\n\nEstudante: {student_name}\n\nMes: {fee_month}/{fee_year}\nEntidade: {entity_code}\nReferencia: {reference_number}\nValor: {amount} MT\nVence: {due_date}\n\nPague via eBanking, Balcões, ATMs da RedeSIMO e pelo USSD - BCI";

        $this->showModal = true;
    }

    public function openEditModal($id)
    {
        $setting = PaymentReminderSetting::findOrFail($id);

        $this->editingId = $setting->id;
        $this->name = $setting->name;
        $this->days_before = $setting->days_before;
        $this->message_template = $setting->message_template;
        $this->enabled = $setting->enabled;
        $this->priority = $setting->priority;

        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        try {
            DB::beginTransaction();

            if ($this->editingId) {
                // Atualizar
                $setting = PaymentReminderSetting::findOrFail($this->editingId);
                $setting->update([
                    'name' => $this->name,
                    'days_before' => $this->days_before,
                    'message_template' => $this->message_template,
                    'enabled' => $this->enabled,
                    'priority' => $this->priority,
                ]);

                session()->flash('success', 'Lembrete atualizado com sucesso!');
            } else {
                // Criar novo
                PaymentReminderSetting::create([
                    'name' => $this->name,
                    'days_before' => $this->days_before,
                    'message_template' => $this->message_template,
                    'enabled' => $this->enabled,
                    'priority' => $this->priority,
                ]);

                session()->flash('success', 'Lembrete criado com sucesso!');
            }

            DB::commit();

            $this->showModal = false;
            $this->loadSettings();
            $this->reset(['editingId', 'name', 'days_before', 'message_template', 'enabled', 'priority']);

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error saving reminder setting', [
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao salvar: ' . $e->getMessage());
        }
    }

    public function toggleEnabled($id)
    {
        try {
            $setting = PaymentReminderSetting::findOrFail($id);
            $setting->update(['enabled' => !$setting->enabled]);

            $this->loadSettings();
            session()->flash('success', 'Status atualizado!');

        } catch (\Exception $e) {
            Log::error('Error toggling reminder setting', [
                'id' => $id,
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao atualizar status.');
        }
    }

    public function delete($id)
    {
        try {
            DB::beginTransaction();

            $setting = PaymentReminderSetting::findOrFail($id);
            $setting->delete();

            DB::commit();

            $this->loadSettings();
            session()->flash('success', 'Lembrete excluído com sucesso!');

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error deleting reminder setting', [
                'id' => $id,
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao excluir lembrete.');
        }
    }

    public function showVariables()
    {
        $this->showVariablesModal = true;
    }

    public function closeVariablesModal()
    {
        $this->showVariablesModal = false;
    }

    public function saveSendTime()
    {
        try {
            // Validar frequência
            if (!in_array($this->sendFrequency, [1, 2])) {
                session()->flash('error', 'Frequência inválida. Use 1 ou 2 vezes por dia.');
                return;
            }

            // Validar formato do primeiro horário
            if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $this->sendTime)) {
                session()->flash('error', 'Formato do primeiro horário inválido. Use HH:MM');
                return;
            }

            // Se for 2x por dia, validar segundo horário
            if ($this->sendFrequency == 2) {
                if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $this->sendTime2)) {
                    session()->flash('error', 'Formato do segundo horário inválido. Use HH:MM');
                    return;
                }

                // Validar que o segundo horário é depois do primeiro
                if ($this->sendTime2 <= $this->sendTime) {
                    session()->flash('error', 'O segundo horário deve ser posterior ao primeiro.');
                    return;
                }
            }

            // Salvar no cache (permanente)
            cache()->forever('payment_reminders_frequency', $this->sendFrequency);
            cache()->forever('payment_reminders_send_time', $this->sendTime);
            cache()->forever('payment_reminders_send_time_2', $this->sendTime2);

            $message = $this->sendFrequency == 1
                ? "Configurado para enviar 1x por dia às {$this->sendTime}"
                : "Configurado para enviar 2x por dia às {$this->sendTime} e {$this->sendTime2}";

            session()->flash('success', $message);

            Log::info('Payment reminders schedule updated', [
                'frequency' => $this->sendFrequency,
                'send_time' => $this->sendTime,
                'send_time_2' => $this->sendTime2,
                'updated_by' => auth()->id(),
            ]);

        } catch (\Exception $e) {
            Log::error('Error saving send time', [
                'error' => $e->getMessage(),
            ]);
            session()->flash('error', 'Erro ao salvar horário: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.settings.payment-reminder-settings', [
            'availableVariables' => PaymentReminderSetting::getAvailableVariables(),
        ]);
    }
}
