<?php

namespace App\Http\Livewire\Admin\Students;

use App\Models\Admin;
use App\Models\Classroom;
use App\Models\User;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Note\Note;
use Exception;
use App\Services\SmsService;
use App\Services\GoogleVisionOCRService;

class EditStudent extends Component
{
    use FindGuard, LivewireAlert, WithFileUploads;

    public  $user, $model;
    public $name, $classroom_id, $sex,$phone_number,$class, $dob, $email, $student_id;
    public $father_name, $mother_name, $birth_place, $birth_district, $birth_province;
    public $document_id, $document_type, $avatar, $currentAvatar;
    public $generateNewPassword = false;
    public $validatedData;
    public $updateMode = false;

    // Upload opcional de BI para preencher automaticamente
    public $bi_document;
    public $processing_bi = false;
    public $bi_processed = false;
    public $extracted_photo_path = null; // Caminho da foto extraída do BI

    protected $listeners = [
        'confirmed',
        'cancelled',
        'confirmedPasswordGeneration'
    ];

    public function mount($slug)
    {
        $this->model = User::query()->where('slug',$slug)->first();
        $this->name = $this->model->name;
        $this->sex = $this->model->sex;
        $this->dob = $this->model->dob;
        $this->class = $this->model->class->class ?? '';
	    //$this->class = $this->model->class->class;
        $this->student_id = $this->model->student_id;
        $this->email = $this->model->email;
 	    $this->phone_number = $this->model->phone_number;
        $this->classroom_id = $this->model->classroom_id;
        $this->father_name = $this->model->father_name;
        $this->mother_name = $this->model->mother_name;
        $this->birth_place = $this->model->birth_place;
        $this->birth_district = $this->model->birth_district;
        $this->birth_province = $this->model->birth_province;
        $this->document_id = $this->model->document_id;
        $this->document_type = $this->model->document_type;
        $this->currentAvatar = $this->model->avatar;
    }

    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'sex' => ['required', 'string', 'max:255'],
        'classroom_id' => ['required', 'string', 'max:255'],
 	'class' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'max:255'],
        'student_id' => ['required', 'string', 'max:255'],
        'dob' => ['required', 'string', 'max:255'],
  	'phone_number' => ['required', 'numeric'],
        'father_name' => ['nullable', 'string', 'max:255'],
        'mother_name' => ['nullable', 'string', 'max:255'],
        'birth_place' => ['nullable', 'string', 'max:255'],
        'birth_district' => ['nullable', 'string', 'max:255'],
        'birth_province' => ['nullable', 'string', 'max:255'],
        'document_id' => ['nullable', 'string', 'max:255'],
        'document_type' => ['nullable', 'string', 'in:BI,Cedula,Passaporte'],
        'avatar' => ['nullable', 'image', 'max:2048'],
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);

        // Processar BI automaticamente quando for feito upload
        if ($propertyName === 'bi_document') {
            $this->processBiDocument();
        }
    }

    /**
     * Processar BI com OCR e preencher formulário automaticamente
     */
    public function processBiDocument()
    {
        if (!$this->bi_document) {
            return;
        }

        $this->processing_bi = true;

        try {
            // Salvar temporariamente
            $tempPath = $this->bi_document->store('temp', 'public');
            $fullPath = storage_path('app/public/' . $tempPath);

            // Processar com OCR
            $ocrService = new GoogleVisionOCRService();
            $result = $ocrService->processDocument($fullPath);

            if ($result['success']) {
                // Extrair dados do BI
                $extractedData = $ocrService->extractBIData($result['text']);

                // Preencher campos automaticamente
                if (!empty($extractedData['full_name'])) {
                    $this->name = $extractedData['full_name'];
                }

                if (!empty($extractedData['dob'])) {
                    $this->dob = $extractedData['dob'];
                }

                if (!empty($extractedData['sex'])) {
                    // Converter M/F para Masculino/Feminino
                    $this->sex = $extractedData['sex'] === 'M' ? 'Masculino' :
                                ($extractedData['sex'] === 'F' ? 'Feminino' : $extractedData['sex']);
                }

                if (!empty($extractedData['father_name'])) {
                    $this->father_name = $extractedData['father_name'];
                }

                if (!empty($extractedData['mother_name'])) {
                    $this->mother_name = $extractedData['mother_name'];
                }

                if (!empty($extractedData['birth_place'])) {
                    $this->birth_place = $extractedData['birth_place'];
                }

                if (!empty($extractedData['birth_district'])) {
                    $this->birth_district = $extractedData['birth_district'];
                }

                if (!empty($extractedData['birth_province'])) {
                    $this->birth_province = $extractedData['birth_province'];
                }

                if (!empty($extractedData['document_id'])) {
                    $this->document_id = $extractedData['document_id'];
                }

                if (!empty($extractedData['document_type'])) {
                    $this->document_type = $extractedData['document_type'];
                }

                // Tentar extrair foto do estudante do BI
                $facePath = $ocrService->extractFaceFromDocument($fullPath);
                if ($facePath && file_exists($facePath)) {
                    // Salvar foto extraída no storage
                    $photoContent = file_get_contents($facePath);
                    $photoStoragePath = 'extracted_photos/' . uniqid('photo_') . '.jpg';
                    Storage::disk('public')->put($photoStoragePath, $photoContent);

                    $this->extracted_photo_path = $photoStoragePath;

                    // Limpar arquivo temporário da face
                    @unlink($facePath);
                }

                $this->bi_processed = true;

                if ($this->extracted_photo_path) {
                    $this->alert('success', 'Dados do BI extraídos com sucesso! Foto do estudante detectada automaticamente. Verifique e corrija se necessário.');
                } else {
                    $this->alert('success', 'Dados do BI extraídos com sucesso! Não foi possível detectar a foto - por favor, faça upload manualmente. Verifique e corrija se necessário.');
                }
            } else {
                $this->alert('error', 'Erro ao processar BI: ' . $result['error']);
            }

            // Limpar arquivo temporário
            Storage::disk('public')->delete($tempPath);

        } catch (Exception $e) {
            $this->alert('error', 'Erro ao processar BI: ' . $e->getMessage());
        }

        $this->processing_bi = false;
    }

    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        $this->model->fill($this->validatedData);

        if ($this->model->isClean() && !$this->generateNewPassword) {
            $this->alert('warning', 'Pelo menos um valor deve mudar.');
            return redirect()->back();
        }

        // Buscar o usuário antes da atualização
        $student = User::find($this->model->id);

        // Apenas gerar nova senha se a opção estiver ativada
        if ($this->generateNewPassword) {
            $password = rand(100000, 500000);
            $hashedPassword = bcrypt($password);
        } else {
            $password = null; // Senha não será alterada
            $hashedPassword = $student->password; // Mantém a senha antiga
        }

        // Handle avatar upload
        $avatarPath = $student->avatar; // Keep current avatar by default
        if ($this->avatar) {
            // Se houver upload manual de avatar, usar esse
            // Delete old avatar if exists
            if ($student->avatar && Storage::disk('public')->exists($student->avatar)) {
                Storage::disk('public')->delete($student->avatar);
            }
            $avatarPath = $this->avatar->store('avatars', 'public');
        } elseif ($this->extracted_photo_path) {
            // Se não houver upload manual mas houver foto extraída do BI, mover para avatars
            // Delete old avatar if exists
            if ($student->avatar && Storage::disk('public')->exists($student->avatar)) {
                Storage::disk('public')->delete($student->avatar);
            }
            $photoContent = Storage::disk('public')->get($this->extracted_photo_path);
            $avatarPath = 'avatars/' . basename($this->extracted_photo_path);
            Storage::disk('public')->put($avatarPath, $photoContent);
            // Limpar foto extraída temporária
            Storage::disk('public')->delete($this->extracted_photo_path);
        }

        // Atualizar os dados do usuário
        $student->update([
            'name' => $this->name,
            'sex' => $this->sex,
            'email' => $this->email,
            'student_id' => $this->student_id,
            'classroom_id' => $this->classroom_id,
            'dob' => $this->dob,
            'password' => $hashedPassword,
            'phone_number' => $this->phone_number,
            'is_paid' => true,
            'father_name' => $this->father_name,
            'mother_name' => $this->mother_name,
            'birth_place' => $this->birth_place,
            'birth_district' => $this->birth_district,
            'birth_province' => $this->birth_province,
            'document_id' => $this->document_id,
            'document_type' => $this->document_type,
            'avatar' => $avatarPath,
        ]);

        // Enviar SMS apenas se a senha foi alterada
        if ($this->generateNewPassword) {
            try {
                $message = "Caro pai, o seu educando " . $this->name . ", foi registrado no sistema de gestão escolar COPMOZ. ID do Estudante: $this->student_id e a sua nova senha: " . $password . "\n\nClique no link para acessar o sistema: https://siga.copmoz.ac.mz";

                $smsService = new SmsService();
                $result = $smsService->send($this->phone_number, $message);

                if (!$result['success']) {
                    $this->alert('warning', 'Dados atualizados, mas houve erro ao enviar SMS');
                }
            } catch (Exception $e) {
                $this->alert('warning', 'Dados atualizados, mas houve erro ao enviar SMS: ' . $e->getMessage());
            }
        }

        Note::createSystemNotification(Admin::class, 'Detalhes', 'Atualizado com sucesso ' . $this->name);
        $this->emit('noteAdded');
        $this->alert('success', 'Atualizado com sucesso ' . $this->name);
        $this->reset();

        return redirect()->route('admin.list.students');
    }


    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    /**
     * Gerar e enviar nova senha imediatamente
     */
    public function generateAndSendNewPassword()
    {
        $this->confirm('Tem certeza de que deseja gerar uma nova senha para este estudante?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, gerar nova senha!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmedPasswordGeneration',
            'onDismissed' => 'cancelled'
        ]);
    }

    /**
     * Confirmação para gerar nova senha
     */
    public function confirmedPasswordGeneration()
    {
        try {
            // Gerar nova senha
            $password = rand(100000, 500000);
            $hashedPassword = bcrypt($password);

            // Atualizar apenas a senha do estudante
            $this->model->update([
                'password' => $hashedPassword,
            ]);

            // Enviar SMS com a nova senha
            try {
                $message = "Caro pai, o seu educando " . $this->name . ", teve a senha alterada no sistema COPMOZ. ID do Estudante: " . $this->student_id . " e a nova senha: " . $password . "\n\nClique no link para acessar o sistema: https://siga.copmoz.ac.mz";

                $smsService = new SmsService();
                $result = $smsService->send($this->phone_number, $message);

                if ($result['success']) {
                    $this->alert('success', 'Nova senha gerada e enviada por SMS com sucesso!');
                } else {
                    $this->alert('warning', 'Nova senha gerada, mas houve erro ao enviar SMS');
                }
            } catch (Exception $e) {
                $this->alert('warning', 'Nova senha gerada, mas houve erro ao enviar SMS: ' . $e->getMessage());
            }

            Note::createSystemNotification(Admin::class, 'Senha Alterada', 'Nova senha gerada para ' . $this->name);
            $this->emit('noteAdded');

        } catch (Exception $e) {
            $this->alert('error', 'Erro ao gerar nova senha: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.students.edit-student', [
          'classes' => Classroom::query()->where(function ($query) {
                $query->orWhere('class', $this->class);
            })->get(),        ]);
    }
}
