<?php

namespace App\Http\Livewire\Admin\Teachers;

use App\Models\Admin;
use App\Models\Subject;
use App\Models\TeacherSubject;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;
use App\Models\Classroom;

class AssignSubject extends Component
{
    use FindGuard, LivewireAlert;

    public $name, $teacher_id, $subject_id, $classroom_id, $class, $model;
    public $year;
    public $validatedData;
    public $updateMode = false;

    public function mount()
    {
        $this->year = (int) date('Y');
    }

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    protected array $rules = [
        'teacher_id' => ['required', 'string', 'max:255'],
        'subject_id' => ['required', 'string', 'max:255'],
        'classroom_id' => ['required', 'string', 'max:255'],
        'class' => ['required', 'string', 'max:255'],
        'year' => ['required', 'integer'],
    ];
    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        // Check if the teacher already has this subject for this class/classroom in the same year
        $check = TeacherSubject::query()
            ->where('teacher_id', $this->teacher_id)
            ->where('classroom_id', $this->classroom_id)
            ->where('class', $this->class)
            ->where('subject_id', $this->subject_id)
            ->where('year', $this->year)
            ->first();

        if ($check) {
            $this->alert('error', 'Professor já tem esta disciplina atribuída para o ano ' . $this->year . '.');
            return;
        }

        TeacherSubject::create([
            'teacher_id' => $this->teacher_id,
            'subject_id' => $this->subject_id,
            'classroom_id' => $this->classroom_id,
            'class' => $this->class,
            'year' => $this->year,
        ]);

        Note::createSystemNotification(Admin::class, 'Novo professor - disciplina', 'Disciplina atribuída ao professor com sucesso ' . Admin::find($this->teacher_id)->name);
        $this->emit('noteAdded');
        $this->alert('success', 'Disciplina atribuída ao professor com sucesso para ' . $this->year);
        $this->reset(['teacher_id', 'subject_id', 'classroom_id', 'class']);
        $this->year = (int) date('Y');
    }


    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }
    public function render()
    {
        return view('livewire.admin.teachers.assign-subject', [
            'subjects' => Subject::query()->orderBy('name')->get(),
            'teachers' => Admin::query()->orderBy('name')->where('role', 2)->get(),
            'classes' => Classroom::query()->where(function ($query) {
                $query->orWhere('class', $this->class);
            })->get(),
        ]);
    }
}
