<?php

namespace App\Http\Livewire\Auth\Admin;

use App\Models\Admin;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class AdminReset extends Component
{
    use LivewireAlert;

    public $email;
    public $password;
    public $password_confirmation;

    public function mount(string $token)
    {
        if (!request()->hasValidSignature()) {
            session()->flash('error', 'Kindly generate a new password reset link.');
            return redirect()->route('admin.forgot');
        }

        $this->email = decrypt($token, true);
    }

    protected array $rules = [
        'email' => ['required', 'string', 'email', 'exists:admins', 'max:255'],
        'password' => ['required', 'string', 'min:8', 'confirmed', 'same:password_confirmation']
    ];

    /**
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function resetPassword()
    {
        $this->validate();

        $user = Admin::query()->firstWhere('email', $this->email);
        if (Hash::check($this->password, $user->password)) {
            $this->alert('warning', 'A nova senha não pode ser igual à sua senha atual.');
            return redirect()->back();
        }

        $user->update([
            'password' => bcrypt($this->password)
        ]);

        auth('admin')->login($user);
        $this->alert('success', 'A senha da conta foi atualizada.');
        return redirect()->route('admin.home');
    }

    public function render()
    {
        return view('livewire.auth.admin.admin-reset')->layout('layouts.auth');
    }
}
