<?php

namespace App\Http\Livewire\Guardian;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;

class Perfil extends Component
{
    use LivewireAlert;

    public $name;
    public $phone;
    public $email;
    public $current_password;
    public $password;
    public $password_confirmation;

    protected $listeners = [
        'confirmedProfile',
        'confirmedPassword',
        'cancelled',
    ];

    public function mount()
    {
        $guardian = Auth::guard('guardian')->user();
        $this->name  = $guardian->name;
        $this->phone = $guardian->phone;
        $this->email = $guardian->email;
    }

    protected function profileRules(): array
    {
        return [
            'name'  => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'phone' => ['required', 'string'],
        ];
    }

    protected function passwordRules(): array
    {
        return [
            'current_password'      => ['required', 'string'],
            'password'              => ['required', 'string', 'min:6', 'confirmed'],
            'password_confirmation' => ['required'],
        ];
    }

    /**
     * @throws ValidationException
     */
    public function updated($field)
    {
        if (in_array($field, ['name', 'email', 'phone'])) {
            $this->validateOnly($field, $this->profileRules());
        }
        if (in_array($field, ['current_password', 'password', 'password_confirmation'])) {
            $this->validateOnly($field, $this->passwordRules());
        }
    }

    public function updateProfile()
    {
        $this->validate($this->profileRules());

        $this->confirm('Tem certeza de que deseja actualizar o perfil?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, actualizar!',
            'cancelButtonText'  => 'Cancelar',
            'onConfirmed' => 'confirmedProfile',
            'onDismissed' => 'cancelled',
        ]);
    }

    public function confirmedProfile()
    {
        $guardian = Auth::guard('guardian')->user();

        $guardian->update([
            'name'  => $this->name,
            'email' => $this->email,
            'phone' => $this->phone,
        ]);

        $this->alert('success', 'Perfil actualizado com sucesso.');
    }

    public function updatePassword()
    {
        $this->validate($this->passwordRules());

        $guardian = Auth::guard('guardian')->user();

        if (!Hash::check($this->current_password, $guardian->password)) {
            $this->addError('current_password', 'A palavra-passe actual está incorrecta.');
            return;
        }

        if (Hash::check($this->password, $guardian->password)) {
            $this->alert('warning', 'A nova palavra-passe é igual à actual.');
            return;
        }

        $this->confirm('Tem certeza de que deseja alterar a palavra-passe?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, alterar!',
            'cancelButtonText'  => 'Cancelar',
            'onConfirmed' => 'confirmedPassword',
            'onDismissed' => 'cancelled',
        ]);
    }

    public function confirmedPassword()
    {
        $guardian = Auth::guard('guardian')->user();

        $guardian->update([
            'password' => bcrypt($this->password),
        ]);

        $this->reset(['current_password', 'password', 'password_confirmation']);
        $this->alert('success', 'Palavra-passe alterada com sucesso.');
    }

    public function cancelled()
    {
        $this->alert('info', 'Operação cancelada.');
    }

    public function render()
    {
        return view('livewire.guardian.perfil', [
            'guardian' => Auth::guard('guardian')->user(),
        ])->layout('layouts.app');
    }
}
