<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class CheckStudentPaymentStatus
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Verifica se é estudante (guard padrão 'web')
        $user = Auth::guard('web')->user();

        // Se não estiver autenticado como estudante, permite passar
        if (!$user) {
            return $next($request);
        }

        // Se for admin autenticado (guard admin), permite passar
        if (Auth::guard('admin')->check()) {
            return $next($request);
        }

        // Se for encarregado autenticado (guard guardian), permite passar
        if (Auth::guard('guardian')->check()) {
            return $next($request);
        }

        // Se o estudante já pagou, permite acesso total
        if ($user->is_paid == 1) {
            return $next($request);
        }

        // Rotas permitidas mesmo sem pagamento
        $allowedRoutes = [
            'user.fee.collection',  // Página de pagamentos
            'user.home',            // Home (para ver o alerta)
            'user.profile',         // Perfil
            'user.credentials',     // Alterar senha
            'logout',               // Logout
            'livewire.message',     // Requisições Livewire (AJAX)
            'livewire.upload',      // Upload via Livewire
        ];

        // URLs permitidas (sem nome de rota)
        $allowedPaths = [
            'storage/references/',  // Download de comprovativos/recibos
            'livewire/',            // Requisições Livewire
        ];

        // Se está tentando acessar rota permitida, deixa passar
        if ($request->routeIs($allowedRoutes)) {
            return $next($request);
        }

        // Se está tentando acessar path permitido, deixa passar
        foreach ($allowedPaths as $path) {
            if (str_contains($request->path(), $path)) {
                return $next($request);
            }
        }

        // Log para debug
        \Log::info('CheckStudentPaymentStatus: Bloqueando acesso', [
            'user_id' => $user->id,
            'is_paid' => $user->is_paid,
            'route' => $request->route() ? $request->route()->getName() : 'no-route',
            'url' => $request->url(),
        ]);

        // Bloqueia TODAS as outras rotas e redireciona para pagamentos
        return redirect()
            ->route('user.fee.collection')
            ->with('warning', 'Por favor, efetue o pagamento da taxa escolar para aceder aos conteúdos.');
    }
}
