<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class AdminPermission extends Model
{
    protected $table = 'admin_permissions';

    protected $fillable = [
        'admin_id',
        'permission_key',
        'granted',
    ];

    protected $casts = [
        'granted' => 'boolean',
    ];

    // Relationship
    public function admin()
    {
        return $this->belongsTo(Admin::class, 'admin_id');
    }

    /**
     * Get available permissions
     * Reads from database if seeded, otherwise returns hardcoded defaults
     */
    public static function availablePermissions()
    {
        // Try to get from database first
        if (PermissionDefinition::isSeeded()) {
            return PermissionDefinition::getGroupedForUI();
        }

        // Fallback to hardcoded values (for backward compatibility)
        return self::getHardcodedPermissions();
    }

    /**
     * Hardcoded permissions for backward compatibility
     */
    private static function getHardcodedPermissions()
    {
        return [
            // POS Permissions
            'pos' => [
                'label' => 'Sistema POS',
                'permissions' => [
                    'pos.access' => 'Acesso ao POS',
                    'pos.dashboard' => 'Ver Dashboard POS',
                    'pos.sales' => 'Realizar Vendas',
                    'pos.sales_report' => 'Ver Relatório de Vendas',
                    'pos.products.view' => 'Ver Produtos',
                    'pos.products.manage' => 'Gerir Produtos (Adicionar/Editar/Excluir)',
                    'pos.categories.view' => 'Ver Categorias',
                    'pos.categories.manage' => 'Gerir Categorias',
                    'pos.settings' => 'Configurações POS',
                ],
            ],
            // Students Permissions
            'students' => [
                'label' => 'Estudantes',
                'permissions' => [
                    'students.view' => 'Ver Estudantes',
                    'students.add' => 'Adicionar Estudantes',
                    'students.edit' => 'Editar Estudantes',
                    'students.delete' => 'Excluir Estudantes',
                    'students.export' => 'Exportar Dados',
                ],
            ],
            // Teachers Permissions
            'teachers' => [
                'label' => 'Professores',
                'permissions' => [
                    'teachers.view' => 'Ver Professores',
                    'teachers.add' => 'Adicionar Professores',
                    'teachers.edit' => 'Editar Professores',
                    'teachers.delete' => 'Excluir Professores',
                    'teachers.assign' => 'Atribuir Disciplinas',
                ],
            ],
            // Classes Permissions
            'classes' => [
                'label' => 'Turmas',
                'permissions' => [
                    'classes.view' => 'Ver Turmas',
                    'classes.add' => 'Adicionar Turmas',
                    'classes.edit' => 'Editar Turmas',
                    'classes.delete' => 'Excluir Turmas',
                ],
            ],
            // Grades/Exams Permissions
            'exams' => [
                'label' => 'Notas e Exames',
                'permissions' => [
                    'exams.view' => 'Ver Notas',
                    'exams.add' => 'Adicionar Notas',
                    'exams.edit' => 'Editar Notas',
                    'exams.delete' => 'Excluir Notas',
                    'exams.reports' => 'Gerar Relatórios',
                ],
            ],
            // Payments Permissions
            'payments' => [
                'label' => 'Pagamentos',
                'permissions' => [
                    'payments.view' => 'Ver Pagamentos',
                    'payments.collect' => 'Cobrar Pagamentos',
                    'payments.manage' => 'Gerir Tipos de Taxas',
                    'payments.reports' => 'Relatórios de Pagamento',
                ],
            ],
            // Library Permissions
            'library' => [
                'label' => 'Biblioteca Digital',
                'permissions' => [
                    'library.view' => 'Ver Biblioteca',
                    'library.add' => 'Adicionar Livros',
                    'library.edit' => 'Editar Livros',
                    'library.delete' => 'Excluir Livros',
                ],
            ],
            // System Permissions
            'system' => [
                'label' => 'Sistema',
                'permissions' => [
                    'system.staff.manage' => 'Gerir Staff',
                    'system.permissions' => 'Gerir Permissões',
                    'system.settings' => 'Configurações do Sistema',
                ],
            ],
            // SMS Permissions
            'sms' => [
                'label' => 'SMS',
                'permissions' => [
                    'sms.send' => 'Enviar SMS',
                    'sms.view' => 'Ver SMS Enviados',
                    'sms.settings' => 'Configurações SMS',
                ],
            ],
            // Certificates Permissions
            'certificates' => [
                'label' => 'Certificados',
                'permissions' => [
                    'certificates.view' => 'Ver Certificados',
                    'certificates.generate' => 'Gerar Certificados',
                    'certificates.download' => 'Baixar Certificados',
                    'certificates.batch' => 'Gerar Lotes de Certificados',
                    'certificates.revoke' => 'Revogar Certificados',
                ],
            ],
            // Student Cards Permissions
            'student_cards' => [
                'label' => 'Cartoes de Estudante',
                'permissions' => [
                    'student_cards.view' => 'Ver Cartoes',
                    'student_cards.generate' => 'Gerar Cartoes',
                    'student_cards.download' => 'Baixar Cartoes',
                    'student_cards.batch' => 'Gerar Lotes de Cartoes',
                ],
            ],
            // Attendance Permissions
            'attendance' => [
                'label' => 'Presença',
                'permissions' => [
                    'attendance.mark' => 'Marcar Presença',
                    'attendance.view' => 'Ver Registos de Presença',
                    'attendance.edit' => 'Editar Presença',
                    'attendance.delete' => 'Excluir Presença',
                    'attendance.report' => 'Relatórios de Presença',
                ],
            ],
            // Secretary Permissions
            'secretary' => [
                'label' => 'Secretaria',
                'permissions' => [
                    'secretary.pre_enrollments.view' => 'Ver Pré-inscrições',
                    'secretary.pre_enrollments.manage' => 'Gerir Pré-inscrições (Confirmar/Anular)',
                    'secretary.enrollment.create' => 'Realizar Matrículas',
                ],
            ],
            // Renewals Permissions
            'renewals' => [
                'label' => 'Renovações de Matrícula',
                'permissions' => [
                    'renewals.view' => 'Ver Renovações de Matrícula',
                    'renewals.manage' => 'Confirmar / Cancelar Renovações',
                    'renewals.settings' => 'Configurar Período de Renovação',
                ],
            ],
            // Communications Permissions
            'communications' => [
                'label' => 'Comunicados e Notificações',
                'permissions' => [
                    'communications.send_notification' => 'Enviar Notificação',
                    'communications.add_notice' => 'Adicionar Comunicado',
                    'communications.view_notices' => 'Ver Comunicados',
                    'communications.payment_reminders' => 'Lembretes de Pagamento',
                ],
            ],
            // Documents Permissions
            'documents' => [
                'label' => 'Documentos e Matrizes',
                'permissions' => [
                    'documents.add_template' => 'Adicionar Matriz',
                    'documents.view_templates' => 'Ver Matrizes',
                    'documents.certificates' => 'Gerir Certificados',
                    'documents.student_cards' => 'Gerir Cartões de Estudante',
                ],
            ],
        ];
    }

    // Helper method to check if admin has permission
    public static function hasPermission($adminId, $permissionKey)
    {
        $permission = self::where('admin_id', $adminId)
            ->where('permission_key', $permissionKey)
            ->first();

        // If no record exists, deny by default
        return $permission ? $permission->granted : false;
    }

    // Grant permission
    public static function grant($adminId, $permissionKey)
    {
        return self::updateOrCreate(
            ['admin_id' => $adminId, 'permission_key' => $permissionKey],
            ['granted' => true]
        );
    }

    // Revoke permission
    public static function revoke($adminId, $permissionKey)
    {
        return self::updateOrCreate(
            ['admin_id' => $adminId, 'permission_key' => $permissionKey],
            ['granted' => false]
        );
    }

    // Get all permissions for an admin
    public static function getAdminPermissions($adminId)
    {
        return self::where('admin_id', $adminId)
            ->where('granted', true)
            ->pluck('permission_key')
            ->toArray();
    }
}
