<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class PaymentReference extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'student_id',
        'classe',
        'turma',
        'entity_code',
        'reference_number',
        'amount',
        'fee_month',
        'fee_year',
        'academic_year',
        'status',
        'approval_status',
        'approved_by',
        'approved_at',
        'approval_notes',
        'reminder_sent',
        'reminder_sent_at',
        'valid_from',
        'payment_starts_at',
        'due_date',
        'expires_at',
        'paid_at',
        'penalty_type',
        'penalty_rate',
        'has_penalty',
        'original_reference_id',
        'metadata'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'fine_amount' => 'decimal:2',
        'penalty_rate' => 'decimal:2',
        'has_penalty' => 'boolean',
        'reminder_sent' => 'boolean',
        'valid_from' => 'datetime',
        'payment_starts_at' => 'datetime',
        'due_date' => 'datetime',
        'expires_at' => 'datetime',
        'paid_at' => 'datetime',
        'approved_at' => 'datetime',
        'reminder_sent_at' => 'datetime',
        'metadata' => 'array',
    ];

    public function student(): BelongsTo
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    public function bmepsTransactions(): HasMany
    {
        return $this->hasMany(BmepsTransaction::class);
    }

    public function feeAssigns(): HasMany
    {
        return $this->hasMany(FeeAssign::class);
    }

    public function approver(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'approved_by');
    }

    public function approve($notes = null)
    {
        $this->update([
            'approval_status' => 'approved',
            'approved_by' => auth('admin')->id(),
            'approved_at' => now(),
            'approval_notes' => $notes,
        ]);
    }

    public function reject($notes = null)
    {
        $this->update([
            'approval_status' => 'rejected',
            'approved_by' => auth('admin')->id(),
            'approved_at' => now(),
            'approval_notes' => $notes,
        ]);
    }

    public function scopeApproved($query)
    {
        return $query->where('approval_status', 'approved');
    }

    public function scopePendingApproval($query)
    {
        return $query->where('approval_status', 'pending_approval');
    }

    /**
     * Relacionamento com a referência original (se esta for uma ref com multa)
     */
    public function originalReference(): BelongsTo
    {
        return $this->belongsTo(PaymentReference::class, 'original_reference_id');
    }

    /**
     * Relacionamento com referências com multa geradas a partir desta
     */
    public function penaltyReferences(): HasMany
    {
        return $this->hasMany(PaymentReference::class, 'original_reference_id');
    }

    /**
     * Verifica se a referência está vencida (passou do due_date)
     */
    public function isOverdue(): bool
    {
        if (!$this->due_date) {
            return false;
        }

        return now()->isAfter($this->due_date) && $this->status === 'pending';
    }

    /**
     * Calcula o valor da multa
     */
    public function calculatePenalty(): float
    {
        if (!$this->isOverdue() || !$this->penalty_rate) {
            return 0;
        }

        if ($this->penalty_type === 'percentage') {
            return round(($this->amount * $this->penalty_rate) / 100, 2);
        } elseif ($this->penalty_type === 'fixed') {
            return $this->penalty_rate;
        }

        return 0;
    }

    /**
     * Calcula o valor total (original + multa)
     */
    public function getTotalAmountAttribute(): float
    {
        return $this->amount + $this->calculatePenalty();
    }

    /**
     * Verifica se já existe uma referência com multa gerada
     */
    public function hasPenaltyReferenceGenerated(): bool
    {
        return $this->penaltyReferences()->exists();
    }
}

