<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class RolePermission extends Model
{
    protected $fillable = [
        'role',
        'permission_key',
        'granted',
    ];

    protected $casts = [
        'role' => 'integer',
        'granted' => 'boolean',
    ];

    /**
     * Get all role names
     */
    public static function getRoleNames(): array
    {
        return [
            0 => 'Super Admin',
            1 => 'Reg Academico',
            2 => 'Professor',
            3 => 'Diretor',
            4 => 'Admin',
            5 => 'Contabilidade',
            6 => 'Secretaria',
        ];
    }

    /**
     * Get role name by ID
     */
    public static function getRoleName(int $role): string
    {
        return self::getRoleNames()[$role] ?? 'Desconhecido';
    }

    /**
     * Get permissions for a specific role (cached)
     */
    public static function getPermissionsForRole(int $role): array
    {
        // Super Admin always has all permissions
        if ($role === 0) {
            return array_keys(Admin::getAllPermissionsStatic());
        }

        $cacheKey = "role_permissions_{$role}";

        return Cache::remember($cacheKey, 3600, function () use ($role) {
            $permissions = self::where('role', $role)
                ->where('granted', true)
                ->pluck('permission_key')
                ->toArray();

            // If no permissions in database, return empty array
            // (will be populated when seeding or first configuration)
            return $permissions;
        });
    }

    /**
     * Check if role has a specific permission
     */
    public static function roleHasPermission(int $role, string $permissionKey): bool
    {
        // Super Admin always has all permissions
        if ($role === 0) {
            return true;
        }

        return in_array($permissionKey, self::getPermissionsForRole($role));
    }

    /**
     * Set permission for a role
     */
    public static function setPermission(int $role, string $permissionKey, bool $granted): void
    {
        self::updateOrCreate(
            ['role' => $role, 'permission_key' => $permissionKey],
            ['granted' => $granted]
        );

        // Clear cache for this role
        Cache::forget("role_permissions_{$role}");
    }

    /**
     * Set multiple permissions for a role
     */
    public static function setPermissions(int $role, array $permissions): void
    {
        // Get all available permissions
        $allPermissions = array_keys(Admin::getAllPermissionsStatic());

        // Update each permission
        foreach ($allPermissions as $permKey) {
            $granted = in_array($permKey, $permissions);
            self::updateOrCreate(
                ['role' => $role, 'permission_key' => $permKey],
                ['granted' => $granted]
            );
        }

        // Clear cache for this role
        Cache::forget("role_permissions_{$role}");
    }

    /**
     * Clear all caches for role permissions
     */
    public static function clearCache(): void
    {
        for ($i = 0; $i <= 6; $i++) {
            Cache::forget("role_permissions_{$i}");
        }
    }

    /**
     * Check if role permissions have been configured
     */
    public static function isConfigured(int $role): bool
    {
        return self::where('role', $role)->exists();
    }

    /**
     * Seed default permissions from Admin model's hardcoded values
     */
    public static function seedDefaultPermissions(): void
    {
        $defaults = [
            // Role 1 = Reg Academico
            1 => [
                'students.view', 'students.add', 'students.edit', 'students.export',
                'payments.view', 'payments.collect', 'payments.reports',
                'classes.view', 'classes.add', 'classes.edit',
                'exams.view', 'exams.add', 'exams.edit',
                'teachers.view',
                'renewals.view', 'renewals.manage', 'renewals.settings',
                'attendance.mark', 'attendance.view', 'attendance.edit', 'attendance.report',
                'sms.send', 'sms.view',
            ],

            // Role 2 = Professor
            2 => [
                'students.view',
                'classes.view',
                'exams.view', 'exams.add', 'exams.edit', 'exams.reports',
                'library.view',
                'attendance.mark', 'attendance.view',
            ],

            // Role 3 = Diretor
            3 => [
                'students.view', 'students.add', 'students.edit', 'students.export',
                'teachers.view', 'teachers.add', 'teachers.edit', 'teachers.assign',
                'classes.view', 'classes.add', 'classes.edit',
                'exams.view', 'exams.add', 'exams.edit', 'exams.reports',
                'payments.view', 'payments.reports',
                'library.view', 'library.add', 'library.edit',
                'renewals.view',
                'system.settings',
                'sms.send', 'sms.view',
                'attendance.mark', 'attendance.view', 'attendance.edit', 'attendance.report',
            ],

            // Role 4 = Admin
            4 => [
                'students.view', 'students.add', 'students.edit', 'students.export',
                'teachers.view', 'teachers.add', 'teachers.edit', 'teachers.assign',
                'classes.view', 'classes.add', 'classes.edit',
                'exams.view', 'exams.add', 'exams.edit', 'exams.reports',
                'payments.view', 'payments.collect', 'payments.manage', 'payments.reports',
                'pos.access', 'pos.dashboard', 'pos.sales', 'pos.sales_report',
                'pos.products.view', 'pos.products.manage',
                'pos.categories.view', 'pos.categories.manage',
                'pos.settings',
                'library.view', 'library.add', 'library.edit', 'library.delete',
                'system.settings', 'system.staff',
                'renewals.view', 'renewals.manage', 'renewals.settings',
            ],

            // Role 5 = Contabilidade
            5 => [
                'students.view',
                'classes.view',
                'payments.view', 'payments.collect', 'payments.manage', 'payments.reports',
                'pos.access', 'pos.dashboard', 'pos.sales', 'pos.sales_report',
                'pos.products.view', 'pos.products.manage',
                'pos.categories.view', 'pos.categories.manage',
                'pos.settings',
            ],

            // Role 6 = Secretaria
            6 => [
                'students.view', 'students.add', 'students.edit', 'students.export',
                'classes.view', 'classes.add', 'classes.edit',
                'teachers.view',
                'exams.view',
                'payments.view',
                'renewals.view', 'renewals.manage', 'renewals.settings',
            ],
        ];

        foreach ($defaults as $role => $permissions) {
            self::setPermissions($role, $permissions);
        }

        self::clearCache();
    }
}
