<?php

namespace App\Services;

use Google\Cloud\Vision\V1\Client\ImageAnnotatorClient;
use Google\Cloud\Vision\V1\Image;
use Google\Cloud\Vision\V1\Feature;
use Google\Cloud\Vision\V1\Feature\Type;
use Google\Cloud\Vision\V1\AnnotateImageRequest;
use Google\Cloud\Vision\V1\BatchAnnotateImagesRequest;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Exception;

class GoogleVisionOCRService
{
    protected $client;

    public function __construct()
    {
        try {
            $credentialsPath = config('services.google_vision.credentials_path');

            // Verificar se o arquivo existe
            if (!file_exists($credentialsPath)) {
                throw new Exception("Arquivo de credenciais não encontrado: {$credentialsPath}");
            }

            // Inicializar cliente Google Vision com credenciais
            putenv('GOOGLE_APPLICATION_CREDENTIALS=' . $credentialsPath);

            $this->client = new ImageAnnotatorClient();
        } catch (Exception $e) {
            Log::error('Erro ao inicializar Google Vision API: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Processar documento (imagem ou PDF) e extrair texto usando OCR
     */
    public function processDocument($filePath)
    {
        // Detectar tipo de arquivo
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));

        if ($extension === 'pdf') {
            return $this->processPDF($filePath);
        } else {
            return $this->processImage($filePath);
        }
    }

    /**
     * Processar PDF - Não suportado no servidor
     * PDFs devem ser convertidos para imagem no navegador usando PDF.js
     */
    public function processPDF($pdfPath)
    {
        Log::warning('Tentativa de processar PDF no servidor. PDFs devem ser convertidos no navegador.');

        return [
            'success' => false,
            'error' => 'PDF não pode ser processado no servidor. Por favor, use imagens (JPG/PNG) ou tire foto do documento. Se você fez upload de PDF, a conversão automática no navegador falhou - tente recarregar a página.',
        ];
    }

    /**
     * Processar imagem e extrair texto usando OCR
     */
    public function processImage($imagePath)
    {
        try {
            // Ler o arquivo de imagem
            $imageContent = file_get_contents($imagePath);

            // Criar objeto Image
            $image = new Image();
            $image->setContent($imageContent);

            // Criar Feature para TEXT_DETECTION
            $feature = new Feature();
            $feature->setType(Type::TEXT_DETECTION);

            // Criar request
            $request = new AnnotateImageRequest();
            $request->setImage($image);
            $request->setFeatures([$feature]);

            // Criar batch request
            $batchRequest = new BatchAnnotateImagesRequest();
            $batchRequest->setRequests([$request]);

            // Realizar OCR
            $response = $this->client->batchAnnotateImages($batchRequest);
            $annotations = $response->getResponses()[0];

            // Verificar se há erro na resposta
            if ($annotations->hasError() && $annotations->getError()) {
                $error = $annotations->getError();
                $errorMessage = $error->getMessage();
                $errorCode = $error->getCode();

                Log::error("Google Vision API Error: Code {$errorCode} - {$errorMessage}");

                return [
                    'success' => false,
                    'error' => "Erro da API: {$errorMessage}",
                ];
            }

            if ($annotations && $annotations->getTextAnnotations()) {
                $textAnnotations = $annotations->getTextAnnotations();

                if (count($textAnnotations) > 0) {
                    // O primeiro elemento contém todo o texto
                    $fullText = $textAnnotations[0]->getDescription();

                    // Calcular confiança média (Google Vision não fornece score direto, vamos usar 95 como padrão)
                    $averageConfidence = 95;

                    Log::info("OCR bem sucedido. Texto extraído: " . strlen($fullText) . " caracteres");

                    return [
                        'success' => true,
                        'text' => $fullText,
                        'confidence' => $averageConfidence,
                        'annotations' => [],
                    ];
                }
            }

            Log::warning("Google Vision não detectou texto na imagem: {$imagePath}");

            return [
                'success' => false,
                'error' => 'Nenhum texto foi detectado na imagem',
            ];

        } catch (Exception $e) {
            Log::error('Erro no processamento OCR: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Extrair dados específicos do BI moçambicano
     */
    public function extractBIData($ocrText)
    {
        // Log temporário para debug
        Log::info("=== TEXTO EXTRAÍDO PELO OCR ===");
        Log::info($ocrText);
        Log::info("=== FIM DO TEXTO ===");

        $data = [
            'document_type' => null,
            'document_id' => null,
            'full_name' => null,
            'dob' => null,
            'father_name' => null,
            'mother_name' => null,
            'birth_place' => null,
            'birth_district' => null,
            'birth_province' => null,
            'sex' => null,
        ];

        // Detectar tipo de documento
        if (preg_match('/BILHETE\s+DE\s+IDENTIDADE|B\.?\s*I\.?/i', $ocrText)) {
            $data['document_type'] = 'BI';
        } elseif (preg_match('/DIRE/i', $ocrText)) {
            $data['document_type'] = 'DIRE';
        }

        // Extrair número do documento (formato: N°: 030402908968I ou N: 0304029089681)
        // OCR frequentemente confunde letra I com número 1
        // Aceita 12 ou 13 dígitos, com ou sem letra final
        if (preg_match('/N[^\d\n\r]*(\d{12,13}[A-Z]?)/ui', $ocrText, $matches)) {
            $biNumber = $matches[1];

            // Se tem 13 dígitos e o último é "1", provavelmente é "I"
            if (strlen($biNumber) == 13 && substr($biNumber, -1) === '1') {
                $biNumber = substr($biNumber, 0, -1) . 'I';
            }

            $data['document_id'] = $biNumber;
        }

        // Extrair nome completo (formato variável: "Nome/Name:", "No/Name", etc)
        // Busca por variações de "Nome/Name" seguido de nome em maiúsculas
        if (preg_match('/No(?:me)?\s*\/\s*Name\s*:?\s*[\n\r\s]*([A-ZÀÁÂÃÉÊÍÓÔÕÚ][A-Za-zÀ-ÿ\s]+?)[\n\r]+/ui', $ocrText, $matches)) {
            $data['full_name'] = trim($matches[1]);
        }

        // Extrair data de nascimento (formato com quebras de linha)
        if (preg_match('/Date\s+of\s+Birth\s*:?\s*[\n\r\s]*(\d{2}\/\d{2}\/\d{4})/i', $ocrText, $matches)) {
            $data['dob'] = $this->formatDate($matches[1]);
        }

        // Extrair nome do pai (formato: Nome do Pai / Father Name:\nSAID SALUM MOHAMED)
        if (preg_match('/Father\s+Name\s*:?\s*[\n\r\s]*([A-ZÀÁÂÃÉÊÍÓÔÕÚ][A-Za-zÀ-ÿ\s]+?)[\n\r]+/ui', $ocrText, $matches)) {
            $data['father_name'] = trim($matches[1]);
        }

        // Extrair nome da mãe (formato: Nome da Mãe / Mother Name:\nDALILA ABDUL SATAR)
        if (preg_match('/Mother\s+Name\s*:?\s*[\n\r\s]*([A-ZÀÁÂÃÉÊÍÓÔÕÚ][A-Za-zÀ-ÿ\s]+?)[\n\r]+/ui', $ocrText, $matches)) {
            $data['mother_name'] = trim($matches[1]);
        }

        // Extrair local de nascimento (formato variável: "Naturalidad/Place of Birth", "Maturalidade", etc)
        // Aceita variações: Naturalidade, Maturalidade, Naturalidad, etc
        // Captura até quebra de linha, ponto seguido de espaço, ou "Altura"
        if (preg_match('/(?:[MN]at[ue]ralidad?e?)\s*\/?\s*Place\s+of\s+Birth\s*:?\s*[\n\r\s]*([A-ZÀÁÂÃÉÊÍÓÔÕÚ][A-Za-zÀ-ÿ\s\.]+?)(?:[\n\r]+|\.(?:\s|$)|(?=Altura))/ui', $ocrText, $matches)) {
            $birthInfo = trim($matches[1]);

            // Normalizar nomes de locais (expandir abreviações e corrigir acentuação)
            $birthInfo = $this->normalizeBirthPlace($birthInfo);

            $data['birth_place'] = $birthInfo;

            // Tentar extrair província/distrito se tiver separadores
            $parts = preg_split('/[,\-]/', $birthInfo);
            if (count($parts) >= 2) {
                $data['birth_district'] = trim($parts[0]);
                $data['birth_province'] = trim($parts[1]);
            } else {
                // Usar o local como distrito e buscar província automaticamente
                $data['birth_district'] = $birthInfo;
                $province = $this->getProvinceFromDistrict($birthInfo);
                if ($province) {
                    $data['birth_province'] = $province;
                }
            }
        }

        // Extrair sexo (formato variável: "Sexo/Sex: M", "Sexo/Bax M", etc)
        // Busca por "Sex" ou "Sexo" seguido opcionalmente de qualquer coisa, depois M ou F
        if (preg_match('/Sex(?:o)?[^\n\r]*?\s+([MF])/i', $ocrText, $matches)) {
            $data['sex'] = strtoupper($matches[1]);
        }

        return $data;
    }

    /**
     * Formatar data para Y-m-d
     */
    protected function formatDate($date)
    {
        $date = str_replace(['/', '.'], '-', $date);
        $parts = explode('-', $date);

        if (count($parts) === 3) {
            return sprintf('%04d-%02d-%02d', $parts[2], $parts[1], $parts[0]);
        }

        return null;
    }

    /**
     * Normalizar nomes de locais (expandir abreviações e corrigir acentuação)
     */
    protected function normalizeBirthPlace($place)
    {
        // Mapa de abreviações e correções
        $normalizations = [
            // Ilha de Moçambique (aceita MOC, MOC., MOÇAMBIQUE, etc)
            '/ILHA\s+DE\s+MO[CÇ]\.?(?:AMBIQUE)?/i' => 'Ilha de Moçambique',
            '/ILHA\s+MO[CÇ]\.?(?:AMBIQUE)?/i' => 'Ilha de Moçambique',

            // Maputo
            '/(?:CIDADE\s+DE\s+)?MAPUTO/i' => 'Maputo',
            '/LOURENCO\s+MARQUES/i' => 'Maputo', // Nome antigo

            // Nampula
            '/(?:CIDADE\s+DE\s+)?NAMPULA/i' => 'Nampula',

            // Beira
            '/(?:CIDADE\s+DA\s+)?BEIRA/i' => 'Beira',

            // Tete
            '/(?:CIDADE\s+DE\s+)?TETE/i' => 'Tete',

            // Quelimane
            '/QUELIMANE/i' => 'Quelimane',

            // Pemba
            '/(?:CIDADE\s+DE\s+)?PEMBA/i' => 'Pemba',

            // Lichinga
            '/LICHINGA/i' => 'Lichinga',

            // Xai-Xai
            '/XAI[\s-]*XAI/i' => 'Xai-Xai',

            // Inhambane
            '/INHAMBANE/i' => 'Inhambane',

            // Chimoio
            '/CHIMOIO/i' => 'Chimoio',

            // Nacala
            '/NACALA/i' => 'Nacala',

            // Mocuba
            '/MOCUBA/i' => 'Mocuba',

            // Mocimboa da Praia
            '/MOCIMBOA\s+(?:DA\s+)?PRAIA/i' => 'Mocímboa da Praia',

            // Maxixe
            '/MAXIXE/i' => 'Maxixe',
        ];

        foreach ($normalizations as $pattern => $replacement) {
            $place = preg_replace($pattern, $replacement, $place);
        }

        return trim($place);
    }

    /**
     * Obter província baseado no distrito
     */
    protected function getProvinceFromDistrict($district)
    {
        // Mapeamento completo de distritos e províncias de Moçambique
        $districtProvinceMap = [
            // Niassa
            'Cuamba' => 'Niassa',
            'Lichinga' => 'Niassa',
            'Lago' => 'Niassa',
            'Majune' => 'Niassa',
            'Mandimba' => 'Niassa',
            'Marrupa' => 'Niassa',
            'Mauá' => 'Niassa',
            'Mavago' => 'Niassa',
            'Mecanhelas' => 'Niassa',
            'Mecula' => 'Niassa',
            'Metarica' => 'Niassa',
            'Muembe' => 'Niassa',
            'N\'gauma' => 'Niassa',
            'Nipepe' => 'Niassa',
            'Sanga' => 'Niassa',

            // Cabo Delgado
            'Ancuabe' => 'Cabo Delgado',
            'Balama' => 'Cabo Delgado',
            'Chiúre' => 'Cabo Delgado',
            'Ibo' => 'Cabo Delgado',
            'Macomia' => 'Cabo Delgado',
            'Mecúfi' => 'Cabo Delgado',
            'Meluco' => 'Cabo Delgado',
            'Metuge' => 'Cabo Delgado',
            'Mocímboa da Praia' => 'Cabo Delgado',
            'Montepuez' => 'Cabo Delgado',
            'Mueda' => 'Cabo Delgado',
            'Muidumbe' => 'Cabo Delgado',
            'Namuno' => 'Cabo Delgado',
            'Nangade' => 'Cabo Delgado',
            'Palma' => 'Cabo Delgado',
            'Pemba' => 'Cabo Delgado',
            'Quissanga' => 'Cabo Delgado',

            // Nampula
            'Angoche' => 'Nampula',
            'Eráti' => 'Nampula',
            'Ilha de Moçambique' => 'Nampula',
            'Lalaua' => 'Nampula',
            'Larde' => 'Nampula',
            'Liúpo' => 'Nampula',
            'Malema' => 'Nampula',
            'Meconta' => 'Nampula',
            'Mecubúri' => 'Nampula',
            'Memba' => 'Nampula',
            'Mogincual' => 'Nampula',
            'Mogovolas' => 'Nampula',
            'Moma' => 'Nampula',
            'Monapo' => 'Nampula',
            'Mossuril' => 'Nampula',
            'Muecate' => 'Nampula',
            'Murrupula' => 'Nampula',
            'Nacala-a-Velha' => 'Nampula',
            'Nacala Porto' => 'Nampula',
            'Nacala' => 'Nampula',
            'Nacarôa' => 'Nampula',
            'Nampula' => 'Nampula',
            'Rapale' => 'Nampula',
            'Ribáuè' => 'Nampula',

            // Zambézia
            'Alto Molócuè' => 'Zambézia',
            'Chinde' => 'Zambézia',
            'Derre' => 'Zambézia',
            'Gilé' => 'Zambézia',
            'Gurué' => 'Zambézia',
            'Ile' => 'Zambézia',
            'Inhassunge' => 'Zambézia',
            'Lugela' => 'Zambézia',
            'Maganja da Costa' => 'Zambézia',
            'Milange' => 'Zambézia',
            'Mocuba' => 'Zambézia',
            'Mocubela' => 'Zambézia',
            'Molumbo' => 'Zambézia',
            'Mopeia' => 'Zambézia',
            'Morrumbala' => 'Zambézia',
            'Mulevala' => 'Zambézia',
            'Namacurra' => 'Zambézia',
            'Namarroi' => 'Zambézia',
            'Nicoadala' => 'Zambézia',
            'Pebane' => 'Zambézia',
            'Quelimane' => 'Zambézia',

            // Tete
            'Angónia' => 'Tete',
            'Cahora Bassa' => 'Tete',
            'Changara' => 'Tete',
            'Chifunde' => 'Tete',
            'Chiuta' => 'Tete',
            'Doa' => 'Tete',
            'Macanga' => 'Tete',
            'Magoé' => 'Tete',
            'Marara' => 'Tete',
            'Marávia' => 'Tete',
            'Moatize' => 'Tete',
            'Mutarara' => 'Tete',
            'Tete' => 'Tete',
            'Tsangano' => 'Tete',
            'Zumbo' => 'Tete',

            // Manica
            'Báruè' => 'Manica',
            'Chimoio' => 'Manica',
            'Gondola' => 'Manica',
            'Guro' => 'Manica',
            'Machaze' => 'Manica',
            'Macossa' => 'Manica',
            'Manica' => 'Manica',
            'Mossurize' => 'Manica',
            'Sussundenga' => 'Manica',
            'Tambara' => 'Manica',
            'Vanduzi' => 'Manica',

            // Sofala
            'Beira' => 'Sofala',
            'Búzi' => 'Sofala',
            'Caia' => 'Sofala',
            'Chemba' => 'Sofala',
            'Cheringoma' => 'Sofala',
            'Chibabava' => 'Sofala',
            'Dondo' => 'Sofala',
            'Gorongosa' => 'Sofala',
            'Machanga' => 'Sofala',
            'Maringué' => 'Sofala',
            'Marromeu' => 'Sofala',
            'Muanza' => 'Sofala',
            'Nhamatanda' => 'Sofala',

            // Inhambane
            'Funhalouro' => 'Inhambane',
            'Govuro' => 'Inhambane',
            'Homoíne' => 'Inhambane',
            'Inhambane' => 'Inhambane',
            'Inharrime' => 'Inhambane',
            'Inhassoro' => 'Inhambane',
            'Jangamo' => 'Inhambane',
            'Mabote' => 'Inhambane',
            'Massinga' => 'Inhambane',
            'Maxixe' => 'Inhambane',
            'Morrumbene' => 'Inhambane',
            'Panda' => 'Inhambane',
            'Vilankulo' => 'Inhambane',
            'Vilanculos' => 'Inhambane',
            'Zavala' => 'Inhambane',

            // Gaza
            'Bilene' => 'Gaza',
            'Chibuto' => 'Gaza',
            'Chicualacuala' => 'Gaza',
            'Chigubo' => 'Gaza',
            'Chókwè' => 'Gaza',
            'Guijá' => 'Gaza',
            'Limpopo' => 'Gaza',
            'Mabalane' => 'Gaza',
            'Mandlakazi' => 'Gaza',
            'Massangena' => 'Gaza',
            'Massingir' => 'Gaza',
            'Xai-Xai' => 'Gaza',

            // Maputo Província
            'Boane' => 'Maputo',
            'Magude' => 'Maputo',
            'Manhiça' => 'Maputo',
            'Marracuene' => 'Maputo',
            'Matola' => 'Maputo',
            'Moamba' => 'Maputo',
            'Namaacha' => 'Maputo',

            // Maputo Cidade
            'Maputo' => 'Maputo',
            'Lourenço Marques' => 'Maputo',
        ];

        // Normalizar o nome do distrito para comparação
        $normalizedDistrict = trim($district);

        // Buscar correspondência exata primeiro
        if (isset($districtProvinceMap[$normalizedDistrict])) {
            return $districtProvinceMap[$normalizedDistrict];
        }

        // Buscar correspondência parcial (case-insensitive)
        foreach ($districtProvinceMap as $districtName => $province) {
            if (stripos($normalizedDistrict, $districtName) !== false || stripos($districtName, $normalizedDistrict) !== false) {
                return $province;
            }
        }

        return null;
    }

    /**
     * Extrair foto do rosto do documento usando detecção de face
     * Retorna o caminho para a imagem recortada do rosto ou null se não encontrar
     */
    public function extractFaceFromDocument($imagePath)
    {
        try {
            // Ler o arquivo de imagem
            $imageContent = file_get_contents($imagePath);

            // Criar objeto Image
            $image = new Image();
            $image->setContent($imageContent);

            // Criar Feature para FACE_DETECTION
            $feature = new Feature();
            $feature->setType(Type::FACE_DETECTION);
            $feature->setMaxResults(10); // Detectar todas as faces para escolher a maior

            // Criar request
            $request = new AnnotateImageRequest();
            $request->setImage($image);
            $request->setFeatures([$feature]);

            // Criar batch request
            $batchRequest = new BatchAnnotateImagesRequest();
            $batchRequest->setRequests([$request]);

            // Realizar detecção de face
            $response = $this->client->batchAnnotateImages($batchRequest);
            $annotations = $response->getResponses()[0];

            // Verificar se há erro na resposta
            if ($annotations->hasError() && $annotations->getError()) {
                $error = $annotations->getError();
                Log::error("Google Vision API Error na detecção de face: {$error->getMessage()}");
                return null;
            }

            $faceAnnotations = $annotations->getFaceAnnotations();

            if ($faceAnnotations && count($faceAnnotations) > 0) {
                // Encontrar a maior face (a foto principal do estudante)
                $largestFace = null;
                $largestArea = 0;

                foreach ($faceAnnotations as $face) {
                    $boundingPoly = $face->getBoundingPoly();
                    if ($boundingPoly) {
                        $vertices = $boundingPoly->getVertices();
                        if (count($vertices) >= 4) {
                            $x1 = $vertices[0]->getX();
                            $y1 = $vertices[0]->getY();
                            $x2 = $vertices[2]->getX();
                            $y2 = $vertices[2]->getY();

                            $width = $x2 - $x1;
                            $height = $y2 - $y1;
                            $area = $width * $height;

                            // Guardar a maior face
                            if ($area > $largestArea) {
                                $largestArea = $area;
                                $largestFace = $face;
                            }
                        }
                    }
                }

                if ($largestFace) {
                    $boundingPoly = $largestFace->getBoundingPoly();
                    $vertices = $boundingPoly->getVertices();

                    // Obter coordenadas do bounding box
                    $x1 = $vertices[0]->getX();
                    $y1 = $vertices[0]->getY();
                    $x2 = $vertices[2]->getX();
                    $y2 = $vertices[2]->getY();

                    // Adicionar margem ao redor da face (20% em cada lado para melhor enquadramento)
                    $width = $x2 - $x1;
                    $height = $y2 - $y1;
                    $margin = 0.20;

                        $x1 = max(0, $x1 - ($width * $margin));
                        $y1 = max(0, $y1 - ($height * $margin));
                        $x2 = $x2 + ($width * $margin);
                        $y2 = $y2 + ($height * $margin);

                    // Recortar a imagem
                    $croppedPath = $this->cropImage($imagePath, $x1, $y1, $x2, $y2);

                    if ($croppedPath) {
                        // Remover fundo e adicionar fundo branco
                        $finalPath = $this->removeBackgroundAndAddWhite($croppedPath);

                        // Limpar imagem recortada temporária
                        if (file_exists($croppedPath)) {
                            @unlink($croppedPath);
                        }

                        if ($finalPath) {
                            Log::info("Face extraída com sucesso com fundo branco");
                            return $finalPath;
                        }

                        // Se falhar remoção de fundo, retornar imagem recortada original
                        Log::warning("Falha ao remover fundo, usando foto recortada original");
                        return $croppedPath;
                    }
                }
            }

            Log::warning("Nenhuma face detectada no documento");
            return null;

        } catch (Exception $e) {
            Log::error('Erro ao extrair face do documento: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Recortar imagem baseado em coordenadas
     */
    protected function cropImage($imagePath, $x1, $y1, $x2, $y2)
    {
        try {
            // Detectar tipo de imagem
            $imageInfo = getimagesize($imagePath);
            if (!$imageInfo) {
                return null;
            }

            $mime = $imageInfo['mime'];

            // Carregar imagem baseado no tipo
            switch ($mime) {
                case 'image/jpeg':
                case 'image/jpg':
                    $source = imagecreatefromjpeg($imagePath);
                    break;
                case 'image/png':
                    $source = imagecreatefrompng($imagePath);
                    break;
                case 'image/gif':
                    $source = imagecreatefromgif($imagePath);
                    break;
                default:
                    Log::warning("Tipo de imagem não suportado para recorte: {$mime}");
                    return null;
            }

            if (!$source) {
                return null;
            }

            // Calcular dimensões do recorte
            $cropWidth = $x2 - $x1;
            $cropHeight = $y2 - $y1;

            // Criar nova imagem recortada
            $cropped = imagecreatetruecolor($cropWidth, $cropHeight);

            // Preservar transparência para PNG
            if ($mime === 'image/png') {
                imagealphablending($cropped, false);
                imagesavealpha($cropped, true);
                $transparent = imagecolorallocatealpha($cropped, 0, 0, 0, 127);
                imagefill($cropped, 0, 0, $transparent);
            }

            // Copiar área recortada
            imagecopyresampled(
                $cropped, $source,
                0, 0, $x1, $y1,
                $cropWidth, $cropHeight,
                $cropWidth, $cropHeight
            );

            // Salvar imagem recortada em arquivo temporário
            $tempPath = sys_get_temp_dir() . '/' . uniqid('face_') . '.jpg';
            imagejpeg($cropped, $tempPath, 90);

            // Limpar recursos
            imagedestroy($source);
            imagedestroy($cropped);

            return $tempPath;

        } catch (Exception $e) {
            Log::error('Erro ao recortar imagem: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Remove background da imagem e adiciona fundo branco
     */
    protected function removeBackgroundAndAddWhite($imagePath)
    {
        try {
            $rembgApiKey = config('services.rembg.api_key') ?? env('REMBG_API_KEY');
            $rembgApiUrl = config('services.rembg.base_url') ?? env('REMBG_API_URL', 'https://api.rembg.com');

            if (!$rembgApiKey || !$rembgApiUrl) {
                Log::warning('rembg API não configurada para remover fundo da foto');
                return null;
            }

            // Aumentar limite de memória temporariamente
            $oldLimit = ini_get('memory_limit');
            ini_set('memory_limit', '256M');

            // Chamar API rembg
            $response = Http::timeout(30)
                ->withHeaders([
                    'x-api-key' => $rembgApiKey,
                ])
                ->attach('image', file_get_contents($imagePath), basename($imagePath))
                ->post($rembgApiUrl . '/rmbg');

            // Restaurar limite de memória
            ini_set('memory_limit', $oldLimit);

            if ($response->successful()) {
                // Obter imagem PNG com fundo removido
                $imageData = $response->body();

                // Criar imagem a partir dos dados
                $noBgImage = imagecreatefromstring($imageData);
                if (!$noBgImage) {
                    Log::error('Falha ao criar imagem a partir da resposta rembg');
                    return null;
                }

                // Obter dimensões
                $width = imagesx($noBgImage);
                $height = imagesy($noBgImage);

                // Criar nova imagem com fundo branco
                $whiteBackground = imagecreatetruecolor($width, $height);

                // Preencher com branco puro (RGB: 255, 255, 255)
                $white = imagecolorallocate($whiteBackground, 255, 255, 255);
                imagefilledrectangle($whiteBackground, 0, 0, $width, $height, $white);

                // Ativar alpha blending para compor corretamente as imagens
                imagealphablending($whiteBackground, true);
                imagesavealpha($whiteBackground, false);

                // Compor a imagem PNG sobre o fundo branco
                imagecopy($whiteBackground, $noBgImage, 0, 0, 0, 0, $width, $height);

                // Salvar em arquivo temporário
                $tempPath = sys_get_temp_dir() . '/' . uniqid('photo_white_') . '.jpg';
                imagejpeg($whiteBackground, $tempPath, 95);

                // Limpar recursos
                imagedestroy($noBgImage);
                imagedestroy($whiteBackground);

                Log::info('Fundo removido e fundo branco adicionado com sucesso');
                return $tempPath;
            }

            Log::warning('rembg API error', [
                'status' => $response->status(),
                'error' => $response->body(),
            ]);

            return null;

        } catch (Exception $e) {
            Log::error('Erro ao remover fundo da foto: ' . $e->getMessage());

            // Restaurar limite de memória em caso de erro
            if (isset($oldLimit)) {
                ini_set('memory_limit', $oldLimit);
            }

            return null;
        }
    }

    /**
     * Destrutor para fechar cliente
     */
    public function __destruct()
    {
        // Cliente fecha automaticamente
    }
}
