<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;

class PaymentReferenceService
{
    // ========================================================================
    // MÉTODO ORIGINAL (V1) - 11 DÍGITOS SEM ANO
    // Mantido para compatibilidade com código existente
    // ========================================================================

    /**
     * Gera referência no formato original (11 dígitos SEM ano)
     *
     * @deprecated Use generateWithYear() para novo formato com ano
     * @param string $entityCode Código da entidade (5 dígitos)
     * @param string $invoiceId Número da fatura/estudante (7 dígitos)
     * @param int $month Mês de pagamento (1-12)
     * @param float $amount Montante
     * @return string Referência de 11 dígitos
     */
    public function generate(string $entityCode, string $invoiceId, int $month, float $amount): string
    {
        // O montante não deve ter ponto/vírgula decimal.
        // Os dois últimos dígitos correspondem às casas decimais.
        $formattedAmount = number_format($amount, 2, '', '');

        $formattedMonth = str_pad($month, 2, '0', STR_PAD_LEFT);

        // A junção de "Entidade + Número Factura/Código Estudante + Mês de Pagamento + Montante".
        $baseString = $entityCode . $invoiceId . $formattedMonth . $formattedAmount;

        $pi = 0;
        // O cálculo do Pi inicia com 0 para i=1.
        for ($i = 0; $i < strlen($baseString); $i++) {
            $digit = (int)$baseString[$i];
            // Si = digito(i) + Pi
            $si = $digit + $pi;
            // Pi = (Si * 10) mod 97
            $pi = ($si * 10) % 97;
        }

        // Calcular "Pn".
        // Pn = (Pi * 10) mod 97
        $pn = ($pi * 10) % 97;

        // Calcular check digit.
        // Check digit = 98 – Pn
        $checkDigit = 98 - $pn;
        $formattedCheckDigit = str_pad($checkDigit, 2, '0', STR_PAD_LEFT);

        // Referência = Número factura + Mês + Check digit (11 dígitos SEM ANO).
        $finalReference = $invoiceId . $formattedMonth . $formattedCheckDigit;

        Log::debug('Payment Reference V1 generated (11 digits, no year)', [
            'entity' => $entityCode,
            'invoice_id' => $invoiceId,
            'month' => $month,
            'amount' => $amount,
            'reference' => $finalReference,
            'version' => 'V1'
        ]);

        return $finalReference;
    }

    // ========================================================================
    // NOVO MÉTODO (V2) - 13 DÍGITOS COM ANO
    // Recomendado para uso em novas implementações
    // ========================================================================

    /**
     * Gera referência no formato V2 (13 dígitos COM ano)
     *
     * @param string $entityCode Código da entidade (5 dígitos)
     * @param string $invoiceId Número da fatura/estudante (7 dígitos)
     * @param int $month Mês de pagamento (1-12)
     * @param int $year Ano de pagamento (ex: 2025)
     * @param float $amount Montante
     * @return string Referência de 13 dígitos
     */
    public function generateWithYear(string $entityCode, string $invoiceId, int $month, int $year, float $amount): string
    {
        // Formatar montante
        $formattedAmount = number_format($amount, 2, '', '');

        // Formatar mês (2 dígitos)
        $formattedMonth = str_pad($month, 2, '0', STR_PAD_LEFT);

        // Formatar ano (últimos 2 dígitos)
        $formattedYear = str_pad(substr((string)$year, -2), 2, '0', STR_PAD_LEFT);

        // A junção COM ANO: "Entidade + Número Factura + Mês + Ano + Montante"
        $baseString = $entityCode . $invoiceId . $formattedMonth . $formattedYear . $formattedAmount;

        $pi = 0;
        // O cálculo do Pi inicia com 0 para i=1
        for ($i = 0; $i < strlen($baseString); $i++) {
            $digit = (int)$baseString[$i];
            // Si = digito(i) + Pi
            $si = $digit + $pi;
            // Pi = (Si * 10) mod 97
            $pi = ($si * 10) % 97;
        }

        // Calcular "Pn"
        // Pn = (Pi * 10) mod 97
        $pn = ($pi * 10) % 97;

        // Calcular check digit
        // Check digit = 98 – Pn
        $checkDigit = 98 - $pn;
        $formattedCheckDigit = str_pad($checkDigit, 2, '0', STR_PAD_LEFT);

        // Referência = Número factura + Mês + Ano + Check digit (13 dígitos COM ANO)
        $finalReference = $invoiceId . $formattedMonth . $formattedYear . $formattedCheckDigit;

        Log::debug('Payment Reference V2 generated (13 digits, with year)', [
            'entity' => $entityCode,
            'invoice_id' => $invoiceId,
            'month' => $month,
            'year' => $year,
            'amount' => $amount,
            'reference' => $finalReference,
            'version' => 'V2'
        ]);

        return $finalReference;
    }

    /**
     * Formata referência para exibição legível
     *
     * @param string $reference Referência numérica
     * @return string Referência formatada com espaços
     */
    public function format(string $reference): string
    {
        $length = strlen($reference);

        if ($length === 13) {
            // Formato V2: XXX XXX XXX XX XX (7 número + 2 mês + 2 ano + 2 check)
            return substr($reference, 0, 3) . ' ' .
                   substr($reference, 3, 3) . ' ' .
                   substr($reference, 6, 3) . ' ' .
                   substr($reference, 9, 2) . ' ' .
                   substr($reference, 11, 2);
        }

        if ($length === 11) {
            // Formato V1: XXX XXX XXX XX (7 número + 2 mês + 2 check)
            return substr($reference, 0, 3) . ' ' .
                   substr($reference, 3, 3) . ' ' .
                   substr($reference, 6, 3) . ' ' .
                   substr($reference, 9, 2);
        }

        // Formato genérico
        $parts = [];
        for ($i = 0; $i < $length; $i += 3) {
            $parts[] = substr($reference, $i, 3);
        }

        return implode(' ', $parts);
    }

    /**
     * Valida formato de referência
     *
     * @param string $reference Referência a validar
     * @return bool True se válida
     */
    public function validate(string $reference): bool
    {
        // Deve ter apenas dígitos
        if (!ctype_digit($reference)) {
            return false;
        }

        // Aceitar 11 ou 13 dígitos
        $length = strlen($reference);
        if ($length !== 11 && $length !== 13) {
            return false;
        }

        // Não pode ser tudo zeros
        if ($reference === str_repeat('0', $length)) {
            return false;
        }

        return true;
    }

    /**
     * Teste comparativo V1 vs V2
     *
     * @return array Resultados do teste
     */
    public function testV1vsV2(): array
    {
        $entityCode = '11111';
        $invoiceId = '1234567';
        $month = 1; // Janeiro
        $amount = 500.00;

        // V1 - 11 dígitos (SEM ano)
        $refV1_2024 = $this->generate($entityCode, $invoiceId, $month, $amount);
        $refV1_2025 = $this->generate($entityCode, $invoiceId, $month, $amount);
        $refV1_2026 = $this->generate($entityCode, $invoiceId, $month, $amount);

        // V2 - 13 dígitos (COM ano)
        $refV2_2024 = $this->generateWithYear($entityCode, $invoiceId, $month, 2024, $amount);
        $refV2_2025 = $this->generateWithYear($entityCode, $invoiceId, $month, 2025, $amount);
        $refV2_2026 = $this->generateWithYear($entityCode, $invoiceId, $month, 2026, $amount);

        return [
            'entity_code' => $entityCode,
            'invoice_id' => $invoiceId,
            'month' => $month,
            'amount' => $amount,

            'v1_results' => [
                'format' => '11 digits (no year)',
                'method' => 'generate()',
                'references' => [
                    '2024' => $refV1_2024,
                    '2025' => $refV1_2025,
                    '2026' => $refV1_2026,
                ],
                'formatted' => [
                    '2024' => $this->format($refV1_2024),
                    '2025' => $this->format($refV1_2025),
                    '2026' => $this->format($refV1_2026),
                ],
                'all_same' => ($refV1_2024 === $refV1_2025 && $refV1_2025 === $refV1_2026),
                'issue' => '⚠️ PROBLEMA: Mesma referência para anos diferentes!'
            ],

            'v2_results' => [
                'format' => '13 digits (with year)',
                'method' => 'generateWithYear()',
                'references' => [
                    '2024' => $refV2_2024,
                    '2025' => $refV2_2025,
                    '2026' => $refV2_2026,
                ],
                'formatted' => [
                    '2024' => $this->format($refV2_2024),
                    '2025' => $this->format($refV2_2025),
                    '2026' => $this->format($refV2_2026),
                ],
                'all_unique' => (
                    $refV2_2024 !== $refV2_2025 &&
                    $refV2_2025 !== $refV2_2026 &&
                    $refV2_2024 !== $refV2_2026
                ),
                'solution' => '✅ SOLUÇÃO: Ano incluído garante unicidade!'
            ],

            'recommendation' => 'Use generateWithYear() para novas referências. generate() mantido apenas para compatibilidade.'
        ];
    }
}

