<?php
namespace App\Services;

use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use App\Helpers\FirebaseHelper;
use Kreait\Firebase\Exception\Messaging\NotFound;

class PushNotificationService
{
    protected $messaging;

    public function __construct()
    {
        // Get Firebase credentials from environment variables
        $firebaseCredentials = FirebaseHelper::getCredentials();

        $this->messaging = (new Factory)
            ->withServiceAccount($firebaseCredentials)
            ->createMessaging();
    }

    public function sendToUsers($users, $title, $body)
    {
        $success = 0;
        $fail = 0;

        foreach ($users as $user) {
            $deviceTokens = $user->deviceTokens;
            $userSuccess = false;

            foreach ($deviceTokens as $deviceToken) {
                $token = $deviceToken->token;

                try {
                    // FCM tokens are typically 140-163 characters long
                    if (strlen($token) >= 140 && strlen($token) <= 200) {
                        $notification = Notification::create($title, $body);
                        $message = CloudMessage::withTarget('token', $token)
                            ->withNotification($notification)
                            ->withHighestPossiblePriority()
                            ->withDefaultSounds();

                        $this->messaging->send($message);
                        Log::channel('log_notifications')->info("Notification sent successfully to {$user->name} via token: " . substr($token, 0, 20) . "...");
                        $success++;
                        $userSuccess = true;
                    } else {
                        // Token length invalid - delete it
                        Log::channel('log_notifications')->warning("Deleting invalid token for user: {$user->name} - Token invalid length (length: " . strlen($token) . ")");
                        $deviceToken->delete();
                        $fail++;
                    }
                } catch (NotFound $e) {
                    // Token not found in Firebase - it's invalid/expired, delete it
                    Log::channel('log_notifications')->warning("Deleting invalid token for user: {$user->name} - Token not found in Firebase");
                    $deviceToken->delete();
                    $fail++;
                } catch (\Kreait\Firebase\Exception\MessagingException $e) {
                    // Firebase messaging error - likely invalid token, delete it
                    Log::channel('log_notifications')->warning("Deleting invalid token for user: {$user->name} - Firebase error: {$e->getMessage()}");
                    $deviceToken->delete();
                    $fail++;
                } catch (\Exception $e) {
                    // Other errors - log but don't delete token
                    Log::channel('log_notifications')->error("Error sending notification to {$user->name}: {$e->getMessage()}");
                    $fail++;
                }
            }

            if (!$userSuccess && $deviceTokens->count() > 0) {
                Log::channel('log_notifications')->error("Failed to send notification to {$user->name} - All tokens were invalid or failed");
            }
        }

        return ['success' => $success, 'fail' => $fail];
    }
}
