<?php

namespace App\Services;

use App\Models\User;
use App\Models\Classroom;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use ZipArchive;

class StudentCardService
{
    /**
     * Remove background using rembg API
     * Falls back to original image if API fails
     */
    private function removeBackground(string $imagePath): ?string
    {
        $rembgApiKey = config('services.rembg.api_key') ?? env('REMBG_API_KEY');
        $rembgApiUrl = config('services.rembg.base_url') ?? env('REMBG_API_URL', 'https://api.rembg.com');

        if ($rembgApiKey && $rembgApiUrl) {
            $result = $this->removeBackgroundWithRembgApi($imagePath, $rembgApiKey, $rembgApiUrl);
            if ($result) {
                Log::info('rembg: Fundo removido com sucesso');
                return $result;
            }
            Log::warning('rembg: Falha ao remover fundo, usando imagem original');
        } else {
            Log::warning('rembg: API não configurada, usando imagem original');
        }

        // Fallback to original image (no background removal)
        return $this->imageToBase64($imagePath);
    }

    /**
     * Redimensionar imagem para economizar memória
     */
    private function resizeImage(string $imagePath, int $maxWidth = 800): ?string
    {
        try {
            $imageInfo = getimagesize($imagePath);
            if (!$imageInfo) {
                return null;
            }

            list($width, $height) = $imageInfo;

            // Se já é pequena, retornar original
            if ($width <= $maxWidth) {
                return $imagePath;
            }

            // Calcular nova altura mantendo proporção
            $newWidth = $maxWidth;
            $newHeight = (int) ($height * ($maxWidth / $width));

            // Criar imagem baseado no tipo
            $mime = $imageInfo['mime'];
            switch ($mime) {
                case 'image/jpeg':
                    $source = imagecreatefromjpeg($imagePath);
                    break;
                case 'image/png':
                    $source = imagecreatefrompng($imagePath);
                    break;
                case 'image/gif':
                    $source = imagecreatefromgif($imagePath);
                    break;
                default:
                    return $imagePath;
            }

            if (!$source) {
                return $imagePath;
            }

            // Criar nova imagem redimensionada
            $resized = imagecreatetruecolor($newWidth, $newHeight);

            // Preservar transparência para PNG
            if ($mime === 'image/png') {
                imagealphablending($resized, false);
                imagesavealpha($resized, true);
            }

            imagecopyresampled($resized, $source, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);

            // Salvar temporariamente
            $tempPath = sys_get_temp_dir() . '/' . uniqid('resized_') . '.jpg';
            imagejpeg($resized, $tempPath, 85);

            imagedestroy($source);
            imagedestroy($resized);

            return $tempPath;

        } catch (\Exception $e) {
            Log::warning('Erro ao redimensionar imagem', [
                'error' => $e->getMessage(),
            ]);
            return $imagePath;
        }
    }

    /**
     * Remove background using rembg API
     */
    private function removeBackgroundWithRembgApi(string $imagePath, string $apiKey, string $apiUrl): ?string
    {
        try {
            // Aumentar limite de memória temporariamente
            $oldLimit = ini_get('memory_limit');
            ini_set('memory_limit', '256M');

            // Redimensionar imagem para economizar memória
            $processedPath = $this->resizeImage($imagePath, 800);
            if (!$processedPath) {
                $processedPath = $imagePath;
            }

            // Use rembg API endpoint as per documentation
            $response = Http::timeout(30)
                ->withHeaders([
                    'x-api-key' => $apiKey,
                ])
                ->attach('image', file_get_contents($processedPath), basename($processedPath))
                ->post($apiUrl . '/rmbg');

            // Limpar arquivo temporário se foi criado
            if ($processedPath !== $imagePath && file_exists($processedPath)) {
                @unlink($processedPath);
            }

            // Restaurar limite de memória
            ini_set('memory_limit', $oldLimit);

            if ($response->successful()) {
                $imageData = $response->body();
                return 'data:image/png;base64,' . base64_encode($imageData);
            }

            Log::warning('rembg API error', [
                'status' => $response->status(),
                'error' => $response->body(),
            ]);

            return null;

        } catch (\Exception $e) {
            Log::warning('rembg API exception', [
                'error' => $e->getMessage(),
            ]);

            // Restaurar limite de memória em caso de erro
            if (isset($oldLimit)) {
                ini_set('memory_limit', $oldLimit);
            }

            return null;
        }
    }



    /**
     * Convert image to base64 for DomPDF compatibility
     */
    private function imageToBase64(string $imagePath): ?string
    {
        try {
            if (!file_exists($imagePath)) {
                return null;
            }

            $imageInfo = getimagesize($imagePath);
            if (!$imageInfo) {
                return null;
            }

            $mime = $imageInfo['mime'];
            $imageData = file_get_contents($imagePath);

            if ($imageData === false) {
                return null;
            }

            return 'data:' . $mime . ';base64,' . base64_encode($imageData);

        } catch (\Exception $e) {
            Log::warning('Erro ao converter imagem para base64', [
                'path' => $imagePath,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Generate a student card PDF for a single student
     */
    public function generateCard(User $student): string
    {
        set_time_limit(60);

        // Ensure classroom relationship is loaded
        if (!$student->relationLoaded('classroom')) {
            $student->load('classroom');
        }

        $classroom = $student->classroom;
        $classLevel = $classroom ? $classroom->class : '';

        // Determine education level text
        $educationLevel = 'ENSINO GERAL';
        if ($classLevel) {
            $level = (int) $classLevel;
            if ($level >= 1 && $level <= 6) {
                $educationLevel = 'ENSINO PRIMÁRIO';
            } elseif ($level >= 7 && $level <= 9) {
                $educationLevel = 'ENSINO SECUNDÁRIO - 1º CICLO';
            } elseif ($level >= 10 && $level <= 12) {
                $educationLevel = 'ENSINO SECUNDÁRIO - 2º CICLO';
            }
        }

        // Get student photo path and process background removal
        $photoPath = null;
        $photoData = null;
        if ($student->avatar) {
            $avatarPath = Storage::disk('public')->path($student->avatar);
            if (file_exists($avatarPath)) {
                $photoPath = $avatarPath;
                // Try to remove background, fallback to original if fails
                $photoData = $this->removeBackground($avatarPath);
                if (!$photoData) {
                    $photoData = $this->imageToBase64($avatarPath);
                }
            }
        }

        // Default placeholder if no photo
        if (!$photoPath) {
            $photoPath = public_path('images/default-avatar.png');
            if (!file_exists($photoPath)) {
                $photoPath = null;
            } else {
                $photoData = $this->removeBackground($photoPath);
                if (!$photoData) {
                    $photoData = $this->imageToBase64($photoPath);
                }
            }
        }

        $data = [
            'student' => $student,
            'studentName' => strtoupper($student->name),
            'studentId' => $student->student_id,
            'classroom' => $classroom,
            'classLevel' => $classLevel,
            'educationLevel' => $educationLevel,
            'academicYear' => date('Y'),
            'photoPath' => $photoPath,
            'photoData' => $photoData, // Base64 image with background removed
            'schoolName' => 'COLÉGIO POLITÉCNICO DE MOÇAMBIQUE',
            'schoolLogo' => public_path('assets/images/logo.png'),
        ];

        // Generate PDF using card template
        $pdf = Pdf::loadView('pdf.student-card', $data)
            ->setOptions([
                'isRemoteEnabled' => true,
                'isHtml5ParserEnabled' => true,
            ])
            ->setPaper([0, 0, 243, 153], 'landscape'); // ID card size: 85.6mm x 54mm at 72dpi

        // Save card
        $year = date('Y');
        $fileName = Str::slug($student->name) . '_cartao_' . $student->student_id . '.pdf';
        $filePath = "student_cards/{$year}/{$fileName}";

        // Ensure directory exists
        $dir = dirname(Storage::disk('public')->path($filePath));
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }

        Storage::disk('public')->put($filePath, $pdf->output());

        Log::info('Cartão de estudante gerado', [
            'student_id' => $student->id,
            'file_path' => $filePath,
        ]);

        return $filePath;
    }

    /**
     * Generate cards for multiple students in a classroom
     */
    public function generateCardsForClassroom(string $classroomId): array
    {
        $students = User::where('classroom_id', $classroomId)
            ->where('is_active', 1)
            ->orderBy('name')
            ->get();

        $results = [
            'success' => [],
            'failed' => [],
        ];

        foreach ($students as $student) {
            try {
                $path = $this->generateCard($student);
                $results['success'][] = [
                    'student_id' => $student->id,
                    'name' => $student->name,
                    'path' => $path,
                ];
            } catch (\Exception $e) {
                $results['failed'][] = [
                    'student_id' => $student->id,
                    'name' => $student->name,
                    'error' => $e->getMessage(),
                ];
                Log::error('Erro ao gerar cartão de estudante', [
                    'student_id' => $student->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return $results;
    }

    /**
     * Generate ZIP file with all cards for a classroom
     */
    public function generateClassroomZip(string $classroomId): string
    {
        $classroom = Classroom::find($classroomId);
        if (!$classroom) {
            throw new \Exception('Turma não encontrada');
        }

        // First generate all cards
        $results = $this->generateCardsForClassroom($classroomId);

        if (empty($results['success'])) {
            throw new \Exception('Nenhum cartão foi gerado');
        }

        // Create ZIP
        $year = date('Y');
        $zipFileName = 'cartoes_' . Str::slug($classroom->name) . '_' . $year . '.zip';
        $zipPath = "student_cards/downloads/{$zipFileName}";
        $fullZipPath = Storage::disk('public')->path($zipPath);

        // Ensure directory exists
        $dir = dirname($fullZipPath);
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }

        $zip = new ZipArchive();
        if ($zip->open($fullZipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new \Exception('Não foi possível criar arquivo ZIP');
        }

        foreach ($results['success'] as $item) {
            if (Storage::disk('public')->exists($item['path'])) {
                $content = Storage::disk('public')->get($item['path']);
                $zip->addFromString(basename($item['path']), $content);
            }
        }

        $zip->close();

        Log::info('ZIP de cartões gerado', [
            'classroom_id' => $classroomId,
            'zip_path' => $zipPath,
            'cards_count' => count($results['success']),
        ]);

        return $zipPath;
    }

    /**
     * Get students without cards for a classroom
     * Returns arrays instead of Eloquent models for Livewire compatibility
     */
    public function getStudentsWithoutCards(string $classroomId): array
    {
        $students = User::where('classroom_id', $classroomId)
            ->where('is_active', 1)
            ->orderBy('name')
            ->get();

        $year = date('Y');
        $withCards = [];
        $withoutCards = [];

        foreach ($students as $student) {
            $fileName = Str::slug($student->name) . '_cartao_' . $student->student_id . '.pdf';
            $filePath = "student_cards/{$year}/{$fileName}";

            $studentData = [
                'id' => $student->id,
                'name' => $student->name,
                'student_id' => $student->student_id,
            ];

            if (Storage::disk('public')->exists($filePath)) {
                $withCards[] = [
                    'student' => $studentData,
                    'card_path' => $filePath,
                ];
            } else {
                $withoutCards[] = $studentData;
            }
        }

        return [
            'with_cards' => $withCards,
            'without_cards' => $withoutCards,
        ];
    }

    /**
     * Get statistics for student cards
     */
    public function getStatistics(?int $academicYear = null): array
    {
        $year = $academicYear ?? (int) date('Y');
        $basePath = "student_cards/{$year}";

        $totalCards = 0;
        $totalStudents = User::where('is_active', 1)->count();

        // Count existing cards
        if (Storage::disk('public')->exists($basePath)) {
            $files = Storage::disk('public')->files($basePath);
            $totalCards = count(array_filter($files, fn($f) => Str::endsWith($f, '.pdf')));
        }

        return [
            'total_students' => $totalStudents,
            'total_cards' => $totalCards,
            'cards_pending' => $totalStudents - $totalCards,
        ];
    }
}
