<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

return new class extends Migration
{
    public function up()
    {
        // Tabela para registar todas as transações com IVA
        Schema::create('iva_transactions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->date('transaction_date');
            $table->enum('transaction_type', ['sale', 'purchase'])->default('sale');
            $table->string('category'); // 'certificado', 'declaracao', 'propina', 'uniforme', etc
            $table->string('description');

            // Dados do cliente/estudante
            $table->uuid('student_id')->nullable();
            $table->string('student_name')->nullable();
            $table->string('nuit_cliente')->nullable();

            // Valores financeiros
            $table->decimal('price_shown', 10, 2); // Preço mostrado ao cliente
            $table->boolean('price_includes_iva'); // Se o preço inclui IVA ou não
            $table->decimal('iva_rate', 5, 2); // Taxa aplicada (vai buscar de pos_settings)

            // Cálculos automáticos
            $table->decimal('base_amount', 10, 2); // Base tributável (sem IVA)
            $table->decimal('iva_amount', 10, 2); // Montante de IVA
            $table->decimal('total_amount', 10, 2); // Total (com IVA)

            // Referências
            $table->string('reference_number')->nullable(); // Nº do certificado, recibo, etc
            $table->string('invoice_number')->nullable(); // Nº da factura (se aplicável)
            $table->uuid('related_id')->nullable(); // ID do certificado/declaração
            $table->string('related_type')->nullable(); // Classe do modelo relacionado

            // Metadados
            $table->text('metadata')->nullable(); // JSON com info adicional
            $table->uuid('created_by')->nullable();
            $table->timestamps();
            $table->softDeletes();

            // Índices
            $table->index('transaction_date');
            $table->index('category');
            $table->index(['student_id', 'transaction_date']);
            $table->index('reference_number');
        });

        // Adicionar configuração global de IVA ao pos_settings
        DB::table('pos_settings')->insert([
            [
                'id' => (string) Str::uuid(),
                'key' => 'iva_enabled',
                'value' => '1',
                'type' => 'boolean',
                'description' => 'Activar sistema de IVA',
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ]);

        // Adicionar campos IVA à tabela fee_type (configuração dinâmica por tipo)
        Schema::table('fee_type', function (Blueprint $table) {
            $table->boolean('iva_applicable')->default(false)->after('description');
            $table->boolean('price_includes_iva')->default(false)->after('iva_applicable');
            $table->decimal('default_price', 10, 2)->nullable()->after('price_includes_iva');
            $table->text('iva_notes')->nullable()->after('default_price');
        });

    }

    public function down()
    {
        Schema::dropIfExists('iva_transactions');

        // Remover configurações adicionadas
        DB::table('pos_settings')->where('key', 'iva_enabled')->delete();

        // Remover campos IVA da tabela fee_type
        Schema::table('fee_type', function (Blueprint $table) {
            $table->dropColumn(['iva_applicable', 'price_includes_iva', 'default_price', 'iva_notes']);
        });
    }
};
