<?php

namespace App\Http\Livewire\Admin\Students;

use App\Jobs\MailJob;
use App\Models\User;
use App\Models\Admin;
use App\Models\Classroom;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;
use Exception;
use Twilio\Rest\Client;

class AddStudent extends Component
{
    use FindGuard, LivewireAlert;

    public $name, $classroom_id, $sex, $phone_number, $student_id, $class, $email, $dob;
    public $validatedData;
    public $updateMode = false;

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    protected array $rules = [
        'name' => ['required', 'string', 'max:255'],
        'sex' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'max:255', 'unique:users'],
        'student_id' => ['required', 'string', 'max:255', 'unique:users'],
        'classroom_id' => ['required', 'string', 'max:255'],
        'class' => ['required', 'string', 'max:255'],
        'dob' => ['required', 'string', 'max:255'],
        'phone_number' => ['required', 'numeric'],
    ];

    /**
     * D0 real time validations
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function submit()
    {
        $this->validatedData = $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }

    public function confirmed()
    {
        // generate password
        $password = rand(100000, 500000);
        $student = User::query()->create([
            'name' => $this->name,
            'sex' => $this->sex,
            'email' => $this->email,
            'student_id' => $this->student_id,
            'classroom_id' => $this->classroom_id,
            'dob' => $this->dob,
            'password' => bcrypt($password),
            'is_paid' => true,
            'phone_number' => $this->phone_number,
        ]);
        // todo send sms
        try {
            $account_sid = getenv("TWILIO_SID");
            $auth_token = getenv("TWILIO_TOKEN");
            $twilio_number = getenv("TWILIO_FROM");

            $client = new Client($account_sid, $auth_token);

            // Generate the callback URL based on your route
            $callbackUrl = route('login');

            // Include the URL on a new line using \n
            $message = "Caro pai, o seu educando " . $this->name . ", foi registado no sistema de gestao escolar COPMOZ, o seu ID do Estudante: $this->student_id  e a sua senha:" . $password . "\n \nClique no link para aceder o sistema: https://siga.copmoz.ac.mz";

            $client->messages->create('+258' . $this->phone_number, [
                'from' => $twilio_number,
                'body' => $message,
            ]);
        } catch (Exception $e) {
            $this->alert('error', 'Erro ao enviar notificações por SMS');
        }
        Note::createSystemNotification(Admin::class, 'Novo estudante', 'Criado com sucesso ' . $this->name);
        $this->emit('noteAdded');
        $this->alert('success', 'Criado com sucesso ' . $this->name);
        $this->reset();
    }

    public function cancelled()
    {
        $this->alert('error', 'Voce cancelou.');
    }

    public function render()
    {
        return view('livewire.admin.students.add-student', [
            'classes' => Classroom::query()->where(function ($query) {
                $query->orWhere('class', $this->class);
            })->get(),
        ]);
    }
}
