<?php

namespace App\Http\Livewire\Admin\Teachers;

use App\Jobs\MailJob;
use App\Models\Admin;
use App\Traits\SysPermissions;
use Illuminate\Validation\ValidationException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use LaravelMultipleGuards\Traits\FindGuard;
use Livewire\Component;
use Note\Note;

class AddTeacher extends Component
{
    use FindGuard, SysPermissions, LivewireAlert;

    public $name;
    public $phone_number;
    public $email, $role;

    public $access = [];

    protected $listeners = [
        'confirmed',
        'cancelled'
    ];

    protected function rules(): array
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:admins'],
            'phone_number' => ['required', 'numeric', 'unique:admins'],
        ];
    }

    /**
     * @param $propertyName
     * @throws ValidationException
     */
    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    /**
     * update user password here
     */
    public function submit()
    {
        $this->validate();
        $this->confirm('Tem certeza de que deseja prosseguir?', [
            'toast' => false,
            'position' => 'center',
            'showConfirmButton' => true,
            'confirmButtonText' => 'Sim, eu tenho certeza!',
            'cancelButtonText' => 'Não, cancelar!',
            'onConfirmed' => 'confirmed',
            'onDismissed' => 'cancelled'
        ]);
    }


    public function linkOrRemovePrivilege(string $privilege)
    {
        // check if it exists in array
        if (in_array($privilege, $this->access)) {
            unset($this->access[array_search($privilege, $this->access)]);
        } else {
            $this->access[] = $privilege;
        }
    }

    public function confirmed()
    {
        $checkemail = Admin::where('email', $this->email)->get();
        if (count($checkemail) >= 1) {
            $this->alert('warning', 'Desculpe, este e-mail já foi enviado.');
            return back();
        }
        // generate password
        $password = rand(100000, 500000);
        // $password = rand(100000, 500000);
        // create officer here
        $hod = Admin::query()->create([
            'name' => $this->name,
            'email' => $this->email,
            'role' => 2,
            'phone_number' => $this->phone_number,
            'password' => bcrypt('password'),
        ]);
        // send user account credentials
        dispatch(new MailJob(
            $this->name,
            $this->email,
            config('app.name') . ' Conta de administrador',
            'Sua conta foi ativada, por favor use ' . $password . ' como sua senha para fazer login.',
            true,
            route('admin.login'),
            'CHECK ACCOUNT'
        ))->onQueue('emails')->delay(1);

        Note::createSystemNotification(Admin::class, 'Nova conta do Admin', 'Você criou uma conta com sucesso ' . $this->name);
        $this->emit('noteAdded');
        $this->alert('success', 'Você criou o administrador com sucesso ' . $this->name);
        $this->reset();
    }

    public function cancelled()
    {
        $this->alert('error', 'Você cancelou.');
    }

    public function render()
    {
        return view('livewire.admin.teachers.add-teacher');
    }
}
